
package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.command.annotations.Command;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.Thermometer;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;

/**
 * This is a driver for a thermometer we have in the APC test bench.
 *
 * @author virieux
 */
public class CanOpenThermometer extends CanOpenSensor14bits implements Thermometer {

    private double temperature;

    public CanOpenThermometer(CanCBXAI814 adc, int numOfAnalogInput) {
        super(adc, numOfAnalogInput);
    }

    /**
     * This methods translate the hexa value return by a sensor to the double
     * value.
     *
     * @param value
     * @return the corresponding double value
     */
    public static double getDoubleValue(int value) {
        double temp = 0;
        FCSLOG.debug("Value in HEXA=" + value);

        double voltage = value * CanCBXAI814.adcVoltsPerBit;
        FCSLOG.debug("Input Value read on the ADC:" + value);
        FCSLOG.debug("Voltage = " + voltage + " volts");
        temp = (voltage / 10) - CanCBXAI814.waterFreezeTemp;
        return temp;
    }

    /**
     * Read temperature on the termometer and returns the new value read.
     * @return
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
        description = "Read temperature on the termometer and returns the new value read.")
    @Override
    public double readTemperature()  {
         
        double voltage;
        try {
            voltage = adc.readVoltage(numOfAnalogInput);
            temperature = (voltage / 10) - CanCBXAI814.waterFreezeTemp;
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(name + ": could'nt read temperature:" + ex);
        }
        return temperature;
    }

    
    /**
     * 
     * @return 
     */
    public double readTemperatureFromPDO() {
        double voltage;
        adc.checkInitialized();
        if (adc.transmissionType != 1) {
            throw new RejectedCommandException("Transmission type for Can Open device must be 1");
        }
        try {
            adc.tcpProxy.updatePDOData();
            voltage = adc.getInputs()[numOfAnalogInput] * CanCBXAI814.adcVoltsPerBit;
            temperature = (voltage / 10) - CanCBXAI814.waterFreezeTemp;
        } catch(Exception ex) {
            FCSLOG.error("cannot read temperature via pdos", ex);
        }
        return temperature;
    }
    
    @Override
    public void postStart() {
        if (adc.isBooted()) {
            adc.initializeAndCheckHardware();
        }
    }
}
