
package org.lsst.ccs.subsystems.fcs.drivers;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.drivers.canopenjni.CanOpenInterface;
import org.lsst.ccs.drivers.canopenjni.PDOData;
import org.lsst.ccs.drivers.commons.DriverException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations;
import org.lsst.ccs.subsystems.fcs.common.SensorPluggedOnTTC580;
import org.lsst.ccs.subsystems.fcs.common.TTC580Interface;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;

/**
 * A model for the CANopen device to read carousel sensors.
 * This sensors are read by PDO. (in response to a command "sync" sent to the CANbus)
 * 
 * @author virieux
 */
public class CanOpenTTC580  extends CanOpenDevice implements TTC580Interface {
    
    public static final String LOCKSENSOR = "lockSensor";
    public static final String FILTERPRESENCE = "filterPresence";
    public static final String XMINUS = "Xminus";
    public static final String XPLUS = "Xplus";
    
    @LookupField(strategy=Strategy.SIBLINGS)
    private CanOpenInterface canInterface;
    
    
    protected int cobid1 = this.nodeID + 0x180;
    protected int cobid2 = this.nodeID + 0x280;
    
    /**
     * map of sensors attached to this device.
     */
    @LookupField(strategy = LookupField.Strategy.TREE)
    protected Map<String, SensorPluggedOnTTC580> sensorsMap = new HashMap<>();    
    
    /**
     * PDO1 represents sensors value for socket at STANDBY.
     * pdo1 is coded on 64bits and represents from right to left :
     * - 3 bits for socket identification
     * - 3 bits unused
     * - 2 bits unused
     * - 4 bits for State of filter at standby for clampXminus
     * - 4 bits for State of filter at standby for clampXplus
     * - 12 bits for ClampXminus lockSensor value
     * - 12 bits for ClampXminus filterPresenceSensor value
     * - 12 bits for ClampXplus lockSensor value
     * - 12 bits for ClampXplus filterPresenceSensor value
     * 
     */
    protected long pdo1 = 0L;
    
    /**
     * PDO2 represents the same data than PDO1 but for another socket, not at standby. 
     */
    protected long pdo2 = 0L;

    public int getCobid1() {
        return cobid1;
    }

    public int getCobid2() {
        return cobid2;
    }
    
    

    public CanOpenTTC580(int nodeID, String serialNB) {
        super(nodeID, serialNB);
    }
    
    @Override
    public void init() {
        FCSLOG.info(name + " =====> initialization of my sensors map");
        StringBuilder sb = new StringBuilder();
        for (Iterator<SensorPluggedOnTTC580> it = sensorsMap.values().iterator(); it.hasNext();) {
            SensorPluggedOnTTC580 sensor = it.next();
            sb.append(sensor.getName());
            sb.append("====>");
            if (sensor.getDeviceName().equals(name)) {
                sb.append(" IS MY SENSOR ");
                sb.append(sensor.toString());
                sb.append("#\n");
            } else {
                it.remove();
            }
        }
        FCSLOG.info(sb.toString());
        FCSLOG.info(name + " =====> MY SENSORS \n" + sensorsMap.toString());
    }    


    /**
     * 
     * @return 
     */
    @Command(type=Command.CommandType.QUERY, description="return PDO1 value")
    @Override
    public long getPdo1() {
        return pdo1;
    }

    /**
     * 
     * @param pdoVal 
     */
    public void setPdo1(long pdoVal) {
        this.pdo1 = pdoVal;
    }

    @Command(type=Command.CommandType.QUERY, description="return PDO2 value")
    @Override
    public long getPdo2() {
        return pdo2;
    }

    public void setPdo2(long pdo2) {
        this.pdo2 = pdo2;
    }
    
    
    
    @Override
    public void doInitializePDOs() throws DriverException {
        canInterface.addReceivedPDO(cobid1);
        canInterface.addReceivedPDO(cobid2);
        initialized = true;
    }
    
    /**
     * juste for tests and debug
     * @throws DriverException 
     */
    @Command(type=Command.CommandType.QUERY, level= Command.ENGINEERING3, 
            description="For DEBUG purpose. Send a sync command to CANbus, "
                    + "process response and update pdo1, pdo2 and sensors values")    
    @Override
    public void updateSensorsFromPDO() throws DriverException {
        updateFromPDO(tcpProxy.sync());
    }
    
    /**
     * process PDOData to retrieve data from this device.
     * @param pdo
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    @Override
    public void updateFromPDO(PDOData pdo) throws DriverException {
        FCSLOG.info(name + " updatingFromPDO = " + pdo);
        
        if (pdo.getPDOs().containsKey(cobid1)) {
            pdo1 = (long) pdo.getPDOs().get(cobid1);
        }
        if (pdo.getPDOs().containsKey(cobid2)) {
            pdo2 = (long) pdo.getPDOs().get(cobid2);
        }
        if (pdo2 == 0) {
            throw new FcsHardwareException(name + " pdo2 can't be 0 " + this.toString());
        }
        /* update from PDO1 */
        short socketID1 = getSocketId(pdo1);
        FCSLOG.info(name + " updatingFromPDO1 = " + pdo1 + " binaire:" + Long.toBinaryString(pdo1));
        if (socketID1 == 0) {
            FCSLOG.info(name + " no socket in STANDBY position");
        } else if (socketID1 == 7) {
            this.raiseAlarm(FcsEnumerations.FcsAlert.CA_SENSOR_ERROR, " error reading socketID for pdo1");
        } else if (socketID1 >= 1 && socketID1 <= 5) {
            updateSocketSensors(socketID1, pdo1);
        }
        
        /* update from PDO2 */
        short socketID2 = getSocketId(pdo2);
        FCSLOG.info(name + " updatingFromPDO2 = " + pdo2 + " binaire:" + Long.toBinaryString(pdo2));
        if (socketID2 == 0) {
            this.raiseAlarm(FcsEnumerations.FcsAlert.CA_SENSOR_ERROR, " socketID for pdo2 can't be 0");
        } else if (socketID2 == 7) {
            this.raiseAlarm(FcsEnumerations.FcsAlert.CA_SENSOR_ERROR, " error reading socketID for pdo2");
        } else if (socketID2 >= 1 && socketID2 <= 5) {
            updateSocketSensors(socketID2, pdo2);
        }
    }
    
    
    
    
    /**
     * update
     * @param socketID 
     * @param pdoValue 
     */
    protected void updateSocketSensors(short socketID, long pdoValue) {
        //TODO a refactoring could be done to not use sensors name
        FCSLOG.info(name + " updating socket sensors values for socketID=" + socketID);
        /*Xminus lock sensor update*/
        sensorsMap.get(LOCKSENSOR + XMINUS + socketID).updateValue(getLockXm(pdoValue));
        /*Xplus lock sensor update*/
        sensorsMap.get(LOCKSENSOR + XPLUS + socketID).updateValue(getLockXp(pdoValue));  
        /*Xminus filter presence sensor update*/
        sensorsMap.get(FILTERPRESENCE + XMINUS + socketID).updateValue(getFilterPresenceXm(pdoValue));        
        /*Xplus filter presence sensor update*/
        sensorsMap.get(FILTERPRESENCE + XPLUS + socketID).updateValue(getFilterPresenceXp(pdoValue)); 
    }    
    
    /**
     * return socketId from PDO value
     * socket identification are the first 3 bits (reading from left to right).
     * @param pdo
     * @return 
     */
    @Command(type=Command.CommandType.QUERY, description="return socketId from PDO value")
    @Override
    public short getSocketId(long pdo) {
        /* 61 = 64 - nbBits (nbBits =3)*/
        return (short) ((pdo >>> 61));
    }
    
    /**
     * return lock sensor for clampXminus
     * @param pdo
     * @return 
     */
    @Command(type=Command.CommandType.QUERY, description="return lock sensor for clampXminus")
    public int getLockXm(long pdo) {
        return (int) (pdo >>> 36) & 0xFFF;
    }
    
    /**
     * return filter presence sensor for clampXminus
     * @param pdo
     * @return 
     */
    @Command(type=Command.CommandType.QUERY, description="return filter presence sensor for clampXminus")
    public int getFilterPresenceXm(long pdo) {
        return (int) (pdo >>> 24) & 0xFFF;
    }
    
    /**
     * return lock sensor for clampXplus
     * @param pdo
     * @return 
     */
    @Command(type=Command.CommandType.QUERY, description="return lock sensor for clampXplus")
    public int getLockXp(long pdo) {
        return (int) (pdo >>> 12) & 0xFFF;
    }    
    
    /**
     * return filter presence sensor value for clampXplus.
     * this value is coded on the last 12 bits (reading from left to right).
     * @param pdo
     * @return 
     */
    @Command(type=Command.CommandType.QUERY, description="return filter presence sensor for clampXplus")
    public int getFilterPresenceXp(long pdo) {
        return (int) pdo & 0xFFF;
    }
    
    @Command(type=Command.CommandType.QUERY)
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(super.toString());
        sb.append("\n/pdo1=");
        sb.append(pdo1);
        sb.append(" in hexa:");
        sb.append(Long.toHexString(pdo1));
        sb.append(" in binaire:");
        sb.append(Long.toBinaryString(pdo1));
        sb.append("\n/pdo2=");
        sb.append(pdo2); 
        sb.append(" in hexa:");
        sb.append(Long.toHexString(pdo2));
        sb.append(" in binaire:");
        sb.append(Long.toBinaryString(pdo2));
        return sb.toString();
    }

}
