package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;

/**
 * This class represents the strain gauge Seneca ZC-SG used in the autochanger
 * for ONLINE clamps. This force sensor is used to know if a ONLINE clamp is
 * locked or not.
 *
 * @author virieux
 */
public class CanOpenStrainGauge extends CanOpenDevice {

    /**
     * The sample number it is the number of sample that enters into the
     * measure. Higher values implies lower response speed but more stability.
     */
    final short sampleNR = 10;

    /**
     * The station can be configured in two modes : Mode=1: a known weight must
     * be used to calibrate the system on site. Mode=0: no need to use a known
     * weight to calibrate the system, the station will use the factory
     * calibration values.
     */
    final short mode = 0;

    /**
     * cell mV/V sensibility 0=±1 mV/V 1=±2 mV/V 2=±4 mV/V 3=±8 mV/V 4=±16 mV/V
     * 5=±32 mV/V 6=±64 mV/V 7=from object 0x2108 sub1
     */
    final short cellSensibility = 0;

    /**
     * Defines the operation that will cause the switch to ON or OFF for the
     * digital output. 0 : The output is normally opened 1 : The output is
     * normally closed
     */
    final short digital_out_logic = 0;

    /**
     * Defines the operation that will cause the switch to ON or OFF for the
     * digital output. 0 : The gross weight exceeds the full scale 1 : The
     * weight is stable and the net weight exceeds the threshold set 2 : The
     * weight is stable
     */
    final short digital_out_mode = 0;

    /**
     * The station can be configured with a digital input or a digital output: -
     * if IN or OUT selection=1: digital output enable/digital input disabled -
     * if IN or OUT selection=0: digital input enable/digital output disabled
     */
    final short digital_in_or_selection = 0;

    /**
     * Sets the sense ratio for the strain gauge used in [mV/V] (floating point
     * 32 bit format).
     */
    final short senseRatio = 10;

    /**
     * If mode 1 is selected (object 0x2107) sets the full scale of the strain
     * gauge in technical units of weight (kg, pounds,etc…) (Floating point 32
     * bit format).
     */
    final short cellFullScale = 0;

    /**
     * If mode 1 is selected (object 0x2107) sets the value of the weight used
     * for the calibration in technical units (kg, pounds, etc) (Floating point
     * 32 bit format).
     */
    final short knownWeightValue = 0;

    /**
     * If the net weight exceeds the threshold value set and the weight is
     * stable, the digital output (if subindex digital out mode=1) is closed or
     * opened (depending subindex digital output logic) (floating point 32 bit
     * format).
     */
    final short thresholdValue = 0;

    /**
     * Weight variation in technical units accepted for the condition of “stable
     * weight” (floating point 32 bit format)
     */
    final short deltaWeight = 0;

    /**
     * Time in units of 100 ms used with delta weight to establish whether or
     * not the weight is stable [s/10].
     */
    final short deltaTime = 0;

    /**
     * The ADC speed and the frequency rejection can be customized by the table:
     * The ADC speed and the frequency rejection can be customized by the table:
     * Value Sampling frequency [Hz] // 50 Hz rejection // 60 Hz rejection 27
     * 151.71 NO NO 55 74.46 NO NO 82 49.95 YES YES 109 37.59 NO YES 155 50.57
     * NO NO 183 24.82 YES NO 210 16.65 YES YES 237 12.53 NO YES
     */
    final short adcSpeed = 0;

    /**
     * The hysteresis can be used to stabilize the input value. If the
     * hysteresis is activated the resolution is limited to ±30000 points. If
     * hysteresis is disabled, the resolution available is the full 24 bit ADC.
     * 0x00=hysteresis disabled 0x80=hysteresis enabled
     */
    final short hysteresis = 0x00;

    /**
     * Integer net value (signed): integer approximation of the floating point
     * value.
     */
    short strain;

    /**
     * The ADC value scaled into 16 bit (unsigned) value. 
     * Where if ADC = 0x8000 means 0 mV on input. 
     * If ADC = 0xFFFF means max positive mV on input. 
     * If ADC = 0 means max negative mV on input.
     */
    int adcValue;

    /**
     * build a Seneca_ZC_SG
     *
     * @param nodeID
     * @param serialNB
     */
    public CanOpenStrainGauge(int nodeID, String serialNB) {
        super(nodeID, serialNB);
    }

    /**
     * configure register1 configuration of register1 is done with index 0x2107
     * The object 0x2107 is used to setup the measure and the digital
     * input/output. See Seneca ZC-SG documentation
     */
    public void configureRegister1() {
        final short size = 01;
        /*sampleNR*/
        this.writeSDO(0x2107, 01, size, sampleNR);
        /*mode*/
        this.writeSDO(0x2107, 02, size, mode);
        /*cellSensibility*/
        this.writeSDO(0x2107, 03, size, cellSensibility);
        /*digital_out_logic*/
        this.writeSDO(0x2107, 04, size, digital_out_logic);
        /*digital_out_mode*/
        this.writeSDO(0x2107, 05, size, digital_out_mode);
        /*digital_in_or_selection*/
        this.writeSDO(0x2107, 06, size, digital_in_or_selection);
    }

    /**
     * configure register2 configuration of register2 is done with index 0x2108
     * The object 0x2108 is used to setup the system measure. See Seneca ZC-SG
     * documentation
     */
    public void configureRegister2() {
        final short size = 01;
        /*senseRatio*/
        this.writeSDO(0x2108, 1, size, senseRatio);
        /*cellFullScale*/
        this.writeSDO(0x2108, 2, size, cellFullScale);
        /*knownWeightValue*/
        this.writeSDO(0x2108, 3, size, knownWeightValue);
        /*subindex 04 and 05 are not used*/
        /*thresholdValue*/
        this.writeSDO(0x2108, 6, size, thresholdValue);
        /*deltaWeight*/
        this.writeSDO(0x2108, 7, size, deltaWeight);
        /*deltaTime*/
        this.writeSDO(0x2108, 8, size, deltaTime);
        /*adcSpeed*/
        this.writeSDO(0x2108, 9, size, adcSpeed);
        /*hysteresis*/
        this.writeSDO(0x2108, 10, size, adcSpeed);
    }

    /**
     * read internal temperature in degrees Celsius.
     *
     * @return
     */
    @Command(type = CommandType.QUERY, description = "read internal temperature in degrees Celsius.")
    public double readTemperature() {
        /* value read is expressed in 0.1°C*/
        /* temperature is coded on 16 bits and is a signed value.*/
        return (short) this.tcpProxy.readSDO(nodeID,0x2030, 01) * 0.1;
    }

    /**
     * Read the strain value (or weight) : Integer net value (signed): integer
     * approximation of the floating point value.
     *
     * @return
     */
    @Command(type = CommandType.QUERY, description = "Read the strain value (or weight) : Integer net value (signed).")
    public short readStrain() {
        /* weight is coded on 16 bits and is a signed value.*/
        return (short) this.tcpProxy.readSDO(nodeID,0x6401, 01);
    }
    
    /**
     * Read the ADC value.
     * @return 
     */
    @Command(type = CommandType.QUERY, description = "Read the ADC value.")
    public int readAdcValue() {
        this.adcValue = (int) this.tcpProxy.readSDO(nodeID,0x6401, 02);
        return adcValue;
    }
    
    /**
     * read 32 bit float input value
     * @return 
     */
    @Command(type = CommandType.QUERY, description = "Read 32 bit float input value.")
    public int readStrain32bitFloat() {
        return (int) this.tcpProxy.readSDO(nodeID, 0x6403, 01);
    }

}
