package org.lsst.ccs.subsystems.fcs.drivers;

import java.util.ArrayList;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.AlertService;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.drivers.canopenjni.CanOpenInterface;
import org.lsst.ccs.drivers.canopenjni.PDOData;
import org.lsst.ccs.drivers.canopenjni.SDOException;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.HasLifecycle;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations;
import org.lsst.ccs.subsystems.fcs.common.EmergencyMessage;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.BAD_SERIAL_NB;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.EMCY;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.HARDWARE_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.HARDWARE_MISSING;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.SDO_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.UNSYNC_BOOT;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterReadinessState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterState;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;

/**
 * This Module starts a tcpip server, waits for the connection of a client whose
 * name is the value of the field clientName. This client is supposed to be a
 * CANopen C-wrapper. When the client starts it scans the CANopen network and
 * send to the server some information on the CANopen bootedNodes living on the
 * CAN bus. The main job of the class CanOpenProxy is to check that the CANopen
 * devices booted on the CAN bus are the CANopen devices which are listed in the
 * configuration system. 
 *
 * @author virieux
 */
public class CanOpenProxy implements HardwareController, ClearAlertHandler, BridgeToHardware, HasLifecycle, AlertRaiser, CanOpenEventListener {

    @LookupField(strategy = Strategy.TOP)
    private Subsystem subs;
    
    @LookupName
    protected String name;

    @LookupField(strategy=Strategy.CHILDREN)
    private CanOpenInterface canInterface;
    
    private int master = 1;
    
    @ConfigurationParameter(description="CANbus rate.")
    private String baud = "125k";

    @ConfigurationParameter(description="CANbus name. Can be 0 for changer or 1 for loader.")    
    private String busName = "0";
    
    @ConfigurationParameter(description="CANbus master nodeID.")    
    private int masterNodeID = 0x8;

    /**
     * A timeout for the booting process : during initialization, we wait for
     * the boot messages coming from CANbus field. When this amount of time is
     * over, we don't wait any more. This has to be tuned during the test bench.
     * UNIT = milliseconds
     *
     */
    @ConfigurationParameter(range = "1000..100000",
            description = "A timeout for the hardware booting process")
    private long hardwareBootTimeout = 1000;

    /**
     * This map of PieceOfHardware which key is CANopen masterNodeID of the hardware
 represents the list of pieces of hardware (CANopen devices) this
 CanOpenProxy manages.
     */
    protected final Map<Integer, PieceOfHardware> hardwareMapByNodeID = new HashMap<>();

    @LookupField(strategy = Strategy.CHILDREN)
    protected final Map<String, PieceOfHardware> hardwareMapByName = new HashMap<>();

    private boolean hardwareIDError;
    protected boolean hardwareBootProcessEnded = false;
    protected int bootedDeviceNB;

    
    private PDOData pdoData = new PDOData();

    private ScheduledFuture<?> checkDevicesHandle;
    private final ScheduledThreadPoolExecutor scheduler = new ScheduledThreadPoolExecutor(1);
    //Used because we have to wait until all the pieces of hardware are booted.
    
    private final Lock lock = new ReentrantLock();

    private final Condition bootingCompleted = lock.newCondition();

    /**
     * Creates a CanOpenProxy with a tcpip port number to start the tcp server
     * on, a client name, and a hardware booting process timeout.
     *
     * @param hardwareBootTimeout
     */
    public CanOpenProxy(long hardwareBootTimeout) {
        this.hardwareBootTimeout = hardwareBootTimeout;
    }
    
    
    
    @Override
    public String getName() {
        return name;
    }
    
    @Override
    public Subsystem getSubsystem() {
        return subs;
    }

    public CanOpenInterface getCanInterface() {
        return canInterface;
    }

    /**
     * for simulation and tests
     * @return 
     */
    @Command
    public PDOData getPdoData() {
        return pdoData;
    }
    
    
    

    /**
     * ***********************************************************************************************
     * ******************** END OF SETTERS AND GETTERS
     * **********************************************
     * ***********************************************************************************************
     * /**
     *
     * /**
     * This methods stops the CWrapper client, and the TCP server.
     */
    @Override
    public void disconnectHardware() {
        try {
        //TODO we have to test if all the hardware is stopped before doing stopServer
            //doesn't work with 5.0.3
            canInterface.stop();
            initialize();
        } catch (DriverException ex) {
            throw new RuntimeException(ex);
        }
    }

    @Override
    public void connectHardware() {
        try {
            //doesn't work with 5.0.3
            canInterface.start();
        } catch (DriverException ex) {
            throw new RuntimeException(ex);
        }
    }

    @Override
    public void init() {

        initialize();

        for (PieceOfHardware poh : hardwareMapByName.values()) {
            hardwareMapByNodeID.put(poh.getNodeID(), poh);
        }

        FCSLOG.info(name + ": init MODULE CanOpenProxy.");
        FCSLOG.info(name + ":NUMBER OF CAN OPEN DEVICES EXPECTED =" + hardwareMapByName.size());
        FCSLOG.info(this.toString());

        //we have to read the errors table for the maxon motor and can open devices.
        CanOpenErrorsTable.loadCanOpenErrorTables();
    }

    /**
     * This method initializes the fields of the tcpProxy. 
     * It's used when we start the Module and when we disconnect hardware.
     */
    private void initialize() {
        this.bootedDeviceNB = 0;
        hardwareBootProcessEnded = false;
        this.hardwareIDError = false;
        hardwareMapByName.values().stream().forEach((poh) -> {
            poh.setBooted(false);
        });
    }

    /**
     * Check if all pieces of hardware in the configuration are booted with the corect serial number.
     * Raise alerts if not.
     * Waits until all hardware is booted or timeout is reached. 
     * 
     */
    @Command(type=Command.CommandType.ACTION, description="Check if all pieces of hardware "
            + "are booted with the corect serial number.")
    @Override
    public void bootProcess() {
        FCSLOG.info(name + ": BEGIN OF BOOT PROCESS");
        initialize();
        
        subs.updateAgentState(FilterState.CAN_DEVICES_BOOTING,
                FilterReadinessState.NOT_READY);
        lock.lock();
        try {
            readDevicesInfo();
            waitForEndOfBooting();
            
        } catch (Exception ex) {
            this.raiseAlarm(HARDWARE_ERROR, " ERROR during booting process", ex);
            
        } finally {
            FCSLOG.debug(name + ": finally in bootProcess");
            hardwareBootProcessEnded = true;
            bootingCompleted.signalAll();
            lock.unlock();
        }
        if (!this.hardwareIDError) {
            FCSLOG.info(name + " ALL HARDWARE is booted");
            subs.updateAgentState(FcsEnumerations.FilterState.HOMING_TO_BE_DONE,
                    FcsEnumerations.FilterReadinessState.READY); 
        } else {
            FCSLOG.info(name + " SOME HARDWARE is MISSING");
        }
        FCSLOG.info(name + " BOOTED HARDWARE=" + listBootedNodes());        
        FCSLOG.info(name + ": END OF BOOT PROCESS");
    }
    
    /**
     * Used with JNI interface to initialize canInterface to be able to received PDOs.
     */
    @Command(type=Command.CommandType.QUERY, description="Initialize PDO for all my devices.")    
    public void initializePDOs() {
        //first step in our spiral development
        hardwareMapByNodeID.values().stream().forEach((poh) -> {
            try {
                poh.initializePDOs();
            } catch (DriverException ex) {
                this.raiseAlarm(HARDWARE_ERROR, " ERROR during PDO initialization : could not addReceivedPDOs", poh.getName(), ex);
            }
        });
    }
    
    /**
     * Update pdoData in sending a sync command. 
     * And update all the PieceOfHardware which are my children.
     * @return 
     */
    @Override
    public PDOData updatePDOData() {
        try {
            this.pdoData = this.canInterface.sync();
        } catch (DriverException ex) {
            this.raiseAlarm(HARDWARE_ERROR, " couldn't update PDOs", ex);
        }
        hardwareMapByNodeID.values().stream().forEach((poh) -> {
            try {
                poh.updateFromPDO(pdoData);
            } catch (DriverException ex) {
                this.raiseAlarm(HARDWARE_ERROR, " ERROR during PDO reading : could not update PDOs", poh.getName(), ex);
            }
        });        
        return pdoData;
    }
    
    @Override
    public void start() {
        subs.updateAgentState(FcsEnumerations.FilterState.WAITING_FOR_CWRAPPER, 
                FcsEnumerations.FilterReadinessState.NOT_READY);
        try {
            canInterface.init(master, baud, busName, masterNodeID); // for JNI
            Thread.sleep(hardwareBootTimeout);
            canInterface.start(); // for CWrapper
        } catch (DriverException | InterruptedException ex) {
            throw new RuntimeException(ex);
        }
        //has to be in postStart otherwise if it is in start and if a hardware doesn't boot FCS doesn't
        //start (Illegal State Transition)
//        bootProcess();
    }

    /**
     * check that all pieces of hardware is booted.
     */
    @Override
    public void postStart() {
        bootProcess();
        initializePDOs();
        //for the GUI
        this.publishData();
    }

    /**
     * This method is called in the checkHardware method during the
     * initialization process. It check devices at fixed rate with a scheduler
     * until all devices are booted or booting timeout is exceeded.
     */
    public void readDevicesInfo() {
        
        final long startTime = System.currentTimeMillis();
        final Runnable checkDevices = () -> {
            try {
                
                // Retrieve some information as serial number, vendor, for the CANopen devices
                // with command (info,node_id)
                retrieveHardwareInfo();
                this.publishData();
                long duration = System.currentTimeMillis() - startTime;
                FCSLOG.fine("duration="+duration + " : " + bootedDeviceNB + " booted devices.");
                if (duration > hardwareBootTimeout || this.bootedDeviceNB == hardwareMapByName.size()) {
                    hardwareBootProcessEnded = true;
                    cancelReadDevicesInfo();
                }
            } catch(Exception ex) {
                FCSLOG.error("ERRORR", ex);
            }
        };
        checkDevicesHandle = scheduler.scheduleWithFixedDelay(checkDevices,
                500, 500, TimeUnit.MILLISECONDS);
    }

    /**
     * This stops the reading of the sensors.
     *
     */
    private void cancelReadDevicesInfo() {
        lock.lock();
        try {
            FCSLOG.debug(name + " => stop waiting for devices info.");
            bootingCompleted.signalAll();
        } finally {
            lock.unlock();
        }
        checkDevicesHandle.cancel(true);
        FCSLOG.debug(name + " => readDevicesInfo canceled");
    }

    /**
     * This method waits until the booting process is completed. This methods is
     * called by postStart and has already acquired the lock.
     */
    private void waitForEndOfBooting() {
        while (!hardwareBootProcessEnded) {
            try {
                FCSLOG.info(name + " waiting until all pieces of hardware are booted.");
                bootingCompleted.await();
            } catch (InterruptedException ex) {
                FCSLOG.info(name + ": InterruptedException received=" + ex);
                break;
            }
        }
        FCSLOG.info(name + " STOP WAITING FOR END OF BOOTING PROCESS");
    }

    @Override
    public void checkStarted() {
        FCSLOG.info(name + " BEGIN checkStarted");
        this.bootProcess();
        FCSLOG.info(name + " END checkStarted");
    }

    /**
     * This shutdowns the scheduler. This is executed during CLOSING phase when
     * the CanOpenProxy is stopped. cf checkStopped
     */
    @Override
    public void shutdownNow() {
        try {
            canInterface.quit();
            canInterface.stop();
            scheduler.shutdown();
            subs.updateAgentState(FcsEnumerations.FilterState.OFF_LINE, 
                    FcsEnumerations.FilterReadinessState.NOT_READY);
        } catch (DriverException ex) {
            FCSLOG.error("cannot shut down properly", ex);
        }
    }

    /**
     * For a piece of hardware that this tcpProxy manages, this methods returns
     * the name of the device when the CANopen node id is given as argument.
     *
     * @param aNodeID
     * @return
     */
    public String getNodeName(int aNodeID) {
        if (this.hardwareMapByNodeID.get(aNodeID) == null) {
            return "UnknownDevice" + aNodeID;
        } else {
            return this.hardwareMapByNodeID.get(aNodeID).getName();
        }
    }



    /**
     * List the can open bootedNodes which are in the bootedNodes table.
     *
     * @return the list of can open bootedNodes and the information stored in
     * this.bootedNodes.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3,
            description = "Print the list of CANopen nodes which are booted on "
            + "the CAN bus. (values are in HEXA)")
    public String listBootedNodes() {
        if (this.bootedDeviceNB == 0) {
            return name + ": no booted CANopen devices.";
        } else {
            StringBuilder sb = new StringBuilder();
            for (PieceOfHardware poh: this.hardwareMapByName.values()) {
                if (poh.isBooted()) {
                    sb.append("\n==>");
                    sb.append(poh);
                }
            }
            return sb.toString();
        }
    }

    /**
     * Check that a nodeID entered by an end user at the console is a valid one:
     * nodeID should be in the hardwareMapByNodeID.
     *
     * @param nodeID
     */
    public void checkNodeID(int nodeID) {
        if (!hardwareMapByNodeID.containsKey(nodeID)) {
            throw new IllegalArgumentException(nodeID
                    + " is not in the hardware list for tcpProxy " + name);
        }
    }

    /**
     * Command to be used by the end user at the console. (for internal usage,
     * see writeSDO(String,String,String,String,String); Write a SDO message and
     * send it to the can open stack. The parameters are given in the following
     * format:
     *
     * @param nodeID FORMAT=hexa
     * @param index FORMAT=HEXA
     * @param subindex FORMAT=HEXA
     * @param size FORMAT=decimal between 0 and 4.
     * @param value FORMAT=decimal
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Send a CanOpen writeSDO command to the Can Bus. "
            + " \nValues of the argument are to be given in hexadecimal format for: index and subindex"
            + "\n and in decimal format for size and value."
            + "\n size represents the number of bytes on which the value is encoded. See device documentation.")
    public String writeSDO(int nodeID, int index, int subindex, int size, int value) {
        checkNodeID(nodeID);
        if (size < 0 || size > 4) {
            throw new IllegalArgumentException("size must be > 0 and < 4");
        }
        try {
            canInterface.wsdo(nodeID, index, subindex, size, value);
            return "OK";
        } catch (DriverException ex) {
            if (ex instanceof SDOException) {
                String errorName = CanOpenErrorsTable.getCommErrorNameByCode(Integer.valueOf(((SDOException)ex).getErrCode()));
                String deviceName = getNodeName(nodeID);
                this.raiseAlarm(SDO_ERROR, ex.getMessage() + " : " +errorName, deviceName);
            }
            throw new SDORequestException(name + " error in writeSDO for nodeID " + nodeID, ex);
        }
    }

    
    
    /**
     * Send a rsdo command to can interface and throw Exceptions or raise Alarms in case of something 
     * went wrong.
     * @param nodeID
     * @param index
     * @param subindex
     * @return 
     */
    public synchronized long readSDO(int nodeID, int index, int subindex) {
        try {
            return canInterface.rsdo(nodeID, index, subindex);
        } catch (SDOException ex) {
            if (ex.getErrCode()==0) {
                // The error code is zero, yet no data was received. 
                throw new ShortResponseToSDORequestException("rsdo request received a too short response", ex);
            } else {
                String errorName = CanOpenErrorsTable.getCommErrorNameByCode(Integer.valueOf(ex.getErrCode()));
                String deviceName = getNodeName(nodeID);
                this.raiseAlarm(SDO_ERROR, ex.getMessage() + " : " + errorName, deviceName);
                FCSLOG.error(ex);
                throw new SDORequestException(ex.getMessage());
            }
        } catch (DriverException ex) {
            String deviceName = getNodeName(nodeID);
            this.raiseAlarm(SDO_ERROR, ex.getMessage(), deviceName);
            FCSLOG.error(ex);
            throw new SDORequestException(ex.getMessage());
        }
    }

    
/**************************************************************************************************
 * Methods delegated to canInterface
 * ************************************************************************************************
 */
    
    /**
     * Send a command sync to CAN bus to read sensors by PDO.
     * @return 
     * @throws DriverException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3,
            description = "Send a command sync to CAN bus to read sensors by PDO.")
    public PDOData sync() throws DriverException {
        PDOData pdoD = this.canInterface.sync();
        FCSLOG.info("PDOData=" + pdoD.toString());
        return pdoD;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3,
            description = " init CAN interface.")
    public void init(int master, String baud, String busName, int nodeID) throws DriverException {
        canInterface.init(master, baud, busName, nodeID);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3,
            description = "add a receive PDO to canInterface.")    
    public void addReceivedPDO(int cobId) throws DriverException {
        canInterface.addReceivedPDO(cobId);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3,
            description = "Scan CAN bus.")    
    public int scan() throws DriverException {
        return canInterface.scan();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3,
            description = "Send a command info to CAN bus for nodeID to retrieve device information.")    
    public String info(int nodeID) throws DriverException {
        return canInterface.info(nodeID);
    }
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3,
            description = "Send a command wsdo to device.") 
    public void wsdo(int nodeId, int index, int subindex, int size, long data) throws DriverException {
        canInterface.wsdo(nodeId, index, subindex, size, data);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3,
        description = "Send a command rsdo to device.") 
    public long rsdo(int nodeId, int index, int subindex) throws DriverException {
        return canInterface.rsdo(nodeId, index, subindex);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3,
        description = "Send a command ssta (slave start) to device.")     
    public void ssta(int nodeId) throws DriverException {
        canInterface.ssta(nodeId);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3,
        description = "Send a command ssto (slave stop) to device.")    
    public void ssto(int nodeId) throws DriverException {
        canInterface.ssto(nodeId);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3,
        description = "Send a command reset to device.")     
    public void reset(int nodeId) throws DriverException {
        canInterface.reset(nodeId);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3,
        description = "Send a command quit to canInterface to device.")    
    public void quit() throws DriverException {
        canInterface.quit();
    }
    
/**************************************************************************************************
 * end of Methods delegated to canInterface
 * ************************************************************************************************
 */    

    /**
     * Identification of the hardware : we want to retrieve the information
     * stored in the hardware of the booted bootedNodes and update the array of
     * bootedNodes with this information. So for each booted node, we send a
     * command : - info,node_ID to the CWrapper and then we update the
     * bootedNodes list with the information returned in the command result. The
     * serial number is a main information to retrieve.
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Identification of the hardware : we want to retrieve the information stored "
                    + "in the hardware of the CANopen devices")
    public void retrieveHardwareInfo() {
        FCSLOG.info(name + ":Identification of the hardware");
        hardwareMapByName.values().stream().filter((poh) -> (!poh.isBooted())).forEach((poh) -> {
            updateDeviceInfo(poh);
        });
        publishData();
    }

    /**
     * send a message info,masterNodeID to the CANbus and update the CANopen device information 
     * with data received from the device.
     *
     * @param poh
     * @throws HardwareException
     */
    private void updateDeviceInfo(PieceOfHardware poh) {
        int nodeID = poh.getNodeID();
        String commandInfo = "info," + Integer.toHexString(nodeID);
        try {
            FCSLOG.debug(name + ":Sending to can open command :" + commandInfo);
            String result = canInterface.info(nodeID);
            FCSLOG.info("result="+ result);
            processInfoMessage(poh, result);
            
        } catch (DriverException ex) {
            if (!"tmpSensorsDevice".equals(poh.getName())) {
                this.hardwareIDError = true;
                this.raiseAlarm(HARDWARE_MISSING, " no response to command : " + commandInfo, 
                        poh.getName(), ex);
            } else {
                this.raiseWarning(HARDWARE_MISSING, " no response to command : " + commandInfo, 
                        poh.getName(), ex);
            }
            
        } catch (FcsHardwareException ex) {
            this.hardwareIDError = true;
            this.raiseAlarm(HARDWARE_ERROR, " Error in boot process for device." , poh.getName(), ex);
        }
    }

    /**
     * This method returns true if : all the hardware items are booted and
     * identified and the hardware have the node ID expected within the
     * configuration and the hardware is initialized.
     *
     * @return a boolean
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if all CANopen devices are booted and identified."
            + "Identified means that the serial numbers match the CANopen nodeID which reside in"
            + "configuration.")
    @Override
    public boolean allDevicesBooted() {
        boolean bootedDevicesNumberOK = this.bootedDeviceNB == hardwareMapByNodeID.size();
        FCSLOG.info(name + ":hardwareBootProcessEnded=" + hardwareBootProcessEnded);
        FCSLOG.info(name + ":canOpenNodeNumbersOK=" + bootedDevicesNumberOK);
        FCSLOG.info(name + ":hardwareIDError=" + this.hardwareIDError);
        return this.hardwareBootProcessEnded && bootedDevicesNumberOK && !this.hardwareIDError;
    }

    @Override
    public void onBootMessage(int nodeID) {
        if (hardwareMapByNodeID.containsKey(nodeID)) {
            PieceOfHardware poh = hardwareMapByNodeID.get(nodeID);
            if (poh.isBooted()) {
                /*the node is already booted */
                checkResetNode(nodeID, poh.getName());
            }
        } else {
            this.raiseAlarm(UNSYNC_BOOT, nodeID + ":UNKNOWN device. This device is not in the harware list.");
        }
    }

    private void checkResetNode(int nodeID, String deviceName) {
        if (hardwareBootProcessEnded) {
            /*the boot process is ended: something weird has just happened*/
            String cause = name + " received an unsynchronous boot message for node ID = " + nodeID
                    + " Has the device been reset or powered on ?";
            this.raiseWarning(UNSYNC_BOOT, deviceName, cause);
        }
    }

    /**
     * Process an emergency message received from a CANopen device on the status
     * bus : - ALARM or WARNING Alert is raised, - a message is logged, -
     * Observers are notified.
     *
     * @param nodeID
     * @param errCode
     * @param errReg
     */
    @Override
    public void onEmergencyMessage(int nodeID, int errCode, int errReg) {
       
        String deviceErrorName = CanOpenErrorsTable.getDeviceErrorNameByCode(errCode);
        String errorRegisterName = CanOpenErrorsTable.getErrorRegisterNameByCode(errReg);

        String deviceName = this.getNodeName(nodeID);
        EmergencyMessage emcyMsg = new EmergencyMessage(nodeID, deviceName, errCode ,
                deviceErrorName, errReg, errorRegisterName);
        // Pass emergency message to affected device
        PieceOfHardware poh = hardwareMapByNodeID.get(nodeID);
        if (poh != null) {
            // The use of the scheduler is required in order to leave the reader thread as soon as possible.
            subs.getScheduler().schedule(() -> {
                poh.onEmergencyMessage(emcyMsg);
            }, 0, TimeUnit.SECONDS);
        }
        FCSLOG.warning(name + " received EMERGENCY message=" + emcyMsg.toString() + " for nodeID=" + Integer.toHexString(nodeID));
        if (errReg == 0 || 0x20 == nodeID || 0x22 == nodeID) {
            //because tempsSensorDevice is always in error when it starts.
            //should be resolved in another way => setting canbus transmission speed for example.
            //proximitySensorsDevice sends also an emergency message deviceErrorCode: 0x3110
            this.raiseWarning(EMCY, emcyMsg.toString(), deviceName);
        } else {
            this.raiseAlarm(EMCY, emcyMsg.toString(), deviceName);
        }
    }

    /**
     * Process the response to a command info,node_ID.
     *
     * @param message
     */
    void processInfoMessage(PieceOfHardware poh, String message) {
        FCSLOG.debug(name + ":Received on socket command = " + message);
        String[] words = message.split(",", -1);
        int nodeID = Integer.parseInt(words[1], 16);

        FCSLOG.debug(name + ":checking serial number for nodeID " + Integer.toHexString(nodeID));
        String type = words[2];
        String vendor = words[3];
        String productCode = words[4];
        String revision = words[5];
        String serialNB = words[6];
        FCSLOG.info(name + "/" + poh.getName() + " informations read on device:" + " type=" + type + " vendor=" + vendor
                + " product code=" + productCode + " revision=" + revision + " serialNB" + serialNB);
        if (poh.getSerialNB().equals(serialNB)) {
            poh.setBooted(true);
            this.bootedDeviceNB++;
        } else {
            this.hardwareIDError = true;
            this.raiseAlarm(BAD_SERIAL_NB, " serial number read on device=" + serialNB
                + " ==>serial number expected=" + poh.getSerialNB() + " ==>device configuration=" + poh, poh.getName());
        }
    }

    /**
     * Return a String with the list of hardware expected in this subsystem. 
     * For debug purpose.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return a printed list of hardware expected in this subsystem.")
    @Override
    public String printHardwareList() {
        return hardwareMapByName.toString();
    }

    /**
     * Return a printed list of hardware with the initialization state. For
     * debug purpose.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return a printed list of hardware with the initialization state.")
    @Override
    public String printHardwareState() {
        StringBuilder sb = new StringBuilder(name + " CANopen devices : {");
        for (PieceOfHardware poh : this.hardwareMapByNodeID.values()) {
            sb.append(poh.printState());
            sb.append(";\n");
        }
        sb.append('}');
        return sb.toString();
    }

    /**
     * For the GUI : Return an Array containing hardware names handled by this
     * component.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return an Array containing hardware names handled by this component.")
    @Override
    public List<String> listHardwareNames() {
        List<String> res = new ArrayList<>();
        for (PieceOfHardware pieceOfHardware : this.hardwareMapByName.values()) {
            res.add(pieceOfHardware.getName());
        }
        return res;
    }

    /**
     * Overridden method from ClearAlertHandler interface to define conditions
     * when Alerts can be cleared.
     *
     * @param alert
     * @return
     */
    @Override
    public ClearAlertCode canClearAlert(Alert alert) {
        switch (alert.getAlertId()) {
            case "FCS001":
                try {
                    if (canInterface.isReady()) {
                        return ClearAlertCode.CLEAR_ALERT;
                    }
                } catch (DriverException ex) {
                    FCSLOG.error(ex);
                }
                return ClearAlertCode.DONT_CLEAR_ALERT;
            case "FCS002":
                return ClearAlertCode.CLEAR_ALERT;

            default:
                return ClearAlertCode.UNKWNOWN_ALERT;
        }
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Publish booting information for all CANopen devices.")
    @Override
    public void publishData() {
        FCSLOG.info(name + " is publishing:" + bootedDeviceNB);
        subs.publishSubsystemDataOnStatusBus(new KeyValueData(name, bootedDeviceNB));
        for (PieceOfHardware device : hardwareMapByName.values()) {
            device.publishData();
        }
    }




    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("name=");
        sb.append(this.name);
        sb.append("/hardwareBootTimeout=");
        sb.append(this.hardwareBootTimeout);
        return sb.toString();
    }

    // -- BridgeToHardware interface methods -----------------------------------

    @Override
    public boolean isReady() {
        try {
            return canInterface.isReady() && allDevicesBooted();
        } catch (DriverException ex) {
            return false;
        }
    }

    @Override
    public AlertService getAlertService() {
        return subs.getAlertService();
    }


}
