
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.PersistencyService;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;

/**
 * This class is the Main Module when carousel is in standalone mode.
 * 
 * CarouselModle implements HardwareController because we must initialize acPLutoGateway during
 * initialization Phase.
 * 
 * @author virieux
 */
public class CarouselMain extends MainModule implements FilterLocator {
    
    @LookupField(strategy=Strategy.CHILDREN)
    private Carousel carousel;
    
    @LookupField(strategy=Strategy.BYNAME)
    private FilterHolder autochanger;
     
    @LookupField(strategy=Strategy.CHILDREN)
    private FilterManager filterManager;
    
    @LookupField(strategy=Strategy.TREE)
    private PersistencyService persistenceService;
    
    private final boolean loadFilterLocationAtStartup = true;
    private final boolean loadFilterLocationAtShutdown = true;
    
    private PieceOfHardware fakePlutoGateway;
    
    /**
     * Build a CarouselMainModule.
     * @param bridge
     * @param plutoGateway 
     */
    public CarouselMain(
            BridgeToHardware bridge,
            PieceOfHardware plutoGateway) {
        super(bridge);
        this.fakePlutoGateway = plutoGateway;
    }

    @Override
    public FilterManager getFilterManager() {
        return filterManager;
    }

    @Override
    public Carousel getCarousel() {
        return carousel;
    }

    @Override
    public PersistencyService getPersistenceService() {
        return persistenceService;
    }
    
    @Override
    public void init() {
        if (!(autochanger instanceof FilterHolder)) {
            final String MSG = name + " ==> autochanger doesn't implements FilterHolder -"
                    + " Please fix groovy description file or CarouselMainModule implementation.";
            FCSLOG.error(MSG);
            throw new IllegalArgumentException(MSG);
        }
        /* define a role for my subsystem in order to make CarouselGUI listen to my subsystem*/
        subs.setAgentProperty("org.lsst.ccs.subsystem.fcs.carousel", "carousel");
        persistenceService.setAutomatic(loadFilterLocationAtStartup, loadFilterLocationAtShutdown);
    }
    
    
    /**
     * Update state in reading sensors.
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Update state in reading sensors.")
    @Override
    public void updateStateWithSensors()  {
        super.updateStateWithSensors();
        this.carousel.updateStateWithSensors();
        this.autochanger.updateStateWithSensors();
    }
    
    /**
     * Check that Carousel hardware is ready to be operated and moved.
     * This means that :
     * - all CAN open devices are booted, identified and initialized,
     * - homing has been done on the controllers.
     * This updates the FCS state and FCS readyness state and publishes on the status bus.
     * 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, 
            description = "Update FCS state and FCS readyness state and publishes on the status bus.")
    @Override
    public void updateFCSStateToReady() {
        if (carousel.isInitialized()) {
            /* The initialization has been done, so now the hardware is ready */
            subs.updateAgentState(FcsEnumerations.FilterState.HARDWARE_READY, 
                FcsEnumerations.FilterReadinessState.READY);
        }
    }
 
    @Override
    public void postStart() {
        //ne va pas marcher car le postStart de CarouselMain est exécuté avant que le matériel soit initialisé.
        //solution d'Etienne : changer le CanOpenProxy en bougeant le bootProcess dans la méthode start().
        checkFiltersLocation();
//        fakePlutoGateway.initializeAndCheckHardware();
    }


    @Override
    public void shutdownNow() {
        /*to save filter location in a file*/
//        this.saveFilterLocation();
        super.shutdownNow();
    }


    
}
