package org.lsst.ccs.subsystems.fcs;

import java.time.Duration;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.locks.Condition;
import java.util.logging.Level;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.AC_SENSOR_ERROR;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.ControlledBySensors;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenCallTimeoutException;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * The Three Online clamps which holds a filter when it is at ONLINE position
 * can ve viewed as a single object. It's the goal of this class to represent
 * this object. Opening or closing the 3 clamps at ONLINE is not the same action
 * as opening or closing online clamps one by one.
 *
 * @author virieux
 */
public class AutochangerThreeOnlineClamps extends MobileItem implements ControlledBySensors {
    
    @LookupField(strategy = Strategy.TREE)
    private Autochanger autochanger;
    private final AutochangerOnlineClamp onlineClampXminus;
    private final AutochangerOnlineClamp onlineClampXplus;
    private final AutochangerOnlineClamp onlineClampYminus;
    private final AutochangerOnlineClamp[] clampsList;

    private FcsEnumerations.LockStatus lockStatus = LockStatus.UNKNOWN;

    private final Condition stateUpdated = lock.newCondition();

    /* This is used when we update the clamps
     state with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;

    /* Tools for current ramp*/
    private ScheduledFuture<?> currentRampHandle;
    
    
    @LookupField(strategy = Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;  

    @ConfigurationParameter(description = "timeout in milliseconds : if closing the clamps last more "
            + "than this amount of time, then the subsystem goes in ERROR.")
    private int timeoutForLockingClamps = 15000;

    @ConfigurationParameter(description = "timeout in milliseconds : if unlocking the clamps last more "
            + "than this amount of time, then the subsystem goes in ERROR.")
    private int timeoutForUnlockingClamps = 15000;
    
    @ConfigurationParameter(description = "maximum time in milliseconds to lock the 3 clamps")
    private int maxTimeToLockAllClamps = 3000;
    
    @ConfigurationParameter(description = "maximum time in milliseconds to unlock the 3 clamps")
    private int maxTimeToUnlockAllClamps = 3000;
    
    @ConfigurationParameter(description = "maximum time in milliseconds to open the 3 clamps")
    private int maxTimeToOpenClampsX = 2000;
    
    @ConfigurationParameter(description = "maximum time in milliseconds to close the 3 clamps")
    private int maxTimeToCloseClampsX = 2000;
    
    @ConfigurationParameter(description = "minimal period for current ramps. should be > 50")
    private int minPeriod = 200;

    /**
     * Create a AutochangerThreeOnlineClamps with 3 AutochangerOnlineClamp.
     *
     * @param onlineClampXminus
     * @param onlineClampXplus
     * @param onlineClampYminus
     */
    public AutochangerThreeOnlineClamps(
            AutochangerOnlineClamp onlineClampXminus,
            AutochangerOnlineClamp onlineClampXplus,
            AutochangerOnlineClamp onlineClampYminus) {
        this.onlineClampXminus = onlineClampXminus;
        this.onlineClampXplus = onlineClampXplus;
        this.onlineClampYminus = onlineClampYminus;
        clampsList = new AutochangerOnlineClamp[]{onlineClampXminus, onlineClampXplus, onlineClampYminus};
    }

    /**
     * set a minimal value for current ramps period
     * @param minPeriod 
     */
    @ConfigurationParameterChanger
    public void setMinPeriod(int minPeriod) {
        if (minPeriod < 50) {
            throw new IllegalArgumentException(minPeriod + " bad value for minPeriod. should be > 50");
        } else {
            this.minPeriod = minPeriod;
        }
    }
    
    

    /**
     * Return lockStatus If lockStatus is being updated, wait until update
     * completion else return immediately lockStatus.
     *
     * @return
     */
    public FcsEnumerations.LockStatus getLockStatus() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.warning(name + ": interrupted in getLockStatus.", ex);
                }

            }
            return lockStatus;

        } finally {
            lock.unlock();
        }
    }

    /**
     * Returns true if LockStatus=LOCKED
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if the 3 clamps are LOCKED.")
    public boolean isLocked() {
        return onlineClampXminus.isLocked()
                && onlineClampXplus.isLocked()
                && onlineClampYminus.isLocked();
    }

    /**
     * Returns true if LockStatus=CLOSED
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if the 3 clamps are CLOSED.")
    public boolean isClosed() {
        return onlineClampXminus.isClosed()
                && onlineClampXplus.isClosed()
                && onlineClampYminus.isClosed();
    }

    /**
     * Returns true if LockStatus=OPENED
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if the 3 clamps are OPENED.")
    public boolean isOpened() {
        return onlineClampXminus.isOpened()
                && onlineClampXplus.isOpened()
                && onlineClampYminus.isOpened();
    }

    /**
     * @return true if the 3 clamps are in Travel between opened position and
     * closed position.
     */
    private boolean isInTravel() {
        return onlineClampXminus.isInTravel()
                && onlineClampXplus.isInTravel()
                && onlineClampYminus.isInTravel();
    }

    /**
     * Returns true if LockStatus=ERROR
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if one of the clamp is in error.")
    @Override
    public boolean isInError() {
        return onlineClampXminus.isInError()
                || onlineClampXplus.isInError()
                || onlineClampYminus.isInError();
    }

    /**
     * Return true if the 3 onlineClamps hardware is initialized.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the 3 onlineClamps hardware is ready : controllers ready, "
            + "controllers parameters checked and controllers configured.")
    public boolean isInitialized() {
        return onlineClampXminus.isInitialized() && onlineClampXplus.isInitialized()
                && onlineClampYminus.isInitialized();
    }

    /**
     * Check that the 3 controllers of the clamps are initialized and that
     * parameters are correct.
     */
    private void enableAndCheckControllersBeforeAction() {
        for (AutochangerOnlineClamp clamp : clampsList) {
            clamp.enableAndCheckControllerBeforeAction();
        }
    }
    
    

    /**
     * Closes and locks the online clamps. This command first release brakes,
     * sends currentToClose to the 3 clamps controllers and then increases
     * progressively current until currentToClamp is reached for every clamps
     * controllers.
     *
     *
     * state before this action : OPENED or CLOSED (if CLOSED, nothing has to be
     * done) state after this action : LOCKED
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
            description = "Closes and locks the online clamps.",
            timeout = AutochangerOnlineClamp.TIMEOUT_FOR_CLOSING)
    public void closeAndLockClamps() {
        updateStateAndCheckSensors();
        //TODO test conditions for closing clamps
        if (isOpened()) {
            onlineClampYminus.close();
            /*See startAction */
            this.executeAction(MobileItemAction.CLOSE_AND_LOCK_ONLINECLAMPS, timeoutForLockingClamps);

        } else if (isLocked()) {
            FCSLOG.info(name + " is already LOCKED - nothing to do.");

        } else {
            throw new RejectedCommandException(name + " has to be OPENED before a closeAndLockClamps");
        }
    }

    /**
     * Unlock and open the online clamps. initial state = LOCKED final state =
     * OPENED This action is not
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
            description = "Unlock and open the online clamps.",
            timeout = AutochangerOnlineClamp.TIMEOUT_FOR_OPENING)
    public void unlockAndOpenClamps() {
        updateStateAndCheckSensors();
        enableAndCheckControllersBeforeAction();
        autochanger.checkConditionsForActioningOnlineClamps();
        this.currentAction = MobileItemAction.UNLOCK_AND_OPEN_ONLINECLAMPS;
        //TODO test conditions for unlocking clamps
        //ce n'est pas une MobileItemAction parce que ça ne marchait pas.
        if (this.isLocked()) {
            doUnlock();
            updateStateAndCheckSensors();
            if (!isClosed()) {
                throw new FailedCommandException(name + "couldn't unlock clamps");
            }
            onlineClampYminus.open();
            openClampsX();
            //TODO 2 lignes suivantes inutiles car déjà fait. A tester sur matériel.
            updateStateAndCheckSensors();
            activateBrakesAndSetCurrentToZero();
            if (!isOpened()) {
                throw new FailedCommandException(name + "couldn't open clamps");
            }            
            
        } else if (this.isOpened()) {
            FCSLOG.info(name + " clamps already UNLOCKED and OPENED - nothing to do");

        } else {
            throw new RejectedCommandException(name + " has to be LOCKED before unlockAndOpen action.");
        }
    }
    
    /**
     * test periodicTask.
     */
    public void testPeriodTask() {
        periodicTaskService.scheduleAgentPeriodicTask(
        new AgentPeriodicTask(name+"-updateStateAndCheckSensors",this::updateStateAndCheckSensors)
                .withIsFixedRate(true)
                .withLogLevel(Level.WARNING)
                .withPeriod(Duration.ofMillis(5000)));
    }

    /**
     * Closes the 3 clamps with a small pressure. initial state = OPENED final
     * state = CLOSED
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Close the 3 online clamps.",
            timeout = AutochangerOnlineClamp.TIMEOUT_FOR_CLOSING)
    public void closeClamps() {
        
        updateStateAndCheckSensors();
        //TODO test conditions for closing clamps
        if (isOpened()) {
            /*See startAction */            
            onlineClampYminus.close();
            this.executeAction(MobileItemAction.CLOSE_ONLINECLAMPS, timeoutForLockingClamps);

        } else if (isClosed()) {
            FCSLOG.info(name + " clamps already CLOSED nothing to do");

        } else {
            throw new RejectedCommandException(name + " has to be unlocked before.");
        }
    }


    @Command(type = Command.CommandType.ACTION, description = "close clamps X",
            timeout = AutochangerOnlineClamp.TIMEOUT_FOR_CLOSING)
    public void closeClampsX() {
        checkParametersToCloseClampsX();
        this.currentAction = MobileItemAction.CLOSE_ONLINECLAMPS;
        onlineClampXminus.enableAndCheckControllerBeforeAction();
        onlineClampXminus.getController().doReleaseBrake();
        onlineClampXplus.enableAndCheckControllerBeforeAction();
        onlineClampXplus.getController().doReleaseBrake();
        onlineClampXminus.sendCurrentToControllerAndSaveValue(onlineClampXminus.getInitialCurrentToClose());
        onlineClampXplus.sendCurrentToControllerAndSaveValue(onlineClampXplus.getInitialCurrentToClose());
        /*tempo de waitTime et rampe de initialCurrentToClose à finalCurrentToClose*/
        FcsUtils.sleep(onlineClampXminus.getWaitTimeToClose(), name);
        this.executeCurrentRampToCloseClampsX();
        this.updateStateAndCheckSensors();
        onlineClampXminus.getController().activateBrake();
        onlineClampXplus.getController().activateBrake();
        onlineClampXminus.getController().writeCurrent((short)0);
        onlineClampXplus.getController().writeCurrent((short)0);
    }

    /**
     * Opens the 3 clamps with a small pressure. initial state = CLOSED final
     * state = OPENED
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Opens the 3 online clamps with a small pressure.",
            timeout = AutochangerOnlineClamp.TIMEOUT_FOR_OPENING)
    public void openClamps() {
        updateStateAndCheckSensors();
        //TODO test conditions for opening clamps
        if (isClosed()) {
            checkReadyForAction();
            checkParametersToOpenClampsX();
            this.currentAction = MobileItemAction.OPEN_ONLINECLAMPS;
            onlineClampYminus.open();
            openClampsX();

        } else if (isOpened()) {
            FCSLOG.info(name + " clamps already OPENED. Nothing to do.");
            throw new RejectedCommandException(name + " clamps already OPENED. Nothing to do.");

        } else {
            throw new RejectedCommandException(name + " has to be closed before.");
        }
    }



    @Command(type = Command.CommandType.ACTION, description = "open softly clampX",
            timeout = AutochangerOnlineClamp.TIMEOUT_FOR_OPENING)
    public void openClampsX() {
        checkParametersToOpenClampsX();
        this.currentAction = MobileItemAction.OPEN_ONLINECLAMPSX;
        onlineClampXminus.enableAndCheckControllerBeforeAction();
        onlineClampXplus.enableAndCheckControllerBeforeAction();
        /* At the beginning clampsX are CLOSED*/
        onlineClampXminus.sendCurrentToControllerAndSaveValue(onlineClampXminus.getFinalCurrentToClose());
        onlineClampXplus.sendCurrentToControllerAndSaveValue(onlineClampXplus.getFinalCurrentToClose());
        FcsUtils.sleep(500, name);
        onlineClampXminus.getController().doReleaseBrake();
        onlineClampXplus.getController().doReleaseBrake();
        this.executeAction(currentAction, timeoutForUnlockingClamps);
    }

    /**
     * Locks clamps : closed with a strong pressure (high current). The clamps
     * have to be CLOSED. At the end of this action, the clamps are CLOSED but a
     * strong pressure to hold safely the clamps.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Locks clamps : closed with a strong pressure (high current).",
            timeout = AutochangerOnlineClamp.TIMEOUT_FOR_CLOSING)
    public void lockClamps() {
        updateStateAndCheckSensors();
        if (isClosed()) {
            this.executeAction(MobileItemAction.LOCK_ONLINECLAMPS, timeoutForLockingClamps);

        } else if (isLocked()) {
            FCSLOG.info(name + " is already LOCKED. Nothing to do.");

        } else {
            throw new RejectedCommandException(name + " have to be CLOSED before to be locked.");
        }
    }


    /**
     * Unlocks clamps : slows down current sent to controller in order to
     * decrease pressure on the clamps. The clamps have to be LOCKED. At the end
     * of this action, the clamps are CLOSED with a small pressure of the clamp
     * hardware on the filter frame.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "unlocks clamps : decreases current in controller to decrease pressure.",
            timeout = AutochangerOnlineClamp.TIMEOUT_FOR_OPENING)
    public void unlockClamps() {
        updateStateAndCheckSensors();
        if (isLocked()) {
            this.executeAction(MobileItemAction.UNLOCK_ONLINECLAMPS, timeoutForLockingClamps);

        } else if (isClosed()) {
            FCSLOG.info(name + " is already CLOSED. Nothing to do.");

        } else {
            throw new RejectedCommandException(name + " have to be CLOSED before to be locked.");
        }
    }


    /**
     * Sends current to all controllers to close clamps and save sentCurrent
     * value. Controllers have been enabled and brakes released when this method
     * is called. used for actions CLOSE_AND_LOCK_ONLINECLAMPS and
     * CLOSE_ONLINECLAMPS see startAction
     */
    private void sendCurrentToCloseAllClamps() {
        FCSLOG.info(name + " sending current to close all clamps");
        for (AutochangerOnlineClamp clamp : clampsList) {
            clamp.sendCurrentToControllerAndSaveValue(clamp.getFinalCurrentToClose());
        }
    }

    /**
     * Sends current to all controllers in order to lock clamps and save
     * sentCurrent value. Controllers have been enabled and brakes released when
     * this method is called.
     */
    private void sendCurrentToLockAllClamps() {
        for (AutochangerOnlineClamp clamp : clampsList) {
            clamp.sendCurrentToControllerAndSaveValue(clamp.getCurrentToClamp());
        }
    }

    /**
     * Return true if the 3 onlineClamps hardware is ready.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the 3 onlineClamps hardware is ready.")
    @Override
    public boolean myDevicesReady() {
        return onlineClampXminus.myDevicesReady() && onlineClampXplus.myDevicesReady()
                && onlineClampYminus.myDevicesReady();
    }

    /**
     * returns true if the action given as argument is completed. The completion
     * of action has been confirmed by sensors.
     *
     * @param action
     * @return
     */
    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        if (action == MobileItemAction.CLOSE_ONLINECLAMPS
                || action == MobileItemAction.UNLOCK_ONLINECLAMPS) {
            /* for action CLOSE_AND_LOCK_ONLINECLAMPS, lockStatus is set to LOCKED in postAction*/
            /* but the action started in method startAction is completed when clamps are CLOSED*/
            return isClosed();

        } else if (action == MobileItemAction.UNLOCK_AND_OPEN_ONLINECLAMPS
                || action == MobileItemAction.OPEN_ONLINECLAMPS) {
            return isOpened();

        } else if (action == MobileItemAction.LOCK_ONLINECLAMPS
                || action == MobileItemAction.CLOSE_AND_LOCK_ONLINECLAMPS) {
            return isLocked();
            
        } else if (action == MobileItemAction.OPEN_ONLINECLAMPSX) {
            return onlineClampXminus.isOpened() && onlineClampXplus.isOpened();

        } else {
            throw new IllegalArgumentException(name + " invalid action:" + action);
        }
    }

    /**
     * reads sensors, computes new lockStatus. used in MobileItem to check
 that action started in startAction is completed.
     */
    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        autochanger.updateStateWithSensors();
    }

    @Override
    public void startAction(FcsEnumerations.MobileItemAction action) {
        enableAndCheckControllersBeforeAction();
        autochanger.checkConditionsForActioningOnlineClamps();
        autochanger.increaseCurrentMonitoring();
        if (action == MobileItemAction.CLOSE_AND_LOCK_ONLINECLAMPS) {
            /* for action CLOSE_AND_LOCK_ONLINECLAMPS, first we close and then we lock, 
             the locking is done in postAction */
            closeClampsX();
            doLock();

        } else if (action == MobileItemAction.CLOSE_ONLINECLAMPS) {
            closeClampsX();

        } else if (action == MobileItemAction.UNLOCK_AND_OPEN_ONLINECLAMPS) {
            /* clamps are LOCKED with a high pressure (currentToClamp previously sent to controllers)*/
            /* to avoid an elastic return we send currentToClamp BEFORE the brakes are released. */

            openClampsX();

        } else if (action == MobileItemAction.UNLOCK_ONLINECLAMPS) {
            /* clamps are LOCKED with a high pressure (currentToClamp previously sent to controllers)*/
            /* to avoid an elastic return we send currentToClamp BEFORE the brakes are released. */
            doUnlock();

        } else if (action == MobileItemAction.LOCK_ONLINECLAMPS) {
            /* clamps are CLOSED*/
            doLock();

        } else if (action == MobileItemAction.OPEN_ONLINECLAMPS) {
            /* tests on hardware without MobileItem in october 2017*/

        } else if (action == MobileItemAction.OPEN_ONLINECLAMPSX) {
           this.executeCurrentRampToOpenClampsX();
            

        } else {
            throw new IllegalArgumentException(name + " invalid action:" + action);
        }
    }

    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay) {
        if (action == MobileItemAction.OPEN_ONLINECLAMPSX) {
            onlineClampXminus.getController().activateBrake();
            onlineClampXplus.getController().activateBrake();
            onlineClampXminus.getController().writeCurrent(0);
            onlineClampXplus.getController().writeCurrent(0);
        } else {
            /* activate brakes on the 3 clamps controllers and set current to 0*/
            activateBrakesAndSetCurrentToZero();
        }
        autochanger.decreaseCurrentMonitoring();
    }

    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public void postAction(FcsEnumerations.MobileItemAction action) {
        //nothing more to be done than in abortAction
    }

    /**
     * activate brakes on the 3 clamps controllers and set current to 0
     */
    private void activateBrakesAndSetCurrentToZero() {
        for (AutochangerOnlineClamp clamp : clampsList) {
            clamp.getController().activateBrake();
            /*
             /* in this case we use writeCurrent method of EPOSController instead of sendCurrentToControllerAndSaveValue
             /* because we don't want to keep on disk this last value of current sent to controller (0).
             */
            clamp.getController().writeCurrent((short)0);
        }
    }

    /**
     * releases brakes on the 3 clamps controllers
     */
    private void doReleaseBrakes() {
        for (AutochangerOnlineClamp clamp : clampsList) {
            clamp.getController().doReleaseBrake();
        }
    }

    private void doLock() {
        sendCurrentToCloseAllClamps();
        FcsUtils.sleep(500, name);
        doReleaseBrakes();
        executeCurrentRampToLockAllClamps();
    }

    /**
     * unlock clamps : clamps are LOCKED with a high pressure (currentToClamp
     * previously sent to controllers) avoid an elastic return we send
     * currentToClamp BEFORE the brakes are released.
     */
    private void doUnlock() {
        sendCurrentToLockAllClamps();
        FcsUtils.sleep(500, name);
        doReleaseBrakes();
        executeCurrentRampToUnlockAllClamps();
    }

    /**
     * Creates an object to be published on the status bus.
     *
     * @return
     */
    public StatusDataPublishedByAutochangerThreeClamps createStatusDataPublishedByThreeClamps() {
        StatusDataPublishedByAutochangerThreeClamps status = new StatusDataPublishedByAutochangerThreeClamps();
        status.setName(name);
        status.setLockStatus(lockStatus);
        return status;
    }

    @Override
    public void publishData() {
        s.publishSubsystemDataOnStatusBus(
                new KeyValueData("autochangerClamps", createStatusDataPublishedByThreeClamps()));
    }

    private void updateStateAndCheckSensors() {
        autochanger.updateStateWithSensors();
        checkSensors(AC_SENSOR_ERROR, name);
    }

    /**
     * This methods updates lockStatus from the values return by the sensors.
     * This values are given in an array of hexa values as arguments of the
     * method.
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Update state.")
    public void updateState() {
        lock.lock();
        try {
            updatingState = true;
            for (AutochangerOnlineClamp clamp : clampsList) {
                clamp.updateState();
            }
            computeLockStatus();

        } finally {

            updatingState = false;
            stateUpdated.signalAll();
            lock.unlock();
            this.publishData();
        }
    }

    /**
     * Compute lockStatus.
     */
    private void computeLockStatus() {
        if (isInError()) {
            this.lockStatus = LockStatus.ERROR;

        } else if (isLocked()) {
            this.lockStatus = LockStatus.LOCKED;

        } else if (isOpened()) {
            this.lockStatus = LockStatus.OPENED;

        } else if (isClosed()) {
            this.lockStatus = LockStatus.CLOSED;

        } else if (isInTravel()) {
            this.lockStatus = LockStatus.INTRAVEL;

        } else {
            /* if lockStatus is not the same for the 3 clamps we are in this case.
             It can happen during the closing (or the opening) of the 3 clamps when a clamp is already closed 
             but not the 2 others. 
             */
            this.lockStatus = LockStatus.UNKNOWN;
        }
    }

    private void executeCurrentRampToCloseClampsX() {
        writeCurrentRampToCloseClampsX();
        waitForEndOfCurrentRamp(timeoutForLockingClamps);
    }

    private void executeCurrentRampToOpenClampsX() {
        FCSLOG.info(name + " executeCurrentRampToOpenClampsX");
        writeCurrentRampToOpenClampsX();
        waitForEndOfCurrentRamp(timeoutForUnlockingClamps);
    }

    /**
     * Sends current alternativaly to the 3 clamps from finalCurrentToClose to
     * currentToClamp and waits until the task is completed.
     *
     */
    private void executeCurrentRampToLockAllClamps() {
        FCSLOG.info(name + " executing current ramp to lock all clamps.");
        writeCurrentRampToLockAllClamps();
        waitForEndOfCurrentRamp(timeoutForLockingClamps);

    }

    private void executeCurrentRampToUnlockAllClamps() {
        writeCurrentRampToUnlockAllClamps();
        waitForEndOfCurrentRamp(timeoutForUnlockingClamps);
    }

    /**
     * cancels current ramp send signal to all threads waiting for
     * currentRampEnded condition.
     */
    private void cancelCurrentRamp() {
        FCSLOG.finest(" => stop writing current");
        this.currentRampHandle.cancel(true);
        FCSLOG.finest(" => current ramp ended");
    }

    /**
     * This method waits until the newCurrentValue ramp is completed.
     */
    private void waitForEndOfCurrentRamp(int timeout) {
        FCSLOG.info(name + " BEGIN WAITING FOR END OF CURRENT RAMP");
        try {
            currentRampHandle.get(timeout, TimeUnit.SECONDS);
        } catch (CancellationException ex) {
            if (currentRampHandle.isDone()) {
                FCSLOG.info(name + " CURRENT RAMP ENDED");
            } else {
                FCSLOG.error(name + " during action " + this.currentAction 
                        + " waiting was cancelled before END of CURRENT RAMP "+ ex);
            }
        } catch (TimeoutException ex) {
            FCSLOG.error(name + " CURRENT RAMP IS DURING TOO LONG - timeout=" + timeout, ex);

        } catch (InterruptedException ex) {
            FCSLOG.error(name + " interrupted while waiting end of current ramp during action=" + currentAction, ex);
            throw new FcsHardwareException(name + " interrupted while waiting end of current ramp during action=" + currentAction, ex);

        } catch (ExecutionException ex) {
            FCSLOG.error(name + " error during action=" + currentAction, ex);
            throw new FcsHardwareException(name + " error during action=" + currentAction + " cause=" + ex);

        } finally {
            cancelCurrentRamp();
            FCSLOG.info(name + " STOP WAITING FOR END OF CURRENT RAMP");
        };
    }

    private void writeCurrentRampToCloseClampsX() {
        writeCurrentRampToClampsX(onlineClampXminus.getInitialCurrentToClose(),
                onlineClampXminus.getFinalCurrentToClose(),
                onlineClampXminus.getIncrementCurrentToClose(),
                onlineClampXplus.getInitialCurrentToClose(),
                onlineClampXplus.getFinalCurrentToClose(),
                onlineClampXplus.getIncrementCurrentToClose(),
                maxTimeToCloseClampsX
        );
    }

    /**
     * sends current alternativaly to clamps X- and X+ with a ramp of current
     * from initialCurrent to finalCurrent. Uses a scheduler.
     */
    private void writeCurrentRampToOpenClampsX() {
        FCSLOG.info(name + " writeCurrentRampToOpenClampsX");
        writeCurrentRampToClampsX(onlineClampXminus.getFinalCurrentToClose(),
                onlineClampXminus.getCurrentToOpen(),
                onlineClampXminus.getIncrementCurrentToOpen(),
                onlineClampXplus.getFinalCurrentToClose(),
                onlineClampXplus.getCurrentToOpen(),
                onlineClampXplus.getIncrementCurrentToOpen(),
                maxTimeToOpenClampsX
        );
    }

    private void writeCurrentRampToLockAllClamps() {
        writeCurrentRampToAllClamps(
                onlineClampYminus.getFinalCurrentToClose(),
                onlineClampYminus.getCurrentToClamp(),
                onlineClampYminus.getIncrementCurrentToClamp(),
                onlineClampXminus.getFinalCurrentToClose(),
                onlineClampXminus.getCurrentToClamp(),
                onlineClampXminus.getIncrementCurrentToClamp(),
                onlineClampXplus.getFinalCurrentToClose(),
                onlineClampXplus.getCurrentToClamp(),
                onlineClampXplus.getIncrementCurrentToClamp(),
                this.maxTimeToLockAllClamps
        );
    }

    private void writeCurrentRampToUnlockAllClamps() {
        writeCurrentRampToAllClamps(
                onlineClampYminus.getCurrentToClamp(),
                onlineClampYminus.getFinalCurrentToClose(),
                onlineClampYminus.getIncrementCurrentToClamp(),
                onlineClampXminus.getCurrentToClamp(),
                onlineClampXminus.getFinalCurrentToClose(),
                onlineClampXminus.getIncrementCurrentToClamp(),
                onlineClampXplus.getCurrentToClamp(),
                onlineClampXplus.getFinalCurrentToClose(),
                onlineClampXplus.getIncrementCurrentToClamp(),
                this.maxTimeToUnlockAllClamps
        );
    }
 
    @Command(type=Command.CommandType.QUERY)
    public void checkParametersToOpenClampsX() {
        int initialValueXminus = onlineClampXminus.getFinalCurrentToClose();
        int finalValueXminus = onlineClampXminus.getCurrentToOpen();
        int incrementCurrentXminus = onlineClampXminus.getIncrementCurrentToOpen();
        int initialValueXplus = onlineClampXplus.getFinalCurrentToClose();
        int finalValueXplus = onlineClampXplus.getCurrentToOpen();
        int incrementCurrentXplus = onlineClampXplus.getIncrementCurrentToOpen();        
        
        final int nbStepXminus = Math.abs((finalValueXminus - initialValueXminus) / incrementCurrentXminus);
        final int nbStepXplus = Math.abs((finalValueXplus - initialValueXplus) / incrementCurrentXplus);
        final int period = this.maxTimeToOpenClampsX / Math.min(nbStepXminus, nbStepXplus);
        if (period < minPeriod) {
            int newMaxTime = minPeriod * Math.min(nbStepXminus, nbStepXplus);
            throw new IllegalArgumentException(maxTimeToOpenClampsX + 
                    " : value too low for maxTimeToOpenClampsX. should be >= " + newMaxTime);
        }
        if (nbStepXminus != nbStepXplus) {
            FCSLOG.finest(name + ": nbstep to open are not equal for the 2 clamps X");
        }
    }    
    
    @Command(type=Command.CommandType.QUERY)
    public void checkParametersToCloseClampsX() {
        int initialValueXminus = onlineClampXminus.getInitialCurrentToClose();
        int finalValueXminus = onlineClampXminus.getFinalCurrentToClose();
        int incrementCurrentXminus = onlineClampXminus.getIncrementCurrentToClose();
        int initialValueXplus = onlineClampXplus.getInitialCurrentToClose();
        int finalValueXplus = onlineClampXplus.getFinalCurrentToClose();
        int incrementCurrentXplus = onlineClampXplus.getIncrementCurrentToClose();        
        
        final int nbStepXminus = Math.abs((finalValueXminus - initialValueXminus) / incrementCurrentXminus);
        final int nbStepXplus = Math.abs((finalValueXplus - initialValueXplus) / incrementCurrentXplus);
        final int period = this.maxTimeToCloseClampsX / Math.min(nbStepXminus, nbStepXplus);
        if (period < minPeriod) {
            int newMaxTime = minPeriod * Math.min(nbStepXminus, nbStepXplus);
            throw new IllegalArgumentException(maxTimeToCloseClampsX 
                    + " : value too low. should be > " + newMaxTime);
        }
        if (nbStepXminus != nbStepXplus) {
            FCSLOG.finest(name + ": nbstep to open are not equal for the 2 clamps X");
        }
    }

    private void writeCurrentRampToClampsX(int initialValueXminus, int finalValueXminus, int incrementCurrentXminus,
            int initialValueXplus, int finalValueXplus, int incrementCurrentXplus, int maxTime) {
        int stepHeightXminus = FcsUtils.getSignedStepHeight(initialValueXminus, finalValueXminus, incrementCurrentXminus);
        int stepHeightXplus = FcsUtils.getSignedStepHeight(initialValueXplus, finalValueXplus, incrementCurrentXplus);
        final int nbStepXminus = Math.abs((finalValueXminus - initialValueXminus) / incrementCurrentXminus);
        final int nbStepXplus = Math.abs((finalValueXplus - initialValueXplus) / incrementCurrentXplus);
        final int period = maxTime / Math.min(nbStepXminus, nbStepXplus);
        FCSLOG.finest("############################################################");
        FCSLOG.finest("initialValue for clampXminus=" + initialValueXminus);
        FCSLOG.finest("finalValue for clampXminus=" + finalValueXminus);
        FCSLOG.finest("initialValue for clampXplus=" + initialValueXplus);
        FCSLOG.finest("finalValue for clampXplus=" + finalValueXplus);
        FCSLOG.finest("period=" + period);
        FCSLOG.finest("nbStepXminus=" + nbStepXminus);
        FCSLOG.finest("stepHeightXplus=" + stepHeightXplus);
        FCSLOG.finest("nbStepXminus=" + nbStepXminus);
        FCSLOG.finest("nbStepXplus=" + nbStepXplus);
        FCSLOG.finest("############################################################");

        final Runnable currentRamp = new Runnable() {
            private int newCurrentValueXminus = initialValueXminus;
            private int newCurrentValueXplus = initialValueXplus;
            private boolean finalValueReachedXminus = false;
            private boolean finalValueReachedXplus = false;
            @Override
            public void run() {
                newCurrentValueXminus = FcsUtils.computeNewCurrentValue(stepHeightXminus, newCurrentValueXminus, finalValueXminus);
                newCurrentValueXplus = FcsUtils.computeNewCurrentValue(stepHeightXplus, newCurrentValueXplus, finalValueXplus);
                
                try {
                    if (!finalValueReachedXminus) {
                        onlineClampXminus.sendCurrentToControllerAndSaveValue(newCurrentValueXminus);
                    }
                    if (!finalValueReachedXplus) {
                        onlineClampXplus.sendCurrentToControllerAndSaveValue(newCurrentValueXplus);
                    }
                } catch (CanOpenCallTimeoutException ex) {
                    FCSLOG.error(ex);
                } finally {
                    finalValueReachedXminus = newCurrentValueXminus == finalValueXminus;
                    finalValueReachedXplus = newCurrentValueXplus == finalValueXplus;
                }
                
                if (finalValueReachedXminus && finalValueReachedXplus) {
                    cancelCurrentRamp();
                }
            }
        };
        this.currentRampHandle = scheduler.scheduleAtFixedRate(currentRamp, 0, period, TimeUnit.MILLISECONDS);
    }

    /**
     * Sends current alternativaly to the 3 clamps controllers with a ramp of
     * current Uses a sheduler.
     */
    private void writeCurrentRampToAllClamps(
            int initialValueYminus, int finalValueYminus, int incrementYminus,
            int initialValueXminus, int finalValueXminus, int incrementXminus,
            int initialValueXplus, int finalValueXplus, int incrementXplus, int maxTime) {
        int stepHeightYminus = FcsUtils.getSignedStepHeight(initialValueYminus, finalValueYminus, incrementYminus);
        int stepHeightXminus = FcsUtils.getSignedStepHeight(initialValueXminus, finalValueXminus, incrementXminus);
        int stepHeightXplus = FcsUtils.getSignedStepHeight(initialValueXplus, finalValueXplus, incrementXplus);
        final int nbStepYminus = Math.abs((finalValueYminus - initialValueYminus) / stepHeightYminus);
        final int nbStepXminus = Math.abs((finalValueXminus - initialValueXminus) / stepHeightXminus);
        final int nbStepXplus = Math.abs((finalValueXplus - initialValueXplus) / stepHeightXplus);
        final int period = maxTime / nbStepYminus;
        if (period < 200) {
            throw new IllegalArgumentException(period + " : bad value for period");
        }

        FCSLOG.finest("############################################################");
        FCSLOG.finest("initialValue for clampYminus=" + initialValueYminus);
        FCSLOG.finest("finalValue for clampYminus=" + finalValueYminus);
        FCSLOG.finest("initialValue for clampXminus=" + initialValueXminus);
        FCSLOG.finest("finalValue for clampXminus=" + finalValueXminus);
        FCSLOG.finest("initialValue for clampXplus=" + initialValueXplus);
        FCSLOG.finest("finalValue for clampXplus=" + finalValueXplus);
        FCSLOG.finest("nbStepYminus=" + nbStepYminus + " nbStepXminus=" + nbStepXminus + " nbStepXplus=" + nbStepXplus);
        FCSLOG.finest("period=" + period);
        FCSLOG.finest("stepHeightYminus=" + stepHeightYminus);
        FCSLOG.finest("stepHeightXminus=" + stepHeightXminus);
        FCSLOG.finest("stepHeightXplus=" + stepHeightXplus);
        FCSLOG.finest("############################################################");

        final Runnable currentRamp = new Runnable() {
            private int newCurrentValueYminus = initialValueYminus;
            private int newCurrentValueXminus = initialValueXminus;
            private int newCurrentValueXplus = initialValueXplus;
            boolean finalValueReachedYminus = false;
            boolean finalValueReachedXminus = false;
            boolean finalValueReachedXplus = false;

            @Override
            public void run() {
                newCurrentValueYminus = FcsUtils.computeNewCurrentValue(stepHeightYminus, 
                        newCurrentValueYminus, finalValueYminus);
                newCurrentValueXminus = FcsUtils.computeNewCurrentValue(stepHeightXminus, 
                        newCurrentValueXminus, finalValueXminus);  
                newCurrentValueXplus = FcsUtils.computeNewCurrentValue(stepHeightXplus, 
                        newCurrentValueXplus, finalValueXplus);                
                try {
                    if (!finalValueReachedYminus) {
                        onlineClampYminus.sendCurrentToControllerAndSaveValue(newCurrentValueYminus);
                    }
                    if (!finalValueReachedXminus) {
                        onlineClampXminus.sendCurrentToControllerAndSaveValue(newCurrentValueXminus);
                    }
                    if (!finalValueReachedXplus) {
                        onlineClampXplus.sendCurrentToControllerAndSaveValue(newCurrentValueXplus);
                    }
                }  catch (CanOpenCallTimeoutException ex) {
                    FCSLOG.error(ex);
                }  finally {
                    finalValueReachedYminus = newCurrentValueYminus == finalValueYminus;
                    finalValueReachedXminus = newCurrentValueXminus == finalValueXminus;
                    finalValueReachedXplus = newCurrentValueXplus == finalValueXplus;
                }             

                if (finalValueReachedYminus && finalValueReachedXminus && finalValueReachedXplus) {
                    cancelCurrentRamp();
                } 
            }
        };
        this.currentRampHandle = scheduler.scheduleAtFixedRate(currentRamp, 0, period, TimeUnit.MILLISECONDS);
    }
}
