
package org.lsst.ccs.subsystems.fcs.singlefiltertest;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.commons.annotations.LookupName;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.AutoChangerTrucksLocation;
import org.lsst.ccs.subsystems.fcs.DigitalSensor;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;

/**
 * A truck is a mobile part in the autochanger. This is a model for the
 * single-filter-test autochanger truck. It's the mechanical item which holds a
 * filter through a latch and moves the filter from STANDBY position to
 * STANDBACK and back. There is 2 trucks in the autochanger, one on each side of
 * the filter. A truck move along a rail. The position of a truck is given by 2
 * sensors (hall effect sensor) which are located on the rails. One is at
 * STANDBY position, the other is at STANDBACK position. The trucks are moved by
 * a motor. They are moved together by a motor which synchronizes the motion of
 * both trucks.
 *
 * @author virieux
 */
public class Truck {

    @LookupName
    protected String name;
    
    private AutoChangerTrucksLocation truckLocation = AutoChangerTrucksLocation.UNKNOWN;

    /**
     * A effect Hall sensor which says if the truck is at standback position or
     * not.
     */
    protected DigitalSensor railSensorStandback;

    /**
     * A effect Hall sensor which says if the truck is at standby position or
     * not.
     */
    protected DigitalSensor railSensorStandby;

    /**
     * ************************************************************************************************
     */
    /**
     * **********Tools for the synchronization of the truck position update
     * reading the 2 sensors *****
     */
    /**
     * ************************************************************************************************
     */
    private volatile boolean updatingTruckLocation = false;
    //When we update the truck position reading the 2 sensors, 
    //we want to take a lock on the truck until
    //the end of the update, so any other thread has to wait for this update.
    private final Lock lock = new ReentrantLock();
    private final Condition truckLocationUpdated = lock.newCondition();

    /**
     * Build a new TruckModule with a tickmillis of 3000.
     * @param railSensorStandback
     * @param railSensorStandby 
     */
    public Truck(DigitalSensor railSensorStandback, DigitalSensor railSensorStandby) {
        this.railSensorStandback = railSensorStandback;
        this.railSensorStandby = railSensorStandby;
    }

    /**
     * ***************************************************************************************************
     */
    /**
     * *******END of Tools for the synchronization of the truck position update
     * reading the 2 sensors ****
     */
    /**
     * ***************************************************************************************************
     */
    /**
     * ***********************************************************************************************
     */
    /**
     * ******************** SETTERS AND GETTERS  *****************************************************
     */
    /**
     * ***********************************************************************************************
     */
    /**
     * 
     * @return 
     */
    public DigitalSensor getRailSensorStandback() {
        return railSensorStandback;
    }

    public void setRailSensorStandback(DigitalSensor railSensorStandback) {
        this.railSensorStandback = railSensorStandback;
    }

    public DigitalSensor getRailSensorStandby() {
        return railSensorStandby;
    }

    public void setRailSensorStandby(DigitalSensor railSensorStandby) {
        this.railSensorStandby = railSensorStandby;
    }

    /**
     * ***********************************************************************************************
     */
    /**
     * ******************** END OF SETTERS AND GETTERS  **********************************************
     */
    /**
     * ***********************************************************************************************
     */

    /**
     * This method returns the truck position (an ENUM). If the truck position
     * is being updated it waits until the end of the update.
     *
     * @return truckLocation
     */
    public AutoChangerTrucksLocation getTruckLocation() {

        lock.lock();
        try {
            while (updatingTruckLocation) {
                try {
                    this.truckLocationUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(ex);
                }

            }
            return truckLocation;

        } finally {
            lock.unlock();
        }

    }

    /**
     * This methods updates the field truckLocation in parsing the values of the
     * 8 channels of the DIO device.
     *
     * @param hexaValueOfInputs
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void updateLocation(int hexaValueOfInputs)   {

        lock.lock();

        try {
            updatingTruckLocation = true;
            FCSLOG.debug(name + " is updating its location in parsing the values sent by rail sensors:" + hexaValueOfInputs);

            this.railSensorStandback.updateValue(hexaValueOfInputs);
            this.railSensorStandby.updateValue(hexaValueOfInputs);

            if (railSensorStandby.isOn() && !railSensorStandback.isOn()) {
                truckLocation = AutoChangerTrucksLocation.STANDBY;

            } else if (!railSensorStandby.isOn() && railSensorStandback.isOn()) {
                truckLocation = AutoChangerTrucksLocation.STANDBACK;
                
            } else if (railSensorStandby.isOn() && railSensorStandback.isOn()) {
                truckLocation = AutoChangerTrucksLocation.ERROR;
                throw new FcsHardwareException(name
                        + " ERROR IN READING RAIL SENSORS : a truck can't be at standby AND at standback ");
                
            } else {
                truckLocation = AutoChangerTrucksLocation.UNKNOWN;
            }
            FCSLOG.debug("truckLocation=" + truckLocation.toString());

        } finally {

            updatingTruckLocation = false;
            truckLocationUpdated.signal();
            lock.unlock();
        }
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(this.name);
        sb.append("/rail sensor at STANDBACK=");
        if (this.railSensorStandback == null) {
            sb.append("NULL");
        } else {
            sb.append(this.railSensorStandback.getName());
        }
        sb.append("/rail sensor at STANDBY=");
        if (this.railSensorStandby == null) {
            sb.append("NULL");
        } else {
            sb.append(this.railSensorStandby.getName());
        }
        return sb.toString();
    }
}
