
package org.lsst.ccs.subsystems.fcs.singlefiltertest;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.AutoChangerTrucksLocation;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceStatus;
import org.lsst.ccs.subsystems.fcs.Filter;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * This class is for the Single Filter Test autochanger. The autochanger in
 * Single Filter Test has no flip rail and no online clamp.
 *
 * To move the autochanger trucks in the Single Filter Test we have 2 pneumatic
 * cylinders each moved by an actuator : when we want the truck to be in stanby
 * position, the standby actuator has to be ON and the standback actuator has to
 * be OFF, when we want the truck to be in standback position, the standback
 * actuator has to be ON and the standby actuator has to be OFF.
 *
 * @author virieux
 */
public class SftAutoChanger extends BasicAutoChanger {

    protected SftTruckMotor sftTrucksMotor;

    private final long timeToConfortAPosition = 2000; 
    private final long timeToGoToStandby = 10000;
    private final long timeToGoToStandback = 10000;

    /**
     * Build a SftAutoChangerModule
     * @param latchXminus
     * @param latchXplus
     * @param truckXminus
     * @param truckXplus
     * @param railsSensorsDIOName
     * @param filterSensorsDIOName
     * @param sftTrucksMotor 
     */
    public SftAutoChanger(
            SftFilterLatch latchXminus,
            SftFilterLatch latchXplus,
            Truck truckXminus,
            Truck truckXplus,
            String railsSensorsDIOName,
            String filterSensorsDIOName,
            SftTruckMotor sftTrucksMotor) {
        super(latchXminus, latchXplus, truckXminus, truckXplus, railsSensorsDIOName, filterSensorsDIOName);
        this.sftTrucksMotor = sftTrucksMotor;
    }

    /**
     * ***********************************************************************************************
     */
    /**
     * ******************** SETTERS AND GETTERS  *****************************************************
     */
    /**
     * ***********************************************************************************************
     */
    
    /**
     * 
     * @return 
     */
    public SftTruckMotor getSftTrucksMotor() {
        return sftTrucksMotor;
    }


    //Used by SimuActuatorModule
    public long getTimeToConfortAPosition() {
        return timeToConfortAPosition;
    }

    //Used by SimuActuatorModule
    public long getTimeToGoToStandback() {
        return timeToGoToStandback;
    }

    //Used by SimuActuatorModule
    public long getTimeToGoToStandby() {
        return timeToGoToStandby;
    }
    
    


    /**
     * ***********************************************************************************************
     */
    /**
     * ******************** END OF SETTERS AND GETTERS  **********************************************
     */
    /**
     * ***********************************************************************************************
     * @return 
     */
    @Override
    public boolean isMovingToStandback() {
        return this.sftTrucksMotor.isMovingToStandback();
    }

    @Override
    public boolean isMovingToStandby() {
        return this.sftTrucksMotor.isMovingToStandby();
    }

    @Override
    public boolean isMoving() {
        return isMovingToStandback() || isMovingToStandby();
    }

    @Override
    public String toString() {
        return name;
    }


    /**
     * Move SFT autochanger to STANDBACK.
     * @return
     * @throws FcsHardwareException 
     */
    @Override
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, 
            description = "Go to STANDBACK")
    public String goToStandback()  {

        checkPreConditionsForTrucksMotion();
        updateTrucksLocationWithSensors();
        if (this.isAtStandback()) {
            return name + " trucks already at standback location";
        }

        if (this.getTrucksLocation().equals(AutoChangerTrucksLocation.ERROR)) {
            FCSLOG.error(name + " Cannot execute goToStandback : ERROR in reading rails sensors");
            //TODO : if trucks at ERROR position in normal mode, put the subsystem in error.
            throw new RejectedCommandException(name + " Cannot execute goToStandback : ERROR in reading rails sensors.");
        }

        if (this.isAtStandby()) {

            //if both the carousel and the autochanger holds the filter we can't move.
            if (!this.isTrucksEmpty() && this.getLatchesState() == FcsEnumerations.LockStatus.LOCKED) {
                this.carousel.updateClampsStateWithSensors();
                if (carousel.isHoldingFilterAtStandby()) {
                    throw new RejectedCommandException(name
                            + " can't move filter to standback because the carousel is holding the filter at standby.");
                }
            }

            // to confort the position at standby
            // here the Thread.sleep can't be avoid
            FCSLOG.debug(name + ": The position is being conforted");
            sftTrucksMotor.moveToStandby();
            try {

                Thread.sleep(timeToConfortAPosition);
            } catch (InterruptedException ex) {
                FCSLOG.debug(ex);
            }
        }
        FCSLOG.info(name + " IS GOING TO STANDBACK POSITION");
        sftTrucksMotor.moveToStandback();

        try {
            //fcslog.debug(name + ": Waiting for the trucks to go to STANDBACK");
            Thread.sleep(timeToGoToStandback);
        } catch (InterruptedException ex) {
            FCSLOG.debug(ex);
        }
        this.sftTrucksMotor.off();

        updateTrucksLocationWithSensors();

        if (this.isAtStandback()) {
            FCSLOG.debug("====> " + name + "  IS AT STANDBACK POSITION");
            return name + " IS AT STANDBACK POSITION";
        } else {
            throw new FailedCommandException(name + "could not go to STANDBACK POSITION");
        }
    }

    /**
     * This methods moves the trucks from standback position to standby
     * position. It first checks the preconditions for a motion of the trucks.
     * Then it checks if the trucks are in standback position otherwise throws a
     * RejectedCommandException. If the trucks are at standback, it first conforts
     * the position (standbackMotor.on) then it moves the trucks to the standby
     * position (standbackMotor.off then standbyMotor.on). The motor is stopped
     * after the temporisation has expired. The command is completed when the
     * position sensors at standby says the trucks are in standby.
     *
     * @return
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, 
            description = "Go to STANDBY.")
    @Override
    public String goToStandby()  {

        checkPreConditionsForTrucksMotion();

        updateTrucksLocationWithSensors();

        if (this.isAtStandby()) {
            return name + "trucks already at standby position";
        }

        if (this.getTrucksLocation().equals(AutoChangerTrucksLocation.UNKNOWN)) {
            FCSLOG.error(name + " Cannot execute goToStandby : Trucks are between standback and standby location.");
            throw new RejectedCommandException(name + " Cannot execute goToStandby : Trucks are between standback and standby location.");
        }

        if (this.getTrucksLocation().equals(AutoChangerTrucksLocation.ERROR)) {
            FCSLOG.error(name + " Cannot execute goToStandby : ERROR in reading rails sensors");
            //TODO : if trucks at ERROR position in normal mode, put the subsystem in error.
            throw new RejectedCommandException(name + " Cannot execute goToStandby : ERROR in reading rails sensors.");
        }

        // to confort the position
        if (this.isAtStandback()) {

            sftTrucksMotor.moveToStandback();

            try {
                FCSLOG.debug(name + ": Waiting for the position to be conforted");
                Thread.sleep(timeToConfortAPosition);
            } catch (InterruptedException ex) {
                FCSLOG.error(ex);
            }
        }
        FCSLOG.info(name + " IS GOING TO STANDBY POSITION");
        sftTrucksMotor.moveToStandby();
        try {
            FCSLOG.debug(name + ": Waiting for the trucks to go to STANDBY");
            Thread.sleep(timeToGoToStandby);
        } catch (InterruptedException ex) {
            FCSLOG.error(ex);
        }
        this.sftTrucksMotor.off();

        updateTrucksLocationWithSensors();

        if (this.isAtStandby()) {

            FCSLOG.debug("====> " + name + "  IS AT STANDBY POSITION");
            return name + " IS AT STANDBY POSITION";
        } else {
            throw new FailedCommandException(name + " could not go to STANDBY POSITION");
        }

    }

    /**
     * Move a filter to STANDBACK position.
     * @param aFilter
     * @return
     * @throws FcsHardwareException 
     */
    @Override
    public String moveFilterToStandback(Filter aFilter)  {
        if (isTrucksEmpty()) {
            throw new RejectedCommandException(name + " unable to moveFilterToStandback: Autochanger is empty");
        }
        goToStandback();
        this.updateLatchesStateWithSensors();
        publishDataAndNotifyObservers();
        return name + ": " + aFilter.getName() + " is at standback position.";
    }

    /**
     * Move filter at STANDBY position.
     * @param aFilter
     * @return
     * @throws FcsHardwareException
     * @throws FailedCommandException
     */
    @Override
    public String moveFilterToStandby(Filter aFilter)  {
        if (isTrucksEmpty()) {
            throw new RejectedCommandException(name + "/moveFilterToStandby: Autochanger is empty");
        }
        carousel.updateClampsStateWithSensors();

        if (!carousel.isReadyToGrabAFilterAtStandby()) {
            throw new RejectedCommandException(name
                    + "can't move a filter to STANDBY because carousel is not ready to grab a filter.");
        }
        goToStandby();
        this.updateLatchesStateWithSensors();
        publishDataAndNotifyObservers();
        return name + ": " + aFilter.getName() + " is at standby position.";
    }


    /**
     * Read sensors and update latches and trucks state.
     * @throws FcsHardwareException
     */
    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, 
            description = "Read sensors and update latches and trucks state")
    public void updateStateWithSensors()  {
        this.updateLatchesStateWithSensors();
        this.updateTrucksLocationWithSensors();
        publishData();

        if (this.getLatchXminus().getPresenceStatus().equals(FilterPresenceStatus.NOFILTER)
                && this.getLatchXplus().getPresenceStatus().equals(FilterPresenceStatus.NOFILTER)) {
            this.setTrucksEmpty(true);
            this.setFilterOnTrucks(null);

        }
    }

    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay)  {
        throw new UnsupportedOperationException("Not supported yet."); 
    }



}
