
package org.lsst.ccs.subsystems.fcs.singlefiltertest;

import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenDAC;

/**
 * 
 * Used only for the single-filter-test hardware.
 * 
 * This is a model for a motor which locks or unlocks a latch. The motor is
 * driven by a CAN-CBX-AO412. We sent to the CAN-CBX-AO412 a current value.
 *
 * @author virieux
 */
public class CanOpenLatchActuator extends LatchActuator {

    /**
     * The DAC which controls the motor.
     */
    private CanOpenDAC dac;

    /**
     * The numero of analog output where we send the value of the current to the
     * motor. (variable)
     */
    @ConfigurationParameter
    private int numOfAnalogOutputForSpeedVoltage;

    /**
     * The numero of analog output where we send the value of the current limit
     * to the motor. current limit (fixed)
     */
    @ConfigurationParameter
    private int numOfAnalogOutputForCurrentLimit;

    /**
     * Build a new CanOpenLatchActuatorModule
     * @param currentLimitInVolt
     * @param speedVoltageInVolt
     * @param speedVoltageMinValue
     * @param speedVoltageMaxValue
     * @param dac
     * @param numOfAnalogOutputForSpeedVoltage
     * @param numOfAnalogOutputForCurrentLimit 
     */
    public CanOpenLatchActuator(
            CanOpenDAC dac,
            double currentLimitInVolt,
            double speedVoltageInVolt,
            int speedVoltageMinValue,
            int speedVoltageMaxValue,
            int numOfAnalogOutputForSpeedVoltage,
            int numOfAnalogOutputForCurrentLimit) {
        super(currentLimitInVolt, speedVoltageInVolt, speedVoltageMinValue, speedVoltageMaxValue);
        this.dac = dac;
        this.numOfAnalogOutputForSpeedVoltage = numOfAnalogOutputForSpeedVoltage;
        this.numOfAnalogOutputForCurrentLimit = numOfAnalogOutputForCurrentLimit;
    }

    public void setNumOfAnalogOutputForSpeedVoltage(int numOfAnalogOutputForSpeedVoltage) {
        this.numOfAnalogOutputForSpeedVoltage = numOfAnalogOutputForSpeedVoltage;
    }

    public void setNumOfAnalogOutputForCurrentLimit(int numOfAnalogOutputForCurrentLimit) {
        this.numOfAnalogOutputForCurrentLimit = numOfAnalogOutputForCurrentLimit;
    }

    @Override
    public String open()  {

        //set the current limit
        int currentLimitInDecimal = (int) (this.currentLimitInVolt * CanOpenDAC.bitOverVolts);
        dac.writeAnalogOutput(this.numOfAnalogOutputForCurrentLimit, currentLimitInDecimal);
        //start the motor
        int speedVoltageInDecimal = (int) (this.speedVoltageInVolt * CanOpenDAC.bitOverVolts);
        dac.writeAnalogOutput(this.numOfAnalogOutputForSpeedVoltage, speedVoltageInDecimal);

        return name + " is OPEN";
    }

    @Override
    public String close()  {

        //set the current limit
        int currentLimitInDecimal = (int) (this.getCurrentLimitInVolt() * CanOpenDAC.bitOverVolts);
        dac.writeAnalogOutput(this.numOfAnalogOutputForCurrentLimit, currentLimitInDecimal);
        //start the motor
        int speedVoltageInDecimal = (int) (this.getSpeedVoltageInVolt() * CanOpenDAC.bitOverVolts);
        dac.writeAnalogOutput(this.numOfAnalogOutputForSpeedVoltage, -speedVoltageInDecimal);
        return name + " is CLOSED";
    }

    @Override
    public String powerOff()  {
        dac.writeAnalogOutput(this.numOfAnalogOutputForCurrentLimit, 0);
        dac.writeAnalogOutput(this.numOfAnalogOutputForSpeedVoltage, 0);
        return name + " is POWER OFF";
    }

}
