
package org.lsst.ccs.subsystems.fcs.drivers;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.net.ServerSocket;
import java.net.Socket;
import org.lsst.ccs.AlertService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.drivers.canopenjni.BootMessageListener;
import org.lsst.ccs.drivers.canopenjni.CanOpenInterface;
import org.lsst.ccs.drivers.canopenjni.EmergencyMessageListener;
import org.lsst.ccs.drivers.canopenjni.PDOData;
import org.lsst.ccs.drivers.canopenjni.SDOException;
import org.lsst.ccs.drivers.commons.DriverException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.utils.CWrapperUtils;
import org.lsst.ccs.utilities.beanutils.WrappedException;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * This Modules : - start a tcpip server on a portNumber, - waits for the
 * connection of the client (which is supposed to be a Can OPen C wrapper), -
 * send commands received by the call method to the client and - send back the
 * client response. - it handles the asynchronous messages coming from the
 * client and for that it starts a thread which listen to the client.
 *
 * @author virieux
 */
public class FcsTcpProxy implements CanOpenInterface {
    
    /**
    * A Logger for CommandDispenser.
    */
    private static final Logger COMMAND_LOG = Logger.getLogger("org.lsst.ccs.subsystems.fcs.drivers.FcsTcpProxy");
    
    @LookupName
    protected String name;
    
    @LookupField(strategy = Strategy.TOP)
    protected Subsystem subs;
    
    @LookupField(strategy=Strategy.TREE)
    private AlertService alertService;

    @ConfigurationParameter(isFinal=true,range="1024..99000",
        description="tcpip port number on which this tcpip server starts on.")
    private int portNumber;

    @ConfigurationParameter(description="Timeout in Millisecond. If a command sent to the CANbus doesn't "
            + "respond during this amount of time, we considere that they could be an issue on the hardware.")    
    private int fieldBusTimeout;
    
    private ServerSocket serverSock;
    private Thread readerThread;
    private volatile boolean stopped = true;
    private volatile boolean stopping = false;
    private volatile boolean tcpServerStarted = false;
    private ClientContext clientContext;

    private CommandDispenser commandDispenser;
    
    @ConfigurationParameter(isFinal=true)
    private String clientName;

    @LookupField(strategy=Strategy.ANCESTORS)
    private CanOpenEventListener listener;

    @Override
    public void addReceivedPDO(int cobId) throws DriverException {

    }

    @Override
    public void setEmergencyMessageListener(EmergencyMessageListener eml) throws DriverException {
        // set differently
    }

    @Override
    public void setBootMessageListener(BootMessageListener bml) throws DriverException {
        // set differently
    }
    
    /**
     * A class to handle the tcpip connexion.
     */
    private static class ClientContext {

        private String clientName;
        private BufferedReader reader;
        private BufferedWriter writer;
        protected Socket socket;

        ClientContext(String name, Socket socket, BufferedReader reader, OutputStream os) {
            try {
                this.clientName = name;
                this.reader = reader;
                this.socket = socket;
                writer = new BufferedWriter(new OutputStreamWriter(os, "ISO-8859-1"), 256);
            } catch (UnsupportedEncodingException e) {
                /*. SHOULD NOT HAPPEN */
                FCSLOG.error(name + ":context not started", e);
                throw new Error(e);
            }
        }
    }

    /**
     * Build a new FcsTcpProxy with a tcpip port number and a tickMillis value 3000.
     * And a fieldBusTimeout value 6000;
     * @param clientName the client name
     * @param portNumber 
     * @param fieldBusTimeout 
     */
    public FcsTcpProxy(String clientName, int portNumber, int fieldBusTimeout) {
        this.portNumber = portNumber;
        this.fieldBusTimeout = fieldBusTimeout;
        this.clientName = clientName;
    }

    public AlertService getAlertService() {
        return alertService;
    }

    @Override
    public void init() {
        stopped = false;
        stopping = false;
        tcpServerStarted = false;
        commandDispenser = new CommandDispenser();
    }

    @Override
    public void start() {
        startServer();
        startThreadReader();
    }
    
    @Override
    public void stop() {
        stopServer();
    }
    
    /**
     * Starts the server tcp on the port portNumber. And waits for a client
     * connection.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, 
            description = "Starts the tcp server.")
    public void startServer() {
        try {
            serverSock = new ServerSocket(portNumber);
            FCSLOG.info(name + ":SERVER STARTED ON PORT:" + portNumber);
        } catch (IOException e) {
            FCSLOG.error(name + ":server not started", e);
            throw new WrappedException(e);
        }

        try {
            FCSLOG.info(name + ":WAITING FOR C-WRAPPER CLIENT...");

            subs.updateAgentState(FcsEnumerations.FilterState.WAITING_FOR_CWRAPPER, 
                    FcsEnumerations.FilterReadinessState.NOT_READY);
            FCSLOG.debug(name + ":DEBUGGING MODE");
            Socket sock = serverSock.accept();
            FCSLOG.info(name + ":socket accept on " + portNumber);
            BufferedReader reader = new BufferedReader(
                    new InputStreamReader(sock.getInputStream(), "ISO-8859-1"), 256);

            String nameAndProtocol = reader.readLine();
            if (nameAndProtocol == null) {
                FCSLOG.error(name + ": nameAndProtocol is null : CWrapper disconnected ?. Bye.");
                stopServer();
                
            } else {
                String[] words = nameAndProtocol.split(" ");
                String cname = words[0];

                clientContext = new ClientContext(cname, sock, reader, sock.getOutputStream());
                FCSLOG.info(name + ":REGISTERED : " + cname);
                tcpServerStarted = true;
                stopped = false;
            }

        } catch (IOException e) {
            // LOG
            FCSLOG.error(name + " unexpected ", e);
        }
    }

    /**
     * Starts a thread which read on the tcp socket, waiting for messages coming
     * from tcp proxy. If the message has a known token (already registred in
     * commandDispenser) it is a response to a can open command that we had sent
     * to the proxy. If the token of the message is null, that means that is's
     * an unsynchronous message coming from the can open stack and transmitted
     * directly from the tcp proxy. That can be : - a boot message :
     * "boot,nodeID" - an emergency message : "emcy,80+NodeID"
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, 
            description = "Starts to listen to the tcp client.")
    public void startThreadReader() {
        FCSLOG.info(name + "startThreadReader");
        if (!tcpServerStarted) {
            String errorMsg = name + " tcp server has to be started before startThreadReader.";
            throw new FcsHardwareException(errorMsg);
        }
        /*We start thread to read from tcpProxy*/
        Runnable readingFromTcpProxy;
        readingFromTcpProxy = new Runnable() {
            @Override
            public void run() {
                while (!stopped) {
                    try {
                        String readline = clientContext.reader.readLine();
                        if (readline == null) {
                            if (stopping) {
                                FCSLOG.info(name + " : CWrapper is stopped.");
                            } else {
                                FCSLOG.error(name + " :nameAndProtocol is null : that should not happen.");
                            }
                        } else {
                            processCommand(readline);
                        }

                    } catch (IOException ex) {
                        FCSLOG.error(ex);
                        FCSLOG.error(name + ": Error in Thread reading from the tcp client.");
                    }
                }
            }
        };
        FCSLOG.info(name + ":STARTING Thread reader");
        readerThread = new Thread(readingFromTcpProxy);
        readerThread.start();
    }
    
    /**
     * Processes a command received from the tcp client.
     * 
     */
    protected void processCommand(String command) {
        COMMAND_LOG.finest(name + ":message read from socket=" + command);
        String[] words = command.split(",");
        String commandWord = words[0];

        if (command.startsWith("emcy")) {
            /*NodeID without the leading zeros*/
            int nodeID = Integer.parseInt(words[1], 16);
            int deviceErrorCode = Integer.parseInt(words[2], 16);
            int errReg = Integer.parseInt(words[3], 16);
            COMMAND_LOG.warning(name + " received EMERGENCY message=" + command + " for nodeID=" + nodeID);
            listener.onEmergencyMessage(nodeID,deviceErrorCode, errReg);
        } else if (command.startsWith("Unknown")) {
            COMMAND_LOG.info(name+  ": Unknown command received by CWrapper:" + command);
        } else if (command.startsWith("boot")) {
            int nodeID = Integer.parseInt(words[1],16);
            FCSLOG.info(name + ":boot message received for nodeID (in hexa): " + Integer.toHexString(nodeID));
            listener.onBootMessage(nodeID);
        } else if (command.startsWith("sync")) {
            processCommandResponse(command, "sync");
        } else {    
            int nodeID = Integer.parseInt(words[1],16);
            processCommandResponse(command,commandWord + Integer.toHexString(nodeID));
        }
    }
    
    private void processCommandResponse(String command, String token) {
        COMMAND_LOG.finest("Command token=" + token);
        if (commandDispenser.isTokenUsed(token)) {
            COMMAND_LOG.finest("Response to a registred command:" + command);
            try {
                commandDispenser.registerResponse(token, command);
            } catch (Exception ex) {
                FCSLOG.error(ex);
                FCSLOG.error(name + ":Error in registering response to command :" + command, ex.getCause());
            }
        } else {
            FCSLOG.warning(name + ":Unsynchronous message read from CANbus=" + command);
        }
    }

    /**
     * Stops the tcp server.
     */
    public void stopServer() {
        if (!tcpServerStarted) {
            FCSLOG.warning(name + " is stopped; nothing to stop.");
            return;
        }
        FCSLOG.info(name + ": ABOUT TO STOP TCP server.....");
        FCSLOG.finest("clientName=" + clientContext.clientName);
        stopping = true;
        try {
            call("quit");
        } catch (Exception ex) {
            throw new FcsHardwareException("cannot stop CWrapper", ex);
        }
        stopped = true;
        readerThread.interrupt();

        try {
            serverSock.close();
            this.tcpServerStarted = false;
            FCSLOG.info(name + " TCPIP SERVER STOPPED:");
        } catch (IOException ex) {
            FCSLOG.error(ex);
            FCSLOG.error(name + " : stop method does not work properly when closing socket",ex);
        }

        try {
            clientContext.reader.close();
            clientContext.writer.close();
            clientContext.socket.close();

        } catch (IOException ex) {
            FCSLOG.error(ex);
            FCSLOG.error(name + " : stop method does not work properly", ex);
        }

    }

    public synchronized boolean isReady() {
        if (clientName == null) {
            throw new IllegalArgumentException("Client name can't be null.");
        }
        if (clientContext == null || clientContext.clientName == null) {
            return false;
        }
        if (stopped) {
            return false;
        }
        if (!tcpServerStarted) {
            return false;
        }
        FCSLOG.debug(name + "client name:= " + clientContext.clientName + "#");
        return clientContext.clientName.equals(clientName);
    }
    
    @Override
    public void init(int master, String baud, String busName, int nodeID) {
        // No-op. Done by the CWrapper
    }

    @Override
    public PDOData sync() throws DriverException {
        String reply = (String)call("sync");
        PDOData pdoD = CWrapperUtils.createPDOData(reply);
        return pdoD;
    }

    @Override
    public int scan() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public String info(int nodeID) throws DriverException {
        return (String)call("info,"+Integer.toHexString(nodeID));
    }

    @Override
    public void wsdo(int nodeId, int index, int subindex, int size, long data) throws DriverException {
        
        String request = CWrapperUtils.buildWsdoCommand(nodeId, index, subindex, size, data);
        
        String sdoResponseLine = (String)call(request);
        
        /*response of a writeSDO command is : wsdo,nodeID,errorCode*/
        String[] words = sdoResponseLine.split(",");

        int errorCode = Integer.parseInt(words[2], 16);
        if (0 != errorCode) {
            throw new SDOException(String.valueOf(errorCode), "request : " + request + " returned " + sdoResponseLine);
        }
    }

    @Override
    public long rsdo(int nodeId, int index, int subindex) throws DriverException {
        
        String request = CWrapperUtils.buildRsdoCommand(nodeId, index, subindex);
        
        String sdoLine = (String) call(CWrapperUtils.buildRsdoCommand(nodeId, index, subindex));
        
        String[] words = sdoLine.split(",");
        int responseLength = words.length;
        /* response of a readSDO command should be : rsdo,nodeID,errorCode,data */
        int errorCode = Integer.parseInt(words[2], 16);
        if (errorCode == 0 && (responseLength > 3)) {
            return Long.parseLong(words[3], 16);
        } else {
            throw new SDOException(String.valueOf(errorCode), "request : " + request + " returned " + sdoLine);
        }
    }
    
    @Override
    public void ssta(int nodeId) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public void ssto(int nodeId) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public void reset(int nodeId) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public void quit() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     * This methods send a command to the tcp client. The command is a String
     * and can be understood by the client. Example for a Can Open command :
     * rsdo,1,1018,0 The command is stored in the command dispenser with a token
     * and command corelation. When the response comes back from the proxy we
     * can retrieve it from the dispenser with the call token.
     *
     * @param command
     * @return result of the call or null if the call failed
     */
    private Object call(String command) throws DriverException {

        if (clientContext == null) {
            throw new DriverException("CWrapper not connected. clientName : "
                    + clientName +", port : " + portNumber);
        }
        if (stopped) {
            throw new IllegalStateException(name + ":textTCP module stopped");
        }

        try {

            String sentCommand = command + "\r\0\n";
            COMMAND_LOG.finest(name + ":Sent command=" + command);
            clientContext.writer.write(sentCommand, 0, sentCommand.length());
            clientContext.writer.flush();

            //if we want to wait for the response we have to store this command.
            if (command.startsWith("rsdo")
                    || command.startsWith("wsdo")
                    || command.startsWith("info")
                    || command.startsWith("sync")
                    || command.startsWith("srtr")) {
                String commandToken = commandDispenser.register(command);
                try {
                    return commandDispenser.getCommandResponse(commandToken, fieldBusTimeout);
                } finally {
                    commandDispenser.remove(commandToken);
                }
            } else {
                return "asynchronous command sent:" + command;
            }

        } catch (IOException ex) {
            FCSLOG.error(name + ":ERROR in tcpProxy call method");
            FCSLOG.error(ex);
            //TODO change this : we shouldn't use a WrappedException
            throw new WrappedException(name + ":ERROR in tcpProxy call method " + clientName, ex);
        }
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(name);
        sb.append("/port=");
        sb.append(this.portNumber);
        return sb.toString();
    }

}
