
package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.drivers.canopenjni.CanOpenInterface;
import org.lsst.ccs.drivers.canopenjni.PDOData;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 * A model for the CANopen device to read carousel sensors.
 * This sensors are read by PDO. (in response to a command "sync" sent to the CANbus)
 * 
 * @author virieux
 */
public class CanOpenTTC580  extends CanOpenDevice {
    
    @LookupField(strategy=Strategy.ANCESTORS)
    private CanOpenInterface canOpen;
    
    /**
     * pdoVal is coded on 64bits and represents from right to left :
     * - 3 bits for socket indentification
     * - 3 bits unused
     * - 2 bits unused
     * - 4 bits for State of filter at standby for clampXminus
     * - 4 bits for State of filter at standby for clampXplus
     * - 12 bits for ClampXminus lockSensor value
     * - 12 bits for ClampXminus filterPresenceSensor value
     * - 12 bits for ClampXplus lockSensor value
     * - 12 bits for ClampXplus filterPresenceSensor value
     * 
     */
    long pdoVal = 0L;

    public CanOpenTTC580(int nodeID, String serialNB) {
        super(nodeID, serialNB);
    }


    /**
     * 
     * @return 
     */
    @Command(type=Command.CommandType.QUERY, description="return PDO1 value")
    public long getPdoVal() {
        return pdoVal;
    }

    /**
     * 
     * @param pdoVal 
     */
    public void setPdoVal(long pdoVal) {
        this.pdoVal = pdoVal;
    }
    
    
    /**
     * send a "sync" command to CANbus, process response and update pdoVal.
     * a reply to a sync command should be :
     * PDO1=xxx, PDO2=xxxx, PDO3=xxxxx etc...
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    @Command(type=Command.CommandType.QUERY, description="send a sync command to CANbus, "
            + "process response and update pdoVal.")
    public void updatePDO() throws DriverException {
         // TODO : The cobid must be configured using CanOpenInterface#addReceivedPDO
        // For the sync to work
        PDOData pdo = canOpen.sync();
        
        // TODO : This will not work as is.
        pdoVal = pdo.getPDOs().get(1);
    }
    
    /**
     * check replyToSync syntaxe and extract the value of PDO1
     * @param replyToSync
     * @return PDO1 value
     * @deprecated do not use, this is CWrapper protocol dependent
     */
    @Deprecated
    protected long extractFirstPDO(String replyToSync) {
        String[] words = replyToSync.split(",");
        if (words[1].startsWith("PDO1") && (words[0].split("=")).length > 0) {
            String pdoValStr = (words[1].split("="))[1];
           return Long.valueOf(pdoValStr,16);
        } else {
            throw new IllegalArgumentException(name + " " + replyToSync + ": bad response to sync command. ");
        }
    }
    
    /**
     * return socketId from PDO value
     * socket identification are the first 3 bits (reading from left to right).
     * @return 
     */
    @Command(type=Command.CommandType.QUERY, description="return socketId from PDO value")
    public short getSocketId() {
        /* 61 = 64 - nbBits (nbBits =3)*/
        return (short) ((pdoVal >>> 61));
    }
    
    /**
     * return lock sensor for clampXminus
     * @return 
     */
    @Command(type=Command.CommandType.QUERY, description="return lock sensor for clampXminus")
    public int getLockXm() {
        return (int) (pdoVal >>> 36) & 0xFFF;
    }
    
    /**
     * return filter presence sensor for clampXminus
     * @return 
     */
    @Command(type=Command.CommandType.QUERY, description="return filter presence sensor for clampXminus")
    public int getFilterPresenceXm() {
        return (int) (pdoVal >>> 24) & 0xFFF;
    }
    
    /**
     * return lock sensor for clampXplus
     * @return 
     */
    @Command(type=Command.CommandType.QUERY, description="return lock sensor for clampXplus")
    public int getLockXp() {
        return (int) (pdoVal >>> 12) & 0xFFF;
    }    
    
    /**
     * return filter presence sensor value for clampXplus.
     * this value is coded on the last 12 bits (reading from left to right).
     * @return 
     */
    @Command(type=Command.CommandType.QUERY, description="return filter presence sensor for clampXplus")
    public int getFilterPresenceXp() {
        return (int) pdoVal & 0xFFF;
    }
}
