
package org.lsst.ccs.subsystems.fcs.drivers;

import java.util.Map;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.errors.EPOSConfigurationException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.PROFILE_POSITION;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.CAN_BUS_TIMEOUT;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.EMCY;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.PARAMETER_ERROR;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenCallTimeoutException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * A general model for a controller EPOS.
 *
 * @author virieux 
 */
public class CanOpenEPOS extends CanOpenDevice implements EPOSController {
    

    /**
     * used in checkParameters and initializeAndCheckHardware methods
     */
    protected boolean parametersOK = true;

    /* Mode of Operation */
    protected EposMode mode;

    /* Parameters */
    @ConfigurationParameter
    private Map<String, Integer> paramsForHoming;
    
    @ConfigurationParameter
    private Map<String, Integer> paramsForProfilePosition;
    
    @ConfigurationParameter
    private Map<String, Integer> paramsForCurrent;

    /**
     * Used to publish on the status bus for the GUI*
     */
    protected boolean enabledToPublish = false;
    
    /**
     * Construction of a CanOpenEPOS.
     * @param nodeID
     * @param serialNB
     * @param paramsForCurrent
     * @param paramsForProfilePosition
     * @param paramsForHoming 
     */
    public CanOpenEPOS(
            int nodeID, String serialNB,
            Map<String, Integer> paramsForCurrent,
            Map<String, Integer> paramsForProfilePosition,
            Map<String, Integer> paramsForHoming) {
        super(nodeID, serialNB);
        this.paramsForCurrent = paramsForCurrent;
        this.paramsForProfilePosition = paramsForProfilePosition;
        this.paramsForHoming = paramsForHoming;
    }

    @Override
    public boolean isEnabledToPublish() {
        return enabledToPublish;
    }

    @Override
    public void setEnabledToPublish(boolean enabledToPublish) {
        this.enabledToPublish = enabledToPublish;
    }
    
    /**
     * 
     * @return true if parameters in the CPU of the controllers
     * have the same values than in configuration system.
     */
    @Override
    public boolean isParametersOK() {
        return parametersOK;
    }
    
    /**
     * Return Epos mode for GUI.
     * @return 
     */
    @Override
    public EposMode getMode() {
        return mode;
    }

    /**
     * Reads the EPOS mode in the CPU of the EPOS device, updates the field mode
     * and returns the mode as a String.
     *
     * @return
     * @throws SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Reads the EPOS mode in the CPU of the EPOS device and "
            + "updates the field mode.")
    @Override
    public EposMode readMode()  {
        try {
            int m = readSDO(0x6061, 0);
            EposMode readMode = EposMode.getMode(m);
            this.mode = readMode;
            return readMode;
        } catch (CanOpenCallTimeoutException ex) {
            String msg = getName() + ":" + CAN_BUS_TIMEOUT.getLongDescription()
                    + " to command: readMode - POWER FAILURE ? ";
            FCSLOG.warning(ex);
                    this.raiseWarning(name + ":" + CAN_BUS_TIMEOUT,CAN_BUS_TIMEOUT.getLongDescription(),
                    msg + ex);
            return mode;
        }
    }

    /**
     * This methods changes the mode to the new mode given as an argument. It
     * writes the new mode in the CPU of the EPOS controller and updates the
     * field mode.
     *
     * @param newMode
     * @throws SDORequestException
     */
    @Override
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Changes EPOS mode in the CPU of the EPOS device and "
            + "updates the field mode.")
    public void changeMode(EposMode newMode)  {
        //TODO check if the mode is already newMode don't change anything.
        this.writeSDO(0x6060, 0, 1, newMode.getMode());
        this.mode = newMode;
        publishData();
    }
    
    /**
     * Changes an EPOS parameter for this controller. Writes the new value on the CPU of the controller.
     * Changes the new value in the configuration.
     * But doesn't save the new parameter in the CPU. To save it, use command saveAllChanges.
     * 
     * @param key
     * @param value
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, 
            description="Changes an EPOS parameter for this controller. "
                    + "Writes the new value on the CPU of the controller."
            + "Changes the new value in the configuration. "
            + "But doesn't save the new parameter in the CPU. To save in CPU controller, "
                    + "use command saveParameters and saveAllChanges."
            + "To save configuration, use command saveAllChanges.")    
    public void changeEPOSParameter(String key, int value) {
        Parameter parameter = Parameter.valueOf(key);
        writeParameter(parameter,value);
        EposMode eposMode = EposMode.getModeForParameter(parameter);
        Map<String, Integer> paramsToChange;
        String paramsToChangeName;
        if (EposMode.CURRENT.equals(eposMode)) {
            paramsToChange = paramsForCurrent;
            paramsToChangeName = "paramsForCurrent";
        } else if (EposMode.PROFILE_POSITION.equals(eposMode)) {
            paramsToChange = paramsForProfilePosition;
            paramsToChangeName = "paramsForProfilePosition";
        } else if (EposMode.HOMING.equals(eposMode)) {
            paramsToChange = paramsForHoming;
            paramsToChangeName = "paramsForHoming";           
        } else throw new IllegalArgumentException(key+ " no parameter list is defined for mode: " + eposMode);
        paramsToChange.put(key, value);
        s.getConfigurationService().getComponentConfigurationEnvironment(name).change(paramsToChangeName,paramsToChange);
        FCSLOG.info(name+ ":changed EPOS parameter " + key + "for mode "+ eposMode);
    }        

    /**
     * This methods writes in the CPU of the EPOS devis the values of the
     * parameters set for the mode. Usually the values of the parameters are
     * given by the configuration system.
     *
     * @param mode
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "This methods writes in the CPU of the EPOS device the values "
            + "of the parameters set for the mode given as argument")
    @Override
    public void writeParameters(EposMode mode)  {
        String modeIS = mode.toString();
        Map<String, Integer> paramsMap = null;
        switch (mode) {
            case HOMING:
                paramsMap = this.paramsForHoming;
                break;
                
            case PROFILE_POSITION:
                paramsMap = this.paramsForProfilePosition;
                break;

            case CURRENT:
                paramsMap = this.paramsForCurrent;
                break;
            
            default:
                assert false : mode;
        }
        if (paramsMap == null) {
            throw new EPOSConfigurationException(name
                    + "parameters for mode :" + modeIS + "are NULL");
        }
        if (paramsMap.isEmpty()) {
            throw new EPOSConfigurationException(name
                    + "parameters for mode :" + modeIS + "are not defined.");
        }
        writeParameters(paramsMap);
    }

    /**
     * This method writes a parameter in the CPU of the EPOS.
     * For end user at the console
     * @param param
     * @param value in decimal
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Set EPOS parameter with the value given as argument.")
    @Override
    public void writeParameter(Parameter param, int value) {
        writeSDO(param.getIndex(), param.getSubindex(), param.getSize(), value);
    }
    
    /**
     * Reads in the EPOS CPU the value of the Parameter.
     *
     * @param param
     * @return value of the parameter in decimal format.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Override
    public int readParameter(Parameter param)  {
        try {
            int value = tcpProxy.readSDOLong(nodeID, param.getIndex(), param.getSubindex());
            FCSLOG.debug(name + " readParameter:" + param.name() + "=" + value
                    + " in decimal format  (valueInHexa=" + Integer.toHexString(value) + ")");
            return value;
        } catch (SDORequestException | ShortResponseToSDORequestException ex) {
            //TODO don't catch Exception here but dans la classe appelante. DAnger !!! 
            String msg = getName() + " Error in reading Parameter: " + param;
            this.raiseWarning(PARAMETER_ERROR, msg, name, ex);
        } 
        return 0;
    }

    /**
     * Reads in the EPOS CPU the hexadecimal value of the Parameter.
     *
     * @param parameterName
     * @return value of the parameter in hexa format.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Reads in the EPOS CPU the hexadecimal value of the Parameter "
            + "which parameter name is given as argument.")
    protected String readParameterInHexa(String parameterName)  {
        return Integer.toHexString(readParameterInHexa(Parameter.valueOf(parameterName)));
    }

    /**
     * Reads in the EPOS CPU the value of the Parameter.
     *
     * @param param
     * @return value of the parameter in hexa format.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    protected int readParameterInHexa(Parameter param)  {
        return readSDO(param.getIndex(), param.getSubindex());
    }

    /**
     * Check if the parameters in the cPU of the controller have the right values.
     * And if the controller is not in fault.
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read controller's parameter in the CPU for each mode and "
            + "warn if a parameter has not the same value as in the configuration."
            + "Then publish data.")
    @Override
    public void initializeAndCheckHardware()  {
        checkBooted();
        this.mode = readMode();
        this.parametersOK = true;
        checkParameters(EposMode.CURRENT);
        checkParameters(EposMode.PROFILE_POSITION);
        checkParameters(EposMode.HOMING);
        checkFault();
        this.initialized = true;
        if (!parametersOK) {
            FCSLOG.warning(name + " Some parameter values are not"
                    + " the same in CPU and configuration system.");
        }
        //publishData for the GUI
        this.publishData();
        FCSLOG.info(name + ": is INITIALIZED.");
    }

    /**
     * Restore default parameters.
     *
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Restore default parameters. AT YOUR OWN RISK.")
    public void restoreParameters()  {
        //TODO if is Power Disabled
        this.writeSDO(0x1011, 1, 0x4, 0x64616F6C);
    }

    /**
     * Write a value in hexa in the control world (index=6040,
     * subindex=0,size=2)
     *
     * @param value in hexa
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Write a value in hexa in the control world (index=6040, subindex=0,size=2)")
    @Override
    public void writeControlWord(int value)  {
        this.writeSDO(0x6040, 0, 0x2, value);
    }

    /**
     * Read the control world (index=6040, subindex=0,size=2)
     *
     * @return value in hexa
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read the control world (index=6040, subindex=0,size=2)")
    public String readControlWord()  {
        return Integer.toHexString(this.readSDO(0x6040, 0));
    }

    /**
     * Read the status world (index=6041, subindex=0,size=2)
     *
     * @return value in hexa
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read the status world (index=6041, subindex=0,size=2)")
    @Override
    public String readStatusWord()  {
        return Integer.toHexString(this.readSDO(0x6041, 0));
    }


    /*Methods available in CURRENT mode*/
    /**
     * In unsCurrent mode this methods send a unsCurrent to the motor. This make run
     * the motor.
     *
     * @param aValue UNIT=mA / FORMAT=decimal the value of the unsCurrent to be
 sent.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Sent current to controller the EPOS CPU. UNIT=mA / FORMAT=decimal")    
    @Override
    public void writeCurrent(int aValue)  {
        if (!EposMode.CURRENT.equals(mode)) {
            throw new RejectedCommandException(name + " is not in CURRENT mode.");
        }
        writeSDO(0x2030, 0, 2, aValue);
        FCSLOG.info(name + ": sent current to controller=" + aValue);
    }

    /**
     * In Current Mode this methods returns the unsCurrent actualy received by the motor.
     *
     * @return the unsCurrent UNIT=mA / FORMAT=decimal
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, alias = "readCurrentActualValue",
            description = "In Current Mode this methods returns the current actualy received by the motor.")
    public int readCurrent()  {
        int unsCurrent = readSDO(0x6078, 0);
        int current = FcsUtils.convertToInteger16(unsCurrent);
        /* Actual Current (Object 6078) is an INTEGER16*/
        FCSLOG.debug(name + ":readCurrent=" + current + " Unsigned value read=" + unsCurrent);
        return current;
    }

    /**
     * In Current Mode this methods returns the average of the unsCurrent received
 by the motor.
     *
     * @return the unsCurrent UNIT=mA / FORMAT=decimal
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, alias = "readCurrentAverage",
            description = "In Current Mode this methods returns the average of the current  received by the motor.")
    public int readCurrentAverageValue()  {

        int current = readSDO(0x2027, 0);
        FCSLOG.debug(name + ":readAverageCurrent=" + FcsUtils.convertToInteger16(current) + " Unsigned value read=" + current);
        return FcsUtils.convertToInteger16(current);
    }    
    
    /**
     * return motor velocity read on controller CPU
     * Unit = RPM
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "return motor velocity read on controller CPU. Unit:rpm")
    @Override
    public int readVelocity() {
        /*velocity is an INTEGER32*/
        long unsVelocity = tcpProxy.readSDOLong(this.nodeID, 0x2028, 0);
        int velocity = FcsUtils.convertToInteger32(unsVelocity);
        FCSLOG.debug(name + " Velocity actual value averaged:" + velocity + " Unsigned value:" + unsVelocity);
        return velocity;
    }

    /**
     * In PROFILE_POSITION mode this methods set the target position. This make
     * run the motor.
     *
     * @param aValue UNIT=mA / FORMAT=decimal the value of the unsCurrent to be
 sent.
     * @throws RejectedCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "In PROFILE_POSITION mode this methods set the target position."
            + " Target position has to be given in microns.")
    @Override
    public void writeTargetPosition(int aValue)  {
        if (!mode.equals(EposMode.PROFILE_POSITION)) {
            throw new RejectedCommandException(name + " is not in PROFILE_POSITION mode");
        }
        writeSDO(0x607A, 0, 4, aValue);
    }
    
    /**
     * set target velocity in mode PROFILE_VELOCITY
     * @param velocity
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "set target velocity in mode PROFILE_VELOCITY.")    
    @Override
    public void writeTargetVelocity(int velocity) {
        if (!mode.equals(EposMode.PROFILE_VELOCITY)) {
            throw new RejectedCommandException(name + " is not in PROFILE_VELOCITY mode");
        }
        writeSDO(0x60FF, 0, 4, velocity);
    }

    /**
     * Check if the Controller is in fault.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the Controller is in fault.")
    @Override
    public void checkFault()  {
        checkBooted();
        if (readNumberOfErrors() > 0) {
            this.inError = true;
            int errorInHexa = this.readErrorRegister();
            this.errorRegister = CanOpenErrorsTable.getErrorRegisterNameByCode(errorInHexa);
            this.errorHistory = this.readErrorHistory();
            publishData();
            String errorHistoryString = displayErrorHistory();
            throw new FcsHardwareException(name
                    + " is in fault. " + errorHistoryString);
        } else {
            resetError();
        }
    }
    
    /**
     * Clear faults on the controller and clear alarm that was raised for this controller.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Clear faults on the controller.", alias="resetFault")
    public void faultReset()  {
        writeControlWord(0x80);
        resetError();
        this.clearAlarm(EMCY, name);
    } 
    
    /**
     * For every parameter to be defined for this mode this method compares the
     * values in configuration with the value stored in controller CPU and
     * throws a FcsHardwareException if these values are not equal. values stored
     * in the controller CPU.
     *
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "For every parameter to be defined for this mode this method "
            + "compares the values in configuration with the value stored in controller CPU"
            + "and throws a FcsHardwareException if these values are not equal.")
    @Override
    public void checkParameters(EposMode aMode)   {
        Map<String, Integer> paramsMap = null;
        switch (aMode) {
            case HOMING:
                paramsMap = this.paramsForHoming;
                break;
                
            case PROFILE_POSITION:
                paramsMap = this.paramsForProfilePosition;
                break;

            case CURRENT:
                paramsMap = this.paramsForCurrent;
                break;
                
            default:
                assert false;
        }
        if (paramsMap == null) {
            throw new FcsHardwareException(name
                    + "parameters for mode :" + aMode.toString() + "are NULL in configuration system.");
        }
        if (paramsMap.isEmpty()) {
            throw new FcsHardwareException(name
                    + "parameters for mode :" + aMode.toString() + "are not defined in configuration system.");
        }
        checkParameters(paramsMap);
    }

    /**
     * Compare values of parameters in the map with the values of parameters in
     * the controller CPU.
     *
     * @param paramsMap
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     */
    public void checkParameters(Map<String, Integer> paramsMap)  {
        boolean errorFound = false;
        for (Map.Entry<String, Integer> entry : paramsMap.entrySet()) {
            String paramName = entry.getKey();
            int configValue = entry.getValue();
            int controllerValue = 0;
            try {
                controllerValue = readParameter(paramName);
 
            } catch (CanOpenCallTimeoutException ex) {
                String cause = getName() + ":" + CAN_BUS_TIMEOUT.getLongDescription()
                        + " to command: checkParameters - POWER FAILURE ? ";
                this.raiseWarning(name + ":" + CAN_BUS_TIMEOUT,CAN_BUS_TIMEOUT.getLongDescription(),cause + ex);
            }

            if (configValue != controllerValue) {
                String cause = paramName + " value found in configuration=" + configValue
                        + " value read in CPU=" + controllerValue + "\n";
                this.raiseWarning(name+":"+PARAMETER_ERROR,PARAMETER_ERROR.getLongDescription(), cause);
                errorFound = true;
                //commented out for tests on autochanger in january 2017
//                CanOpenEPOS.this.writeParam(paramName,configValue);
            }

        }
        parametersOK = !errorFound;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            alias = "refreshGUI",
            description = "Publish data for the controller on the status bus.")
    @Override
    public void publishData() {
        super.publishData();
        s.publishSubsystemDataOnStatusBus(new KeyValueData(name, 
                createStatusDataPublishedByEPOSController()));
    }
}
