package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.canopenjni.CanOpenInterface;
import org.lsst.ccs.drivers.canopenjni.PDOData;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.subsystems.fcs.common.ADCInterface;
import org.lsst.ccs.subsystems.fcs.errors.HardwareNotDetectedException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

/**
 *
 * This is the driver class for the ADC we use in the FCS test bench at APC and in
 * SINGLE FILTER TEST CAN-CBX-AI814.
 *
 * @author virieux
 */
public class CanOpenADC extends CanOpenDevice implements ADCInterface {

    @LookupField(strategy=LookupField.Strategy.ANCESTORS)
    private CanOpenInterface canOpen;
    
    /**
     * One bit read on the input of the ADC represents 0.3125 mV
     */
    public static final double adcVoltsPerBit = 0.3125;
    public static final double waterFreezeTemp = 273.15;

    /**
     * The transmission types for the analogue inputs are described in the
     * following: acyclic, synchronous: The transmission is initiated if a
     * SYNC-message has been received (PDO- transmission type 0) and data has
     * changed. cyclic, synchronous: The transmission is initiated if a defined
     * number of SYNC-messages have been received (PDO-transmission type
     * 1...240). synchronous, remote request: The state of the inputs is latched
     * with each SYNC-message and is transmitted after the reception of a
     * RTR-frame (PDO-transmission type 252). asynchronous, remote request:
     * After the reception of a RTR-frame the last latched state of the inputs
     * is transmitted (PDO-transmission type 253). event controlled,
     * asynchronous: The transmission is initiated if the state of selected
     * inputs has changed (PDO-transmission type 254, 255).
     */
    @ConfigurationParameter
    protected int transmissionType;
    
    long pdoVal = 0L;
    
    
    private int[] inputs = new int[8];
    

    /**
     * Build a CanOpenADC with a CANopen node ID, a serial number and a transmission type.
     * @param nodeID
     * @param serialNB
     * @param transmissionType 
     */
    public CanOpenADC(int nodeID, String serialNB, int transmissionType) {
        super(nodeID, serialNB);
        this.transmissionType = transmissionType;
    }

    /**
     * Read the analog input whom number is given as a parameter
     *
     * @param inputNB
     * @return the value read
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws org.lsst.ccs.subsystems.fcs.errors.HardwareNotDetectedException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Override
    @Command
    public int readAnalogInput(int inputNB)  {

        if (!tcpProxy.isReady()) {
            throw new HardwareNotDetectedException(name + ": hardware not connected.");
        }

        if (isBooted()) {
            if ((inputNB < 1) || (inputNB > 8)) {
                throw new IllegalArgumentException("CAN-CBX-AI814 : input has to be 1..8");
            }
            return tcpProxy.readSDO(nodeID, 0x6401, inputNB);

        } else {
            String message = String.format("Module %s / nodeID=%s is UNREACHABLE",
                    getName(), nodeID);
            FCSLOG.error(message);

            throw new HardwareNotDetectedException("CanOpenADC MISSING",
                    this.getName(), this.nodeID, this.serialNB);
        }
    }

    /**
     * Read the voltage at the ADC input inputNB on the ADC and returns the
     * value in Volt
     *
     * @param inputNB : integer from 1 to 8
     * @return the value read in hexa (on 16bits)
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws org.lsst.ccs.subsystems.fcs.errors.HardwareNotDetectedException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Override
    public double readVoltage(int inputNB)  {
        return readAnalogInput(inputNB) * CanOpenADC.adcVoltsPerBit;
    }

    @Override
    @Command
    public void initializeAndCheckHardware()  {
        this.writeTransmissionTypeToDevice();
        this.initialized = true;
    }

    /**
     * This methods writes the transmission type given by the configuration to
     * the device.
     *
     * @return a message
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command
    protected String writeTransmissionTypeToDevice()  {
        this.tcpProxy.writeSDO(this.nodeID, 0x1801, 0x2, 0x1, this.transmissionType);


        this.tcpProxy.writeSDO(this.nodeID, 0x1802, 0x2, 0x1, this.transmissionType);

        return String.format("Transmission type of device %s is now %s", getName(), this.transmissionType);
    }
    
    protected void updateWithPDO(int cob_id, long pdoVal) {
        int raw_cobId = cob_id - nodeID;
        if(raw_cobId == 0x280) {
            inputs[3]= (int)(pdoVal >> 48);
            inputs[2]=(int)(pdoVal >> 32 & 0xFFFF);
            inputs[1]=(int)(pdoVal >> 16 & 0xFFFF);
            inputs[0]=(int)(pdoVal & 0xFFFF);
        } else if (raw_cobId == 0x380) {
            inputs[7]=(int)(pdoVal >> 48);
            inputs[6]=(int)(pdoVal >> 32 & 0xFFFF);
            inputs[5]=(int)(pdoVal >> 16 & 0xFFFF);
            inputs[4]=(int)(pdoVal & 0xFFFF);
        } else {
            FCSLOG.warning("wrong cobid received : " + raw_cobId);
        }
    }
    
    /**
     * send a "sync" command to CANbus, process response and update pdoVal.
     * a reply to a sync command should be :
     * PDO1=xxx, PDO2=xxxx, PDO3=xxxxx etc...
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    @Command(type=Command.CommandType.QUERY, description="send a sync command to CANbus, "
            + "process response and update pdoVal.")
    public void updatePDO() throws DriverException {
           // TODO : The cobid must be configured using CanOpenInterface#addReceivedPDO
        // For the sync to work
        PDOData pdo = canOpen.sync();
        
        // TODO : This will not work as is.
        pdoVal = pdo.getPDOs().get(2);
        
    }
}
