
package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupName;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;

/**
 * This is the model for a single hook in the loader. 
 * Each one is checked in CLOSED and OPENED position by 2 sensors.
 * These sensors have to give a consistent response otherwise the hook is in ERROR. 
 * There is 4 hooks in the loader to hold a filter.
 *
 * @author virieux
 */
public class LoaderHook {
    
    @LookupName
    private String name;

    private final DigitalSensor closeSensor;
    private final DigitalSensor openSensor;
    
    private FcsEnumerations.LockStatus lockStatus = LockStatus.UNKNOWN;

    //Used because we have to wait for the update from the sensors to know the state
    //of the hook
    protected final Lock lock = new ReentrantLock();
    private final Condition stateUpdated = lock.newCondition();
    private boolean closed;
    private boolean opened;
    private boolean inError;

    /**
     * This is used when we update the latch state with the values returned by
     * the sensors.
     */
    protected volatile boolean updatingState = false;

    public LoaderHook(DigitalSensor closeSensor, DigitalSensor openSensor) {
        this.closeSensor = closeSensor;
        this.openSensor = openSensor;
    }




    public FcsEnumerations.LockStatus getLockStatus() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(name + ": getLockStatus was interrupted while waiting for update.");
                }

            }
            return lockStatus;

        } finally {
            lock.unlock();
        }
    }

    /**
     * This methods updates lockStatus from the values return by the sensors.
     *
     * @param hexaValues
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Update state in reading sensors.")
    public void updateStateWithSensors(int[] hexaValues) {
        lock.lock();
        try {
            updatingState = true;
            this.closeSensor.updateValue(hexaValues);
            this.openSensor.updateValue(hexaValues);

            closed = this.closeSensor.isOn();
            opened = this.openSensor.isOn();
            inError = closed && opened;

            if (inError) {
                lockStatus = FcsEnumerations.LockStatus.ERROR;
            } else if (closed) {
                lockStatus = FcsEnumerations.LockStatus.CLOSED;
            } else if (opened) {
                lockStatus = FcsEnumerations.LockStatus.OPENED;
            } else {
                lockStatus = FcsEnumerations.LockStatus.INTRAVEL;
            }

        } finally {

            updatingState = false;
            stateUpdated.signal();
            lock.unlock();
        }

    }

    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "display sensors info")
    public String toString() {
        StringBuilder sb = new StringBuilder(name);
        sb.append("#closeSensor=");
        sb.append(this.closeSensor.toString());
        sb.append("#openSensor=");
        sb.append(this.openSensor.toString());
        return sb.toString();
    }
    
    /**
     * To create an object StatusDataPublishedByLoaderHook for publication on the 
     * STATUS bus.
     * @return 
     */
    StatusDataPublishedByLoaderHook createStatusDataPublishedByLoaderHook() {
        return new StatusDataPublishedByLoaderHook(name,
                closeSensor.isOn(),
                openSensor.isOn(),
                inError,
                lockStatus
        );
    }

}
