
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupName;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;

/**
 * Represents a model for a force sensor.
 * A force sensor is used in loader subsystem to check if the clamp is CLAMPED or not.
 * It's connected to the gateway.
 * The value returned by the force sensor is coded on 1 byte.
 * The analog value is used mainly to be displayed on the GUI.
 * The gateway gives also a status to say if the clamp is CLAMPED or not.
 * 
 * @author virieux
 */
public class ForceSensor {
    
    @LookupName
    private String name;
    /**
     * transfert function coefficient 
     */
    @ConfigurationParameter
    public double transfert_coeff = 0.4932;
    
    /**
     * transfert function constante
     */
    @ConfigurationParameter
    public double transfert_Constante = 467.99;
    
    /**
     * voltage value read on the gateway.
     * format = decimal
     * unit = mV (millivolts)
     */
    private int voltage; 
    
    /**
     * force value obtained from voltage through a transfert function.
     * format = decimal
     * unit = N (newton)
     */
    private double force;
    
    /**
     * byte number on the gateway
     */
    @ConfigurationParameter
    private final int byteNumero;
    
    /**
     * min value for range where hooks are clamped
     * over this value and below maxRangeValue, hooks are clamped.
     * below this value, hooks are not clamped
     */
    @ConfigurationParameter(description="Min value for range where hooks are clamped: "
            + "over this value and below maxRangeValue, hooks are clamped.  Unit = N")
    private int minRangeValue = 778;
    
    /**
     * max value for range where hooks are clamped
     * under this value and over minRangeValue, hooks are clamped
     * above this value hooks are not clamped
     * 
     */
    @ConfigurationParameter(description="Max value for range where hooks are clamped:"
            + "below this value and over minRangeValue, hooks are clamped.  Unit = N")
    private int maxRangeValue = 920; 
    
    @ConfigurationParameter(description="Below this limit, there is no load on the clamp  so "
            + "clamp is really unclamped."
            + "Over this limit and below minRangeValue, clamp is UNDER_LOAD, neither clamped, "
            + "neither unclamped.  Unit = N")
    private int underLoadLimit = 30;
    
    /**
     * computed from force sensor value.
     * - UNCLAMPED : value is lower that underLoadLimit.
     * - UNDER_LOAD : value is between underLoadLimit and minRangeValue
     *      filter is not clamped enough so carrier can't be move 
     *      but too much clamped so autochanger can't close its latches.
     * - CLAMPED : value is between minRangeValue and maxRangeValue
     * - ERROR : value is bigger than maxRangeValue
     */
    private LockStatus forceStatus;


    /**
     * Builds a ForceSensor with a byte number.
     * @param byteNumero
     */
    public ForceSensor(int byteNumero) {
        this.byteNumero = byteNumero;
    }

    /**
     * Returns voltage value read on force sensor in mV.
     * @return 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
        description = "Returns force sensor analog Value. Unit= mV") 
    public int getVoltage() {
        return voltage;
    }

    /**
     * returns force value read on force senor in Newton. 
     * @return force
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
        description = "Returns force sensor value in Newton. Unit = N")    
    public double getForce() {
        return force;
    }

    /**
     * return force Status
     * @return 
     */
    public LockStatus getForceStatus() {
        return forceStatus;
    }

    /**
     * return byteNumero
     * used by simulation
     * @return 
     */
    public int getByteNumero() {
        return byteNumero;
    }

    /**
     * 
     * @return transfert_coeff
     * used by simulation
     */
    public double getTransfert_coeff() {
        return transfert_coeff;
    }

    /**
     * 
     * @return transfert_Constante
     * used by simulation
     */
    public double getTransfert_Constante() {
        return transfert_Constante;
    }

    /**
     * return high limit for force sensor value
     * @return 
     */
    public int getMaxRangeValue() {
        return maxRangeValue;
    }
    
    
       
    
    /**
     * updates voltage of force sensor from an array of hexa values.
     * @param hexaValues 
     */
    public void updateForce(int[] hexaValues) {
        FCSLOG.finest(name + " updating voltage and force value from byte " 
                + byteNumero );
        /* voltage sent by sensor is given in 0.1V so it has to be multiplied by 100
        to be expressed in mV*/
        voltage = 100 * hexaValues[byteNumero];
        force = transfertFunction(voltage);
        computeForceStatus();
    }
    
    /**
     * update forceStatus from field force
     */
    private void computeForceStatus() {
        if (force < underLoadLimit) {
            forceStatus = LockStatus.UNCLAMPED;
            
        } else if (force < minRangeValue) {
            forceStatus = LockStatus.UNDER_LOAD;
            
        } else if (force < maxRangeValue) {
            forceStatus = LockStatus.CLAMPED;
            
        } else {
            forceStatus = LockStatus.OVER_LOAD;
        }      
    }
    
    /**
     * Computes force value from voltage given as argument.
     * As the beginning the function is not linear.
     * When closing the loader clamp, at beginning of closing, a  small force begins 
     * to be applied on the dynamometric axis but no force is applied on the filter. 
     * @param tension
     * @return 
     */
    public double transfertFunction(int tension) {
        double transfertValue = transfert_coeff * tension - transfert_Constante;
        if (transfertValue > 0) {
            return transfertValue;
        } else {
            return 0;
        } 
    }
    
    /**
     * 
     * @return true if force sensor value is in range values where clamp is clamped.
     * If true, hooks are CLAMPED.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
        description = "Returns true if force sensor value is in range values where clamp is clamped.")
    public boolean isClamped() {
        return force > this.minRangeValue && force < this.maxRangeValue;
    }

}
