
package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.commons.annotations.LookupName;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.BinarySensor;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.PlutoGatewayInterface;

/**
 * This class is a model for the fakeAutochanger that we use on loader and carousel test bench 
 * in standalone mode.
 * 
 * As there is no autochanger, a numeric sensor is setup to simulate that the autochanger holds the filter
 * at STANDBY or HANDOFF position.
 * It's also a model for the fakeCarousel and the fakeLoader in autochanger-standalone subsystem.
 * 
 * FakeFilterHolder implements interface FilterHolder.
 * In the whole FCS, carousel, autochanger and loader implements FilterHolder too.
 * 
 * @author virieux
 */
public class FakeFilterHolder implements FilterHolder {
    
    @LookupField(strategy=Strategy.TOP)
    private Subsystem s;
    private final PlutoGatewayInterface plutoGateway;
        
    private final BinarySensor holdingFilterSensor;
    
    @LookupName
    private String name;
    
    /**
     * holdingFilter is updated by method updateStateWithSensors
     */
    private boolean holdingFilter = false;
    
    //Used because we have to wait for the update from the sensors
    private final Lock lock = new ReentrantLock();
    private final Condition stateUpdated = lock.newCondition();
    private volatile boolean updatingState = false;

    /**
     * Build a FakeFilterHolder with a pluto gateway name and a couple of digital sensors.
     * Used in 
     * @param plutoGateway 
     * @param holdingFilterSensor
     */
    public FakeFilterHolder(PlutoGatewayInterface plutoGateway,
            BinarySensor holdingFilterSensor) {
        this.plutoGateway = plutoGateway;
        this.holdingFilterSensor = holdingFilterSensor;
    }

    /**
     * Return true if the fake filter holder is holding the filter.
     * This command doesn't read again the sensor.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the fake autochanger is holding the filter. "
                    + "This command doesn't read again the sensor.")
    @Override
    public boolean isHoldingFilter() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(name + ": has been interrupted while waiting for end of update.");
                }

            }
            return holdingFilter;

        } finally {
            lock.unlock();
        }
    }
    
    
    /**
     * This methods updates the field holdingFilter in reading the holding filter sensor.
     * Then it published status data on the STATUS bus.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, 
            description = "Update clamp state in reading sensors.")
    @Override
    public void updateStateWithSensors()  {
        
        plutoGateway.checkBooted();
        plutoGateway.checkInitialized();
        
        lock.lock();
        try {
            updatingState = true;
            this.plutoGateway.updateValues();
            computeState(this.plutoGateway.getHexaValues());
        } finally {

            updatingState = false;
            stateUpdated.signal();
            lock.unlock();
            this.publishData();
        }
    }
    
    private void computeState(int[] hexaValues) {
            this.holdingFilterSensor.updateValue(hexaValues);
            this.holdingFilter = this.holdingFilterSensor.isOn();
    }
    
    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    public void publishData() {
        KeyValueData kvd;
            kvd = new KeyValueData(name, holdingFilterSensor.isOn());
        s.publishSubsystemDataOnStatusBus(kvd);
    }

    /**
     * For the fake autochanger there is no autochanger trucks, so the position has not a real meaning.
     * The autochanger is supposed to be at HANDOFF.
     * Idem for fakeCarousel and fakeLoader.
     * @return 
     */
    @Override
    public boolean isAtHandoff() {
        return true;
    }

    /**
     * For the fake autochanger there is no autochanger trucks, so the position has not a real meaning.
     * The autochanger is supposed to be at STANDBY.
     * Idem for fakeCarousel and fakeLoader.
     * @return 
     */
    @Override
    public boolean isAtStandby() {
        return true;
    }

}
