package org.lsst.ccs.subsystems.fcs;

import java.util.Map;
import java.util.TreeMap;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.PROFILE_POSITION;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NO_FILTER;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;
import org.lsst.ccs.subsystems.fcs.common.PDOStorage;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerForCarousel;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This is a representation of the hardware of the carousel. It receives
 * commands from the FCSMAinModule and send back an acknowledge. It publishes
 * data on the status bus. In engineering mode it can receive commands from the
 * engineering console.
 *
 *
 * @author virieux
 *
 */
public class Carousel extends MobileItem implements HardwareController, FilterHolder {

    private static final long serialVersionUID = -2376279469784152348L;

    @ConfigurationParameter(range = "0..500000")
    private int rotationTimeout = 20000;

    private int position = 0;
    private int relativeTargetPosition;

    private final int speed = 0;
    private int current = 0;

    /*
     0.1 tour = 436224 pas (2Pi = 4260 X 1024)
     1 socket = 0.2 tour = 1/5 tour = 872448
     */
    private final int stepNB = 4362240;
    private final int halfTurn = stepNB / 2;

    /* si encoderSign = -1, l'encodeur va de 0 a -stepNB*/
    /* si encoderSign = 1, l'encodeur va de 0 a stepNB*/
    @ConfigurationParameter(description = "if encoderSign = 1, encoder goes from 0 to stepNB,"
            + "if encoderSign = -1, encoder goes from 0 to stepNB")
    public int encoderSign = -1;

    /*This actuator opens the clamp Xminus when the carousel is halted at standby filterPosition.*/
    @LookupField(strategy = Strategy.BYNAME)
    private EPOSController clampXminusController;

    /*This actuator opens the clamp Xplus when the carousel is halted at standby filterPosition.*/
    @LookupField(strategy = Strategy.BYNAME)
    private EPOSController clampXplusController;

    /*Controls carousel rotation.*/
    @LookupField(strategy = Strategy.BYNAME)
    private EPOSControllerForCarousel carouselController;

    /*CANOpen devices to read the values of the clamps sensors.*/
    @LookupField(strategy = Strategy.BYNAME)
    private PieceOfHardware hyttc580;

    /*CANOpen devices to read the values of the brakes sensors and temperatures.*/
    @LookupField(strategy = Strategy.BYNAME)
    protected PieceOfHardware ai814;

    @LookupField(strategy = Strategy.BYNAME)
    private PieceOfHardware pt100;

    /*To be able to know if the autochanger holds a filter. */
    @LookupField(strategy = Strategy.BYNAME)
    private FilterHolder autochanger;

    @LookupField(strategy = Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;

    /**
     * A map to store the sockets by their names. The key of this map is the
     * socket name.
     *
     */
    @LookupField(strategy = Strategy.CHILDREN)
    protected final Map<String, CarouselSocket> socketsMap = new TreeMap<>();

    @LookupField(strategy = Strategy.BYNAME)
    private BridgeToHardware tcpProxy;

    private boolean initialized = false;
    private boolean homingDone = false;

    /**
     * *** lifecycle methods *************************************************
     */
    @Override
    public void init() {
//        /**
//         * What has to be done for each tick of the timer. We have to publish on the
//         * status bus to refresh the GUI.
//         */
//        periodicTaskService.scheduleAgentPeriodicTask(
//                new AgentPeriodicTask(name+"-publishData",this::publishData)
//                        .withIsFixedRate(true)
//                        .withLogLevel(Level.WARNING)
//                        .withPeriod(Duration.ofSeconds(5)));
    }

    /**
     * ***********************************************************************
     */
    /**
     *
     * @return true if CANopen devices are booted and initialized and homing has
     * been done.
     */
    @Command(description = "Return true if CANopen devices are booted and initialized and homing has been done.")
    boolean isInitialized() {
        //TODO test also if homing has been done.
        return this.myDevicesReady();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if carouselController homing is done.")
    public boolean isHomingDone() {
        return homingDone;
    }

    public int getStepNB() {
        return stepNB;
    }

    public int getEncoderSign() {
        return encoderSign;
    }

    /**
     * Return a CarouselSocket which name is given as parameter.
     *
     * @param socketName
     * @return
     */
    public CarouselSocket getSocketByName(String socketName) {
        if (socketsMap.containsKey(socketName)) {
            return socketsMap.get(socketName);
        } else {
            throw new IllegalArgumentException(name + ": no such name for socket:" + socketName);
        }
    }

    /**
     * return carousel position.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return carousel position.", alias = "printPosition")
    public int getPosition() {
        return position;
    }

    /**
     * @return the clampXminusController
     */
    public EPOSController getClampXminusController() {
        return clampXminusController;
    }

    /**
     * @return the clampXminusController
     */
    public EPOSController getClampXplusController() {
        return clampXplusController;
    }

    /**
     * This method returns the clampX- which is at standby filterPosition. It
     * can returns null if there is no socketAtStandby halted at standby
     * filterPosition.
     *
     * @return
     */
    public CarouselClamp getClampXminus() {
        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            return null;
        } else {
            return socketAtStandby.getClampXminus();
        }
    }

    /**
     * This method returns the clampX+ which is at standby filterPosition. It
     * can returns null if there is no socketAtStandby halted at standby
     * filterPosition.
     *
     * @return
     */
    public CarouselClamp getClampXplus() {
        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            return null;
        } else {
            return socketAtStandby.getClampXplus();
        }
    }

    /**
     * Used to publish on the STATUS bus for the GUI. Returns
     *
     * @return true if a socket is HALTED at STANDBY filterPosition, false
     * otherwise.
     */
    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if a socket is HALTED at STANDBY position, false otherwise.")
    public synchronized boolean isAtStandby() {
        if (this.isRotating()) {
            return false;
        } else {
            for (CarouselSocket socket : socketsMap.values()) {
                if (socket.isAtStandby()) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Return the socket HALTED at STANDBY filterPosition if there is one.
     * Otherwise return null.
     *
     * @return
     */
    public synchronized CarouselSocket getSocketAtStandby() {
        if (this.isRotating()) {
            return null;
        } else {
            for (CarouselSocket socket : socketsMap.values()) {
                if (socket.isAtStandby()) {
                    return socket;
                }
            }
        }
        return null;
    }

    /**
     * This methods return the filter which is in the socketAtStandby at standby
     * filterPosition. When there is no filter at standby filterPosition it
     * returns null.
     *
     * @return filter at standby filterPosition or null if there is no filter at
     * standby filterPosition.
     */
    public synchronized Filter getFilterAtStandby() {

        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            return null;
        } else {
            return socketAtStandby.getFilter();
        }
    }

    /**
     * Returns name of filter which is in the socket at STANDBY position or
     * NO_FILTER if there is no filter at STANDBY.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Returns name of filter at STANDBY position  or"
            + " NO_FILTER if there is no filter at STANDBY.")
    public synchronized String getFilterAtStandbyName() {

        Filter filterAtStandby = getFilterAtStandby();
        if (filterAtStandby == null) {
            return NO_FILTER;

        } else {
            return filterAtStandby.getName();
        }

    }

    /**
     * This methods returns true if the autochanger is holding a filter at
     * STANDBY position.
     *
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if Autochanger is holding filter at STANDBY.")
    public boolean isAutochangerHoldingFilter() {
        /**
         * To be done when a switch is connected : read the switch sensor and
         * update a boolean of the class like : autochangerHoldingFilter.
         */
        autochanger.updateStateWithSensors();
        return autochanger.isHoldingFilter();
    }
    
    /**
     * Executed when all components HasLifeCycle of subsystem have been checked. 
     */
    @Override
    public void postStart()  {
        FCSLOG.fine(name + " BEGIN postStart.");
        if (carouselController.isBooted()) {
            initializeController();
        }
        if (clampXminusController.isBooted()) {
            initializeClampController(clampXminusController);
        }
        if (clampXplusController.isBooted()) {
            initializeClampController(clampXplusController);           
        }     
        FCSLOG.fine(name + " END postStart.");
    }
    
    private void initializeController() {
        try {
            carouselController.initializeAndCheckHardware();
            this.initialized = true;          
            homing();

        } catch (FcsHardwareException | FailedCommandException ex) {
            this.raiseAlarm(FcsEnumerations.FcsAlert.HARDWARE_ERROR, name + " couldn't initialize controller", 
                    carouselController.getName(), ex);
        }
        if (!this.isHomingDone()) {
            this.raiseAlarm(FcsEnumerations.FcsAlert.HARDWARE_ERROR, name + " couldn't do homing", 
                    carouselController.getName());
        }
    }
    
    
    private void initializeClampController(EPOSController controller) {
        try {
            controller.initializeAndCheckHardware();
            if (controller.readMode() != EPOSEnumerations.EposMode.CURRENT) {
                this.raiseAlarm(FcsEnumerations.FcsAlert.HARDWARE_ERROR,
                        name + "  is not in CURRENT mode.",
                        controller.getName());
            }
        } catch (FcsHardwareException | FailedCommandException ex) {
            this.raiseAlarm(FcsEnumerations.FcsAlert.HARDWARE_ERROR,
                    name + " couldn't initialize controller",
                    controller.getName(), ex);
        }
    }    

    /**
     * CheckStarted is executed by completeInitialization when a first start has
     * failed. Because end user could have changed many things, we have to check
     * again that all is correct.
     *
     */
    @Override
    public void checkStarted() {

        FCSLOG.info(name + " BEGIN checkStarted");

        //check that all hardware is booted and identified.
        tcpProxy.bootProcess();

        postStart();


        FCSLOG.info(name + " END checkStarted");
    }


    /**
     * to be done before each rotation
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "CarouselHoming")
    public void homing() {
        homingDone = false;
        carouselController.disable();
        carouselController.setPositionSensorTypeEncoderSSI();
        int ssiPosition = carouselController.readSSIPosition();
        carouselController.setPositionSensorTypeSinusIncrementalEncoder();
        int newSSIPosition = computeNewPosition(ssiPosition);
        carouselController.enable();
        carouselController.defineAbsolutePosition(newSSIPosition);
        homingDone = true;
        carouselController.changeMode(PROFILE_POSITION);
        carouselController.disable();
        updatePosition();
        publishData();
    }

    private void checkHomingDone() {
        if (!isHomingDone()) {
            throw new FcsHardwareException(name + " homing has to be done before motion.");
        }
    }

    public void checkSensorTypeIncremental() {
        int sensorType = carouselController.readPositionSensorType();
        if (sensorType != 8) {
            throw new FcsHardwareException(name + " PositionSensorType has to be set to Incremental Encoder before motion.");
        }
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     *
     */
    @Override
    public void publishData() {
        s.publishSubsystemDataOnStatusBus(new KeyValueData("carousel",
                createStatusDataPublishedByCarousel()));
        for (CarouselSocket socket : socketsMap.values()) {
            socket.publishData();
        }
    }

    /**
     * Create an object StatusDataPublishedByCarousel to be published on the
     * STATUS bus.
     *
     * @return status
     */
    //TODO refactorize this method because now FilterManager publishes on the bueses the filters state.
    public StatusDataPublishedByCarousel createStatusDataPublishedByCarousel() {
        StatusDataPublishedByCarousel status = new StatusDataPublishedByCarousel();
        status.setName(name);
        status.setPosition(position);
        status.setCurrent(current);
        status.setSpeed(speed);
        status.setHomingDone(homingDone);

        if (Carousel.this.isAtStandby()) {
            status.setAtStandby(true);
            //FCSLOG.debug(name + " Filter at STANDBY=" + this.getFilterAtStandbyName());
            status.setFilterAtStandbyName(this.getFilterAtStandbyName());
            status.setSocketAtStandbyName(this.getSocketAtStandby().getName());
        } else {
            status.setAtStandby(false);
        }
        /*List of filters on carousel*/
        Map<String, String> mapSocketFilter = new TreeMap<>();
        for (Map.Entry<String, CarouselSocket> entry : socketsMap.entrySet()) {
            CarouselSocket socket = entry.getValue();
            if (socket.getFilter() == null) {
                mapSocketFilter.put(socket.getName(), NO_FILTER);
            } else {
                mapSocketFilter.put(socket.getName(), socket.getFilter().getName());
            }
        }
        status.setFiltersOnCamera(mapSocketFilter);
        return status;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(name);
        sb.append("\n Numbers of sockets: ");
        sb.append(socketsMap.size());
        for (CarouselSocket socket : socketsMap.values()) {
            sb.append("\n Socket name ");
            sb.append(String.valueOf(socket.getName()));
            sb.append(" / ");
            sb.append(String.valueOf(socket));
        }
        return sb.toString();
    }

    /**
     * This method let us know if the carousel is ready to receive a filter at
     * standby filterPosition : - the carousel must not rotate - an empty
     * socketAtStandby is at standby filterPosition.
     *
     * @return true if the filterPosition of the carousel matches the
     * filterPosition when one of its sockets is at standby filterPosition and
     * this socketAtStandby is empty. false
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if Carousel is stopped and no filter is at STANDBY position")
    public boolean isReadyToGrabAFilterAtStandby() {

        if (this.isRotating()) {
            return false;
        }
        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            return false;
        }
        return socketAtStandby.isEmpty() && socketAtStandby.isReadyToClamp();
    }

    /**
     *
     * @return true if a filter is clamped at STANDBY position and carousel is
     * stopped
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if a filter is clamped at STANDBY position")
    public boolean isHoldingFilterAtStandby() {
        if (this.isRotating()) {
            return false;
        }
        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null || socketAtStandby.isEmpty()) {
            return false;
        }
        return socketAtStandby.isClampedOnFilter();
    }

    /**
     * Returns true if carousel is rotating
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if carousel is rotating.")
    public boolean isRotating() {
        return isMoving();
    }

    /**
     * Updates the filterPosition of the carousel in reading the CPU of the
     * controller.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Update carousel position in reading controller.")
    public void updatePosition() {
        if (this.homingDone) {
            try {
                this.position = carouselController.readPosition();
            } catch (ShortResponseToSDORequestException ex) {
                FCSLOG.warning(name + "=> ERROR IN READING CONTROLLER:", ex);
            }
            this.publishData();
        } else {
            throw new RejectedCommandException("Carousel homing not done : "
                    + "do carousel homing before updatePosition.");
        }
    }

    /**
     * Updates the field current in reading the CPU of the controller.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update field current in reading controller.")
    public void updateCurrent() {
        try {
            this.current = carouselController.readCurrent();
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(name + "=> ERROR IN READING CONTROLLER:", ex);
        }
        this.publishData();
    }

    /**
     * Read the clamp sensors one by one with RSDO command
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void updateClampsStateWithSensorsFromSDO() {

        for (CarouselSocket socket : socketsMap.values()) {
            socket.updateClampsStateWithSensors();
        }
    }

    /**
     * Read the clamps state from PDO : all the clamp sensors are read at one
     * time.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void updateClampsStateWithSensorsFromPDO() {

        PDOStorage pdoStore = this.tcpProxy.readPDOs();
        FCSLOG.finest(name + ":pdoStore=" + pdoStore.toString());
        /* perhaps do it only for socket at STANDBY*/
        for (CarouselSocket socket : socketsMap.values()) {
            socket.updateClampsStateWithSensors(pdoStore);
        }
    }

    /**
     * Read the clamps state from PDO : all the clamp sensors are read at one
     * time. Tested with success on single-filter-test in April 2013
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
//    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
//            description = "Read clamps sensors and update clamps state")
    public void updateClampsStateWithSensors() {
        updateClampsStateWithSensorsFromPDO();
    }

    /**
     * This method has no action on hardware. This method updates the
     * socketAtStandby at standby filterPosition when a filter is put on the
     * carousel at standby filterPosition. A filter must be detected in the
     * socketAtStandby at standby filterPosition (socketAtStandby not empty).
     *
     * @param filter
     * @throws RejectedCommandException
     */
    private synchronized void putFilterOnCarousel(Filter filter) {

        CarouselSocket socketAtStandby = getSocketAtStandby();

        if (socketAtStandby == null) {
            throw new RejectedCommandException(name + ": there is no socket at standby "
                    + "position to put a filter in.");
        }

        socketAtStandby.putFilterOnSocket(filter);
        this.publishData();
    }

    //TODO throw an exception if filter is not on carousel.
    private int getStandbyPositionForFilter(Filter filter) {
        CarouselSocket socket = socketsMap.get(filter.getSocketName());
        if (socket == null) {
            throw new IllegalArgumentException(name + " no socket name for filter " + filter.getName());
        } else {
            return socket.getStandbyPosition();
        }
    }

    /**
     * Grab a filter at STANDBY position. TO-DO : is this used ?
     *
     * @param filterName
     * @return
     * @throws FcsHardwareException
     */
    public String grabFilterAtStandby(String filterName) {

        FcsMain fcsMainModule = (FcsMain) s.getComponentLookup()
                .getComponentByName("main");
        fcsMainModule.checkFilterName(filterName);

        Filter filterToGrabbe = fcsMainModule.getFilterByName(filterName);

        // the entered filter name is correct
        FCSLOG.info("Filter to move : " + filterName);
        FCSLOG.info("Filter location : " + filterToGrabbe.getFilterLocation());
        return name + grabFilterAtStandby(filterToGrabbe);
    }

    /**
     * This method has to be executed when a filter has just been moved on the
     * carousel at standby filterPosition by the autochanger. It updates the
     * carousel socket at standby or throw an exception.
     *
     * This method tests if the clamps are Locked on a filter at standby
     * filterPosition The clamps Locks automaticaly, and the filter presence
     * sensor tells us if the filter is here or not. If the clamps are Locked on
     * a filter at standby filterPosition, this method updates the
     * socketAtStandby at standby filterPosition : it puts the filter on the
     * socketAtStandby.
     *
     * @param filter
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public synchronized String grabFilterAtStandby(Filter filter) {
        if (filter == null) {
            throw new IllegalArgumentException(name + ": grabbeFilterAtStandby must receive a non null argument as a filter");
        }
        if (this.isRotating()) {
            throw new RejectedCommandException(name + ": grabbeFilterAtStandby can't grabbe a filter while rotating");
        }
        CarouselSocket socketAtStandby = getSocketAtStandby();

        if (socketAtStandby == null) {
            throw new RejectedCommandException(name + ": grabbeFilterAtStandby can't be executed because there is no socket at standby position.");
        }

        socketAtStandby.updateClampsStateWithSensors();

        if (socketAtStandby.isEmpty()) {
            throw new RejectedCommandException(name + ": grabbeFilterAtStandby can't be executed because no filter is detected in socket at standby position.");
        }

        if (socketAtStandby.isClampedOnFilter()) {
            FCSLOG.info(name + ": grabbing " + filter.getName() + " at standby position.");
            putFilterOnCarousel(filter);

            FCSLOG.info(name + ": " + filter.getName() + " is grabbed on carousel.");
            return name + ": " + filter.getName() + " is grabbed at standby position";
        } else {
            throw new RejectedCommandException(name + ": grabbeFilterAtStandby can't be executed because the clamps are not CLAMPED ON FILTER.");
        }

    }

    /**
     * This unlock clamps at STANDBY.
     *
     * @param filterName
     * @throws FcsHardwareException
     */
    public void ungrabFilterAtStandby(String filterName) {

        if (filterName == null) {
            throw new IllegalArgumentException("Filter to ungrabb should not be null");
        }

        CarouselSocket socketAtStandby = getSocketAtStandby();

        if (socketAtStandby == null) {
            throw new RejectedCommandException(name + ":A socket has to be halted at standby position for this operation.");
        }

        if (!socketAtStandby.isClampedOnFilter()) {
            throw new RejectedCommandException(name + ": can't ungrabb filter which is already unclamped.");
        }

        FCSLOG.info(name + ": ungrabbing " + filterName + " at standby position.");

        //TODO test if the filter is held by the autochanger trucks.
        // The clamps can be opened.
        this.unlockClamps();
        FCSLOG.info(name + ": socket at standby state : " + socketAtStandby.toString());

        FCSLOG.info(name + ": " + filterName + " is ungrabbed from carousel.");
    }

    /**
     * Release clamps at standby position to get ready to clamp again.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Release clamps at standby position to get ready to clamp again")
    public void releaseClamps() {

        if (getSocketAtStandby() == null) {
            throw new RejectedCommandException(name
                    + ":Can't release clamps when no socket is halted at standby position.");
        } else {
            getSocketAtStandby().releaseClamps();
        }
    }

    /**
     * Unlocks the clamps at STANDBY.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws FailedCommandException
     * @throws RejectedCommandException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Unlock the clamps at STANDBY.")
    public void unlockClamps() {

        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            throw new RejectedCommandException("Can't unlock clamps while a socket is not halted at standby position.");
        } else {
            FCSLOG.info("Unlocking clamps at standby.");
            socketAtStandby.unlockClamps();
        }
    }

    /**
     * Return true if the filter given as argument is at STANDBY position.
     *
     * @param filter
     * @return
     */
    public boolean isAtStandby(Filter filter) {
        return this.getPosition() == this.getStandbyPositionForFilter(filter);
    }

    public void releaseClampsContact() {
        //TODO;
    }

    @Override
    public boolean myDevicesReady() {
        return this.tcpProxy.allDevicesBooted() && this.initialized;
    }

    /**
     * ****************************************************************
     */
    /**
     * ************ ROTATION COMMANDS *********************************
     */
    /**
     * ****************************************************************
     */
    /**
     * Check if carousel rotation is permitted.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException
     * (RuntimeException)
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if carousel rotation is permitted.")
    public void checkConditionsForRotation() {
        String message;
        if (!initialized) {
            throw new FcsHardwareException("Carousel hardware is not initialized. Can't rotate.");
        }

        //commented out in July 2017 because no TTC-580
//        updateClampsStateWithSensors();
        if (this.isAtStandby() && this.getSocketAtStandby().isUnclampedOnFilter()) {
            message = "Filter at STANDBY position is not held by clamps. Can't rotate carousel.";
            FCSLOG.error(message);
            throw new RejectedCommandException(message);
        }

        if (!autochanger.isAtHandoff()) {
            throw new RejectedCommandException(name + " can't rotate if autochanger is not at HANDOFF position.");
        }
    }

    /**
     * A command to rotate carousel.
     *
     * @param relativePosition
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Rotate carousel to a relative position.",
            timeout = 60000)
    public void rotateToRelativePosition(int relativePosition) {

//        checkConditionsForRotation();
        if (Math.abs(relativePosition) > halfTurn) {
            throw new IllegalArgumentException(relativePosition + ":  |relativePosition| should be <= " + halfTurn);
        }

        FCSLOG.info(name + " is at position: " + position + "; about to rotate to relative position: " + relativePosition);
        relativeTargetPosition = relativePosition;

        //rotate the carousel
        this.executeAction(MobileItemAction.ROTATE_CAROUSEL_TO_RELATIVE_POSITION, rotationTimeout);
        this.publishData();
    }

    /**
     * Rotate carousel to move a socket which name is given as argument to
     * STANDBY position. This methods computes the shortest way to go to STANDBY
     * position.
     *
     * @param socketName
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move a socket which name is given as argument to STANDBY position.",
            alias = "moveSocketToStandby",
            timeout = 50000)
    public void rotateSocketToStandby(String socketName) {

        FcsUtils.checkSocketName(socketName);

        CarouselSocket socketToMove = this.socketsMap.get(socketName);

        int requiredPosition = socketToMove.getStandbyPosition();

        String msgWhenCompleted = name + ":" + socketName + " is at STANDBY position on carousel.";

        if (position != requiredPosition) {
            FCSLOG.info(name + " is at position: " + position + "; about to rotate to position: "
                    + requiredPosition);
            int deltaPosition = requiredPosition - position;
            FCSLOG.info(name + " delta position = " + deltaPosition);
            /* compute shortest way*/
            relativeTargetPosition = computeShortestRelativePosition(deltaPosition);
            rotateToRelativePosition(relativeTargetPosition);
        }
        FCSLOG.info(msgWhenCompleted);
    }

    /**
     * Move an empty socket to STANDBY position. The socket moved is the first
     * empty one in the socketsMap.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void rotateEmptySocketAtStandby() {
        for (Map.Entry<String, CarouselSocket> entry : socketsMap.entrySet()) {
            CarouselSocket socket = entry.getValue();
            if (socket.isEmpty()) {
                rotateSocketToStandby(entry.getKey());
                return;
            }
        }
        FCSLOG.error("no empty socket on carousel - nothing to do.");
    }

    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        if (MobileItemAction.ROTATE_CAROUSEL_TO_RELATIVE_POSITION.equals(action)) {
            return carouselController.isTargetReached();
        }
        return false;
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        try {
            carouselController.checkFault();
            updatePosition();
            updateCurrent();
            FCSLOG.debug(name + " position=" + this.position);
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(name + "=> SDO ERROR IN READING CONTROLLER:", ex);
        }
    }

    /**
     * Starts action ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION or
     * ROTATE_CAROUSEL_TO_RELATIVE_POSITION. This : - enable controller, - make
     * controller go to required position previously written in field
     * absoluteTargetPosition or relativeTargetPosition, - writeControlWord "3F"
     * or "7F" on controller depending if we want to go to a relative position
     * or to an absolute position.
     *
     * @param action
     */
    @Override
    public void startAction(FcsEnumerations.MobileItemAction action) {

        /* carouselController should be initialized*/
        carouselController.checkInitialized();

        /* carouselController should not be in fault*/
        carouselController.checkFault();

        /* a homing must be done before every motion*/
        homing();
        checkHomingDone();

        s.updateAgentState(FcsEnumerations.FilterState.valueOf("ROTATING"),
                FcsEnumerations.FilterReadinessState.NOT_READY);

        if (MobileItemAction.ROTATE_CAROUSEL_TO_RELATIVE_POSITION.equals(action)) {
            carouselController.enableAndWriteRelativePosition(this.relativeTargetPosition);
        }
    }

    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay) {
        FCSLOG.debug(name + " is ABORTING action " + action.toString() + " within delay " + delay);
        this.carouselController.stopPosition();
    }

    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay) {
        this.carouselController.stopPosition();
    }

    @Override
    public void postAction(FcsEnumerations.MobileItemAction action) {
        /* a tempo to let time to the motion command to complete.*/
        try {
            Thread.sleep(3000);
        } catch (InterruptedException ex) {
            FCSLOG.warning(ex);
        }
        this.carouselController.stopPosition();

        /* At the end of an action a homing has to be done.*/
        homing();
        this.carouselController.disable();

        if (!s.isInState(AlertState.ALARM)) {
            s.updateAgentState(FcsEnumerations.FilterState.CAROUSEL_STOPPED);
        }

    }

    /**
     *
     * @param pos
     * @return
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "compute new position to take in account tours which can have been runned")
    public int computeNewPosition(int pos) {
        int newPos;
        int modulo = pos % stepNB;
        if (pos * encoderSign > 0 || modulo == 0) {
            newPos = modulo;
        } else {
            newPos = encoderSign * (stepNB - Math.abs(modulo));
        }
        FCSLOG.finest("stepNB=" + stepNB + " pos=" + pos + " modulo=" + modulo
                + " ==> new Position = " + newPos);
        return newPos;
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Compute relative position to find the shortest way to run carousel.")
    public int computeShortestRelativePosition(int d) {
        int relativePosition;
        if (Math.abs(d) <= halfTurn) {
            relativePosition = d;
        } else {
            int s;
            if (d > 0) {
                s = -1;
            } else {
                s = 1;
            }
            relativePosition = s * Math.min(Math.abs(d), stepNB - Math.abs(d));
        }
        return relativePosition;
    }

    /**
     * ****************************************************************
     */
    /**
     * ************ END of ROTATION COMMANDS **************************
     */
    /**
     * ****************************************************************
     */

    /**
     * ****************************************************************
     */
    /**
     * ************ methods which override FilterHolder *** ***********
     */
    /**
     * ****************************************************************
     */
    /**
     * Return true if carousel is holding a filter at STANDBY position.
     *
     * @return
     * @throws FcsHardwareException
     */
    @Override
    public boolean isHoldingFilter() {
        return this.isHoldingFilterAtStandby();
    }

    @Override
    public boolean isAtHandoff() {
        return false;
    }

    @Override
    public void updateStateWithSensors() {
        this.updateClampsStateWithSensors();
    }

    /**
     * ****************************************************************
     */
    /**
     * ************ END of methods which override FilterHolder ********
     */
    /**
     * ****************************************************************
     */
}
