package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.MASTER_ENCODER;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.PROFILE_POSITION;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.AC_TRUCKS_ERROR;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.*;
import org.lsst.ccs.subsystems.fcs.common.ADCInterface;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerForLinearRail;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenCallTimeoutException;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;

/**
 * This is a model for autochanger trucks. Autochanger trucks move a filter from
 * STANDBY position to ONLINE position or HANDOFF position. ONLINE position is
 * just in front of the focal front. STANDBY position is where an exchange can
 * be done with the carousel. HANDOFF position is a position an exchange can be
 * done with the loader. The trucks are moved by 2 controllers, a master and a
 * slave. Each controller has an absolute encoder.
 * **************************************** MASTER CONTROLLER FOR truckXminus
 * SLAVE CONTROLLER FOR truckXplus ****************************************
 *
 * @author virieux
 */
public class AutochangerTwoTrucks extends MobileItem {

    AutochangerTruck truckXminus;
    AutochangerTruck truckXplus;

    private final int minActualPositionValue = 0;
    private final int maxActualPositionValue = 998000;

    @ConfigurationParameter(description = "STANDBY position in micron")
    private int standbyPosition = 992000;

    @ConfigurationParameter(description = "HANDOFF position in micron")
    private int handoffPosition = 70000;

    @ConfigurationParameter(description = "ONLINE position in micron")
    private int onlinePosition = 2600;

    @ConfigurationParameter(description = "timeout for trucks motion from ONLINE to STANDBY in milliseconds")
    private long timeoutForTrucksMotion = 20000;    
    
    /******************** parameters to move filter to online position and adjust position (docking process) ********/
    
    @ConfigurationParameter(description = "prelimary ONLINE position in micron, used by process which"
            + "moves a filter at ONLINE position.")    
    private int prelimaryTargetPosition = 3367;
    
    @ConfigurationParameter(description = "natural position at ONLINE position in micron  used by process which"
            + "moves a filter at ONLINE position.")
    private int naturalPosition = 3215;
    
    @ConfigurationParameter(description = "minimum value of voltage read on proximitySensor. Unit: Volts")
    private double umin = 4.2;
    
    @ConfigurationParameter(description = "maximum value of voltage read on proximitySensor. Unit: Volts")
    private double umax = 4.6;    
    
    @ConfigurationParameter(description = "voltage target on proximitySensor. Unit: Volts")
    private double utarget = 3.85;
    
    @ConfigurationParameter(description = "adjustment factor. Unit: micrometre/volt")
    private double adjustmentFactor = 835;
    
    /*************end of  parameters to move filter to online position and adjust position (docking process) ********/
    


    private int current = 0;
    private int speed = 0;
    private int masterPosition;
    private int slavePosition;
    private int deltaPosition;

    /**
     * masterPosition is reached if |targetPosition - masterPosition| <
     * positionRange
     */
    @ConfigurationParameter(description = "masterPosition is reached if |targetPosition - masterPosition| < positionRange")
    private int positionRange = 2;

    /**
     * an alignment of slave must be done when |deltaPosition| > deltaPositionMax
     */
    @ConfigurationParameter(description = "an alignment of slave must be done when |deltaPosition| > deltaPositionMax")
    private int deltaPositionMax = 100;

    private int absoluteTargetPosition = 0;

    private EPOSControllerForLinearRail linearRailMasterController;
    private EPOSControllerForLinearRail linearRailSlaveController;

    @LookupField(strategy = Strategy.TREE)
    private Autochanger autochanger;
    
    @LookupField(strategy = Strategy.TREE)
    private ADCInterface proximitySensorsDevice;
    
    @ConfigurationParameter(description = "input number on the ADC where proximity sensor is plugged")
    private int proximitySensorInput = 1;


    /**
     * true if Handoff position sensors for TruckXminus and TruckXplus return
     * value 1*
     */
    private boolean atHandoff = false;

    /**
     * true if Online position sensors for TruckXminus and TruckXplus return
     * value 1 *
     */
    private boolean atOnline = false;

    /**
     * true if Standby position sensors for TruckXminus and TruckXplus return
     * value 1 *
     */
    private boolean atStandby = false;
    
    /* Tools for check if position is reached when moving trucks to a given position*/
    private ScheduledFuture<?> checkPositionHandle;

    /**
     * Build a new AutoChangerTrucksModule.
     *
     * @param truckXminus
     * @param truckXplus
     * @param linearRailMasterController
     * @param linearRailSlaveController
     */
    public AutochangerTwoTrucks(
            AutochangerTruck truckXminus,
            AutochangerTruck truckXplus,
            EPOSControllerForLinearRail linearRailMasterController,
            EPOSControllerForLinearRail linearRailSlaveController
    ) {
        this.truckXminus = truckXminus;
        this.truckXplus = truckXplus;
        this.linearRailMasterController = linearRailMasterController;
        this.linearRailSlaveController = linearRailSlaveController;
    }

    @ConfigurationParameterChanger
    public void setNaturalPosition(int pos) {
        int minLimit = onlinePosition - 1500;
        int maxLimit = onlinePosition + 1500;
        if (pos < minLimit || pos > maxLimit) {
            throw new IllegalArgumentException(pos + " bad value for natural position"
                    + " must be > " + minLimit + " and < " + maxLimit);
        }
        this.naturalPosition = pos;
    }
    
    

    /**
     * Return actual trucks masterPosition. Doesn't read again the
     * masterPosition on the controller.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return actual trucks position (master position). Doesn't read again the position on the controller.")
    public int getPosition() {
        return masterPosition;
    }

    /**
     * Return HANDOFF masterPosition. Doesn't read again the masterPosition on
     * the controller.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return HANDOFF position. Doesn't read again the position on the controller.")
    public int getHandoffPosition() {
        return handoffPosition;
    }

    /**
     * Return STANDBY masterPosition. Doesn't read again the masterPosition on
     * the controller.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return STANDBY position. Doesn't read again the position on the controller.")
    public int getStandbyPosition() {
        return standbyPosition;
    }

    /**
     * Return ONLINE position. Doesn't read again the position on the
     * controller.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return ONLINE position. Doesn't read again the position on the controller.")
    public int getOnlinePosition() {
        return onlinePosition;
    }

    /**
     * Return the timeout for trucks motion. If trucks motion duration is
     * greater than timeoutForTrucksMotion, an ALERT is raised.
     *
     * @return timeoutForTrucksMotion
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return timeout for trucks motion.")
    public long getTimeoutForTrucksMotion() {
        return timeoutForTrucksMotion;
    }

    /**
     * Returns the boolean field atHandoff. If the atHandoff boolean is being
     * updated and waits for a response from a sensor, this methods waits until
     * update is completed. If the field atHandoff is not being updated, it
     * returns its immediatly.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the carrier is at handoff position. "
            + "This command doesn't read again the sensors.")
    public boolean isAtHandoff() {
        return atHandoff;
    }

    /**
     * Returns the boolean field atOnline. If the atOnline boolean is being
     * updated and waits for a response from a sensor, this methods waits until
     * update is completed. If the field atOnline is not being updated, it
     * returns its immediatly.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the carrier is at ONLINE position. "
            + "This command doesn't read again the sensors.")
    public boolean isAtOnline() {
        return atOnline;
    }

    /**
     * Returns the boolean field atStandby. If the atHandoff boolean is being
     * updated and waits for a response from a sensor, this methods waits until
     * update is completed. If the field atStandby is not being updated, it
     * returns its immediatly..
     *
     * @return atStandby
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the carrier is at STANDBY position. "
            + "This command doesn't read again the sensors.")
    public boolean isAtStandby() {
        return atStandby;
    }

    /**
     * Return true if the homing of the trucks has been done.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the homing of the trucks has been done.")
    public boolean isHomingDone() {
        return linearRailSlaveController.isHomingDone()
                && linearRailMasterController.isHomingDone();
    }

    /**
     * Return true if TruckXminus position sensors or TruckXminus position
     * sensors are in error. Or position sensors are non consistant.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if TruckXminus position sensors or TruckXminus position sensors are in error.")
    public boolean isPositionSensorsInError() {
        return truckXminus.isPositionSensorsInError()
                || truckXplus.isPositionSensorsInError()
                || positionSensorsNonConsistant();
    }

    private boolean positionSensorsNonConsistant() {
        return (isAtStandby() && isAtHandoff()
                || (isAtStandby() && isAtOnline()
                || (isAtHandoff() && isAtOnline())));
    }

    /**
     * check that truckXminus is controlled by master controller and truckXplus
     * is controlled by slave controller.
     */
    private void checkControllersName() {
        if (truckXminus.getControllerName().contains("Master")) {
            FCSLOG.info(name + " is controlled by " + truckXminus.getControllerName());
        } else {
            throw new FcsHardwareException("truckXminus has to be controlled by Master controller.");
        }

        if (truckXplus.getControllerName().contains("Slave")) {
            FCSLOG.info(name + " is controlled by " + truckXplus.getControllerName());
        } else {
            throw new FcsHardwareException("truckXplus has to be controlled by Slave controller.");
        }
    }

    @Override
    public void postStart() {
        FCSLOG.fine(name + " BEGIN postStart.");

        if (linearRailSlaveController.isBooted() && linearRailMasterController.isBooted()) {
            initializeControllers();
        }
        FCSLOG.fine(name + " END postStart.");
    }

    public void initializeControllers() {
        try {
            checkControllersName();
            linearRailSlaveController.initializeAndCheckHardware();
            linearRailMasterController.initializeAndCheckHardware();
//            configureControllers();
            homing();

        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(name, ex);

        } catch (FailedCommandException | FcsHardwareException ex) {
            //TODO voir quel controleur n'a pas pu etre initialisé.
            this.raiseAlarm(FcsEnumerations.FcsAlert.HARDWARE_ERROR, "Couldn't initialize controllers",
                    name, ex);
        }
//        this.checkDeltaPosition();
    }

    /**
     * Configure Autochanger trucks master and slave controllers.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Configure Autochanger trucks master and slave controllers.",
            alias = "initControllers")
    //tested on CPPM testbench in september 2015

    public void configureControllers() {
        FCSLOG.info(name + " Begin configuration of the controllers");
        linearRailMasterController.configureDigitalInputOfLinearRails();
        linearRailSlaveController.configureDigitalInputOfLinearRails();
        linearRailMasterController.configureDigitalOutputOfLinearRails();
        linearRailSlaveController.configureDigitalOutputOfLinearRails();
        FCSLOG.info(name + " End configuration of the controllers");
    }

    /**
     * Return true if both controllers are initialized and homing of trucks are
     * done.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if both controllers are initialized and homing of trucks are done.")
    public boolean isInitialized() {
        return linearRailMasterController.isInitialized()
                && linearRailSlaveController.isInitialized()
                && this.isHomingDone();
    }

    /**
     * This method sends Exceptions if the conditions for trucks motion are not
     * filled. Autochanger trucks motion is allowed if : - controllers are
     * initialized - homing is done - trucks are empty OR (loaded with a filter
     * but latches are closed AND carousel or loader doesn't hold filter)
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the motion of trucks are allowed.")
    public void checkConditionsForTrucksMotion() {

        if (!linearRailSlaveController.isInitialized() || !linearRailMasterController.isInitialized()) {
            String msg = name + " can't move autochanger trucks because controllers are not both initialized:\n";
            FCSLOG.error(msg);
            throw new RejectedCommandException(msg);
        }
        if (!isHomingDone()) {
            throw new RejectedCommandException(name
                    + " can't be moved because the homing has not been done yet. ");
        }

        updatePosition();
        checkDeltaPosition();
        autochanger.checkFilterSafetyBeforeMotion();
    }

    /**
     * Move Autochanger trucks to the Handoff position. If Autochanger trucks
     * are already at HANDOFF position, it does nothing.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move Autochanger trucks to the Handoff position.",
            alias = "goToHandoff")
    public void goToHandOff() {
        s.updateAgentState(FcsEnumerations.FilterState.MOVING_TRUCKS_TO_STANDBY,
            FcsEnumerations.FilterReadinessState.NOT_READY);
        moveToAbsoluteTargetPosition(this.handoffPosition);
    }

    /**
     * Move Autochanger trucks to the Online position. If Autochanger trucks are
     * already at ONLINE position, it does nothing.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move Autochanger trucks to the Online position.")
    public void goToOnline() {
        s.updateAgentState(FcsEnumerations.FilterState.MOVING_TRUCKS_TO_ONLINE,
                FcsEnumerations.FilterReadinessState.NOT_READY);
        moveToAbsoluteTargetPosition(this.onlinePosition);
    }

    
    
    /**
     * Move Autochanger trucks to the Standby position. If Autochanger trucks
     * are already at STANDBY position, it does nothing.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move Autochanger trucks to the Standby position.")
    public void goToStandby() {
        s.updateAgentState(FcsEnumerations.FilterState.MOVING_TRUCKS_TO_STANDBY,
                FcsEnumerations.FilterReadinessState.NOT_READY);
        moveToAbsoluteTargetPosition(this.standbyPosition);
        
    }

    /**
     * Move the trucks to an absolute position given as argument.
     * Doesn't align slave at the end of motion.
     * At the end of 
     *
     * @param targetPosition
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move Autochanger trucks to the absolute position given as argument. "
                    + "At the end of motion : doesn't align slave, active brakes and disable controllers",
            alias = "mobeABSPos", timeout = 20000)
    public void moveToAbsoluteTargetPosition(int targetPosition) {

        if (targetPosition < minActualPositionValue || targetPosition > maxActualPositionValue) {
            throw new IllegalArgumentException(name + "argument has to be between "
                    + minActualPositionValue + " and " + maxActualPositionValue);
        }

        this.updatePosition();
        if (masterPosition == targetPosition) {
            FCSLOG.info(name
                    + " is already at target position=" + targetPosition);
        } else {

            checkConditionsForTrucksMotion();
            checkControllersMode();
            checkReadyForAction();
            this.absoluteTargetPosition = targetPosition;

            FCSLOG.info(name + " going to absolute position: " + absoluteTargetPosition);
            goToPosition(targetPosition);
            checkPositionSensors();
            publishData();
        }
    }
    
    
    /**
     * move filter at online position and lock filter with a docking process.
     */
    @Command(type=Command.CommandType.ACTION, description="goToOnline with filter, adjust position, "
            + "close and lock clamps")
    public void moveAndClampFilterOnline() {
        FCSLOG.info(name + " BEGIN process docking ONLINE.");
        s.updateAgentState(FcsEnumerations.FilterState.MOVING_TRUCKS_TO_ONLINE,
            FcsEnumerations.FilterReadinessState.NOT_READY);
        /* move to prelimaryTargetPosition */
        moveToAbsoluteTargetPosition(prelimaryTargetPosition);
        /* align slave if |deltaPosition| > deltaPositionMax */
        alignSlave();
        /* adjust position*/
        double umeasured = proximitySensorsDevice.readVoltage(this.proximitySensorInput);
        int newTargetPosition = (int) (prelimaryTargetPosition 
                + adjustmentFactor * (this.utarget - umeasured));
        FCSLOG.info(name + " new target position=" + newTargetPosition);
        moveToAbsoluteTargetPosition(newTargetPosition);
        /* close and lock clamps*/
        autochanger.getOnlineClamps().closeAndLockClamps();
        umeasured = proximitySensorsDevice.readVoltage(this.proximitySensorInput);
        if (umeasured > umax) {
            this.raiseAlarm(AC_TRUCKS_ERROR, " voltage returned by proximity sensor is too high.");
        } else if (umeasured < umin) {
            this.raiseAlarm(AC_TRUCKS_ERROR, " voltage returned by proximity sensor is too low.");
        }
        goToPosition(naturalPosition);
        FCSLOG.info(name + " END process docking ONLINE.");
        autochanger.updateFCSStateToReady();
        this.publishData();
    }
    
    /**
     * move trucks to target position.
     * don't use MobileItem
     * @param targetPosition 
     */
    @Deprecated //see moveToAbsoluteTargetPosition
    public void moveTrucksToPosition(int targetPosition) {
        this.updatePosition();
        checkConditionsForTrucksMotion();
        checkControllersMode();
        checkReadyForAction();
        goToPosition(targetPosition);
        publishData();
    }
    
    /**
     * no control are done
     * user must check that this moving is possible.
     * @param targetPosition 
     */
    private void goToPosition(int targetPosition) {
        FCSLOG.info(name + " moving to position " + targetPosition);
        this.currentAction = MobileItemAction.MOVE_TO_ABSOLUTE_POSITION; 
        this.absoluteTargetPosition = targetPosition;
        this.enableControllersAndReleaseBrake();
        try {
            linearRailMasterController.writeTargetPosition(targetPosition);
            linearRailMasterController.writeControlWord(0x3F);
            checkActionCompleted(250, 250);
            waitForEndOfMotion(this.timeoutForTrucksMotion);
        } catch (CanOpenCallTimeoutException ex) {
            FCSLOG.error(ex);
            
        } catch (Exception ex) {
            FCSLOG.error(ex);
            throw new FcsHardwareException(name + " could not go to position: " + targetPosition +
                    " because " + ex);
        } finally {
            //because we don't want to let the controllers on power
            stopGoingToPosition();
        }
        FCSLOG.info(name + " end moving to position " + targetPosition);
    }
    
    private void stopGoingToPosition() {
        this.linearRailMasterController.activateBrake();
        this.linearRailSlaveController.activateBrake();
        /* mandatory : we must let time to the controller to activate the brake */
        /* otherwise if we disable before brake are holding, a filter could fall. */
        try {
            Thread.sleep(500);
        } catch (InterruptedException ex) {
            FCSLOG.error(ex);
        }
        linearRailMasterController.stopPosition();
        linearRailMasterController.disable();
        linearRailSlaveController.disable();
    }
    

    /* not used anymore october 2017*/
    @Override
    public void startAction(FcsEnumerations.MobileItemAction action) {
        switch (action) {
            case MOVE_TO_ABSOLUTE_POSITION:
                checkControllersMode();
                this.enableControllersAndReleaseBrake();
                linearRailMasterController.writeTargetPosition(this.absoluteTargetPosition);
                linearRailMasterController.writeControlWord(0x3F);
                this.increaseCurrentMonitoring();
                break;

            default:
                assert false : action;
        }
    }

    /**
     * This method is used in the startAction method. It is was has to be done
     * before moving the trucks.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Enable and release brake for the 2 controllers.")
    public void enableControllersAndReleaseBrake() {
        linearRailMasterController.enable();
        linearRailSlaveController.enable();
        linearRailMasterController.doReleaseBrake();
        if (linearRailMasterController.isEnabled()) {
            linearRailSlaveController.doReleaseBrake();
        } else {
            throw new FcsHardwareException(name + " can't release Slave Controller brake "
                    + " because Master Controller is disabled.");
        }
    }

    private void checkControllersMode() {
        if (!linearRailMasterController.isInMode(EposMode.PROFILE_POSITION)) {
            throw new FcsHardwareException(name + " : linearRailMasterController is not"
                    + " in PROFILE_POSITION mode, can't move.");
        }

        if (!linearRailSlaveController.isInMode(EposMode.MASTER_ENCODER)) {
            throw new FcsHardwareException(name + " : linearRailSlaveController is not"
                    + " in MASTER_ENCODER mode, can't move. Slave controller mode =" + linearRailSlaveController.getMode());
        }
    }

    /**
     * move slave truk to masterPosition. When moving along the rails with slave
     * controller in MASTER_ENCODER mode, at the end of motion there is a small
     * misalignement. This consists in : setting slaveController mode to
     * PROFILE_POSITION, go to absolute masterPosition pos, and then set
     * slaveController mode back to MASTER_ENCODER mode.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Align slave controller position to master controller position.")
    public void alignSlave() {
        //master controller must be disabled and brake activated
        if (linearRailMasterController.isEnabled()) {
            throw new FcsHardwareException(name + " master controller must be disabled.");
        } else if (!linearRailMasterController.isBrakeActivated()) {
            throw new FcsHardwareException(name + " master controller brake must be activated.");
        }

        updatePosition();
        if (Math.abs(deltaPosition) > deltaPositionMax) {
            linearRailSlaveController.changeMode(PROFILE_POSITION);
            try {
                doAlignSlave(masterPosition);
            } finally {
                stopAlignSlave();
            }
        }
    }
    
    private void stopAlignSlave() {
        FCSLOG.info(name + " STOP ALIGN SLAVE");
        linearRailSlaveController.activateBrake();
        /* mandatory : we must let time to the controller to activate the brake */
        /* otherwise if we disable before brake are holding, a filter could fall. */
        try {
            Thread.sleep(500);
        } catch (InterruptedException ex) {
            FCSLOG.error(ex);
        }
        linearRailSlaveController.changeMode(MASTER_ENCODER);
        linearRailSlaveController.disable();
        FCSLOG.info(name + " END STOP ALIGN SLAVE");
    }

    private void doAlignSlave(int pos) {
        FCSLOG.info(name + " slave's alignment to be done. deltaPosition= " + deltaPosition);
        this.currentAction = MobileItemAction.ALIGN_SLAVE;
        int timeoutForSlaveAlignment = 20000;
        if (!linearRailSlaveController.isInMode(PROFILE_POSITION)) {
            throw new FcsHardwareException(name + " can't align slave truck because"
                    + " couldn't change its mode to PROFILE_POSITION.");
        }
        linearRailSlaveController.enableAndReleaseBrake();
        linearRailSlaveController.writeTargetPosition(pos);
        linearRailSlaveController.writeControlWord(0x3F);
        checkActionCompleted(250,250);
        waitForEndOfMotion(timeoutForSlaveAlignment);
        this.publishData();
        FCSLOG.info(name + " end of slave's alignment. deltaPosition= " + deltaPosition);
    }

    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay) {
        FCSLOG.info(name + "ABORTING action: " + action.toString());
        stopGoingToPosition();
        this.decreaseCurrentMonitoring();
    }

    @Override
    //tested on CPPM testbench in september 2015
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay) {
        this.linearRailMasterController.quickStop();
        this.linearRailSlaveController.quickStop();
    }

    /* not used anymore october 2017*/
    @Override
    public void postAction(FcsEnumerations.MobileItemAction action) {
        FCSLOG.info(name + " doing postAction: " + action.toString());
        this.publishData();
        //because we don't want to let the controllers on power
        this.linearRailMasterController.activateBrakeAndDisable();
        this.linearRailSlaveController.activateBrakeAndDisable();
        FCSLOG.info(name + ":" + action.toString() + " completed - doing postAction: checking position sensors.");

        if (action == MOVE_TO_ABSOLUTE_POSITION) {
            checkPositionSensors();
        }
        publishData();
        alignSlave();
        autochanger.updateFCSStateToReady();
    }

    /**
     * This method is called in the postAction method to check that after a
     * motion to well known position as STANDBY, HANDOFF and ONLINE, position
     * sensors confirm the trucks position. Otherwise it throws a
     * FailedCommandException.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void checkPositionSensors() {
        /*Read position sensors and check that values return are consistant with trucks position
         otherwise throw an Exception.*/
        autochanger.updateStateWithSensors();
        if (this.absoluteTargetPosition == this.standbyPosition && !this.atStandby) {
            throw new FailedCommandException(name
                    + ": check with sensors: STANDBY sensors don't confirm trucks position.");

        } else if (this.absoluteTargetPosition == this.handoffPosition && !this.atHandoff) {
            FCSLOG.info("this.absoluteTargetPosition=" + this.absoluteTargetPosition);
            FCSLOG.info("this.handoffPosition=" + this.handoffPosition);
            FCSLOG.info("this.atHandoff=" + this.atHandoff);
            throw new FailedCommandException(name
                    + ": check with sensors: HANDOFF sensors don't confirm trucks position.");

        } else if (this.absoluteTargetPosition == this.onlinePosition && !this.atOnline) {
            throw new FailedCommandException(name
                    + ": check with sensors: ONLINE sensors don't confirm trucks position.");
        }
    }

    /**
     * Do homing of both controllers.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Do homing for both  controllers.")
    public void homing() {
        linearRailMasterController.homing();
        linearRailSlaveController.homing();
        updatePosition();
    }

    /**
     * Updates the field position of the trucks in reading the CPU of the
     * controller.
     *
     * *********************************************************************
     * MASTER CONTROLLER FOR truckXminus SLAVE CONTROLLER FOR truckXplus
     * *********************************************************************
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    //tested on CPPM testbench in april 2017
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update trucks position in reading controller.")
    public void updatePosition() {
        this.truckXminus.updatePosition();
        this.truckXplus.updatePosition();
        this.masterPosition = this.truckXminus.getPosition();
        this.slavePosition = this.truckXplus.getPosition();
        this.deltaPosition = this.masterPosition - this.slavePosition;
        this.publishData();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read position on controllers, compute difference and throw exception"
            + "if difference is > 1000.")
    public void checkDeltaPosition() {
        updatePosition();
        if (Math.abs(deltaPosition) > 1000) {
            String msg = "Too much difference between trucks position. "
                    + "Alignment of Slave Truck to be done.";
            this.raiseAlarm(AC_TRUCKS_ERROR, msg);
            throw new FcsHardwareException(name + ":" + msg);
        }
        this.publishData();
    }

    /**
     * For the GUI and end users for test purpose. Update positon in reading the
     * master controller and returns the value read.
     *
     * @return trucks masterPosition
     * @throws RejectedCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update trucks position in reading controller and return position.")
    public int readPosition() {
        updatePosition();
        return masterPosition;
    }

    /**
     * @return true if both controllers are initialized and homing of trucks are
     * done.
     */
    @Override
    public boolean myDevicesReady() {
        //commented out for CPPM testbench
        //return autochanger.myDevicesReady();
        return this.isInitialized();
    }

    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        switch (action) {
            case MOVE_TO_ABSOLUTE_POSITION:
                return linearRailMasterController.isTargetReached()
                        && Math.abs(this.masterPosition - this.absoluteTargetPosition) < positionRange;
            case ALIGN_SLAVE:
                return linearRailSlaveController.isTargetReached()
                        && Math.abs(this.deltaPosition) < deltaPositionMax ;

            default:
                assert false : action;
        }
        return false;
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        try {
            linearRailMasterController.checkFault();
            linearRailSlaveController.checkFault();
            updatePosition();
            FCSLOG.debug(name + " position=" + this.masterPosition);
            autochanger.updateStateWithSensors();
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(name + "=> SDO ERROR IN READING TRUCKS CONTROLLER:", ex);
        } catch (SDORequestException ex) {
            FCSLOG.error(name + "=> SDO ERROR IN READING TRUCKS CONTROLLER:", ex);
        } catch (CanOpenCallTimeoutException ex) {
            /*can be catched because updateStateWithSensorsToCheckIfActionIsCompleted is always called 
            * in a Runnable so it will be read again in the next run.
            */
            FCSLOG.error(name + "=> SDO ERROR IN READING TRUCKS CONTROLLERS:CanOpenCallTimeoutException", ex);
        }
    }

    void updateStateWithSensors(int[] readHexaValues) {
        truckXminus.updateStateWithSensors(readHexaValues);
        truckXplus.updateStateWithSensors(readHexaValues);
        this.atHandoff = truckXminus.isAtHandoff() && truckXplus.isAtHandoff();
        this.atOnline = truckXminus.isAtOnline() && truckXplus.isAtOnline();
        this.atStandby = truckXminus.isAtStandby() && truckXplus.isAtStandby();
    }

    /**
     * release brake on slave controller. This command is forbidden if slave is
     * in MASTER_ENCODER mode and master is disabled because in this case,
     * releasing slave could lead to a filter fall.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Release slave controller brake.")
    public void releaseBrakeSlave() {
        if (linearRailSlaveController.isInMode(MASTER_ENCODER)
                && !linearRailMasterController.isEnabled()) {
            throw new RejectedCommandException(name + " couldn't release slave controller brake"
                    + "because master controller is disabled and slave controller is in MASTER_ENCODER mode");
        } else {
            linearRailSlaveController.doReleaseBrake();
        }
    }

    /**
     * release brake on master controller. This command is forbidden if slave is
     * in MASTER_ENCODER mode and master is disabled because in this case,
     * releasing slave could lead to a filter fall.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Release master controller brake.")
    public void releaseBrakeMaster() {
        linearRailMasterController.doReleaseBrake();
    }
    
    
     /**
     * check periodicaly if position pos is reached, update sensors state.
     * @param waitTime
     * @param period 
     */
    public void checkActionCompleted(int waitTime, int period) {
        final Runnable checkPosition = () -> {
            updateStateWithSensorsToCheckIfActionIsCompleted();
            if (isActionCompleted(currentAction)) {
                FCSLOG.info(name + " ==> " + currentAction + " " + absoluteTargetPosition 
                        + " position  has been reached.");
                cancelCheckPosition();
            } else {
                FCSLOG.info(name + " ==> " + currentAction + " " + absoluteTargetPosition 
                        + " position  not yet reached.");
            }
        };
        this.checkPositionHandle = scheduler.scheduleAtFixedRate(checkPosition, waitTime, period, 
                TimeUnit.MILLISECONDS);
    }
    
    private void cancelCheckPosition() {
        this.checkPositionHandle.cancel(true);
        FCSLOG.debug(" => check position ended");
    }
    
    private void waitForEndOfMotion(long timeout) {
        FCSLOG.info(name + " BEGIN WAITING FOR END OF MOTION");
        try {
            checkPositionHandle.get(timeout, TimeUnit.MILLISECONDS);
        } catch (CancellationException ex) {
            if (checkPositionHandle.isDone()) {
                FCSLOG.info(name + " CHECKING POSITION ENDED");
            } else {
                FCSLOG.error(name + " during action " + this.currentAction 
                        + " waiting was cancelled before END of motion "+ ex);
            }
        } catch (TimeoutException ex) {
            FCSLOG.error(name + currentAction.getFailureMsg() + " GOING TO POSITION IS LASTING TOO LONG - timeout=" + timeout, ex);
            throw new FcsHardwareException(name + currentAction.getFailureMsg() + " motion is lasting too long - timeout " + timeout, ex);

        } catch (InterruptedException ex) {
            FCSLOG.error(name + " interrupted while waiting end of motion during action=" + currentAction, ex);
            throw new FcsHardwareException(name + " interrupted while waiting end of motion during action=" + currentAction, ex);
            
        } catch (ExecutionException ex) {
            FCSLOG.error(name + " error during action=" + currentAction, ex);
            throw new FcsHardwareException(name + " error during action=" + currentAction + ex);

        } finally {
            cancelCheckPosition();
            FCSLOG.info(name + " STOP WAITING FOR END OF MOTION");
        }
    }   
    
    /**
     * speed up online clamps current monitoring for the 3 clamps
     */
    @Command(type=Command.CommandType.QUERY)
    public void increaseCurrentMonitoring() {
        //commented for tests
//        truckXminus.increaseCurrentMonitoringSpeed();
//        truckXplus.increaseCurrentMonitoringSpeed();
    }
    
    /**
     * slow down online clamps current monitoring for the 3 clamps
     */
    
    @Command(type=Command.CommandType.QUERY)
    public void decreaseCurrentMonitoring() {
        truckXminus.decreaseCurrentMonitoringSpeed();
        truckXplus.decreaseCurrentMonitoringSpeed();
    } 
    
    

    /**
     * Creates an object to be published on the status bus.
     *
     * @return
     */
    public StatusDataPublishedByAutoChangerTrucks createStatusDataPublishedByAutoChangerTrucks() {
        StatusDataPublishedByAutoChangerTrucks s = new StatusDataPublishedByAutoChangerTrucks();
        s.setMasterPosition(masterPosition);
        s.setSlavePosition(slavePosition);
        s.setCurrent(current);
        s.setSpeed(speed);
        s.setHomingDone(isHomingDone());
        s.setAtHandoff(atHandoff);
        s.setAtOnline(atOnline);
        s.setAtStandby(atStandby);
        s.setInError(isPositionSensorsInError());
        return s;
    }

    @Override
    public void publishData() {
        s.publishSubsystemDataOnStatusBus(new KeyValueData(name,
                createStatusDataPublishedByAutoChangerTrucks()));
    }
    


}
