package org.lsst.ccs.subsystems.fcs;

import java.time.Duration;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.locks.Condition;
import java.util.logging.Level;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.AC_SENSOR_ERROR;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.ControlledBySensors;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerWithBrake;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This is a model for online clamp which holds a filter when it is at ONLINE
 * position. The online clamps are part of the autochanger subsystem. There is 3
 * online clamps.
 *
 * @author virieux
 */
public class AutochangerOnlineClamp extends MobileItem implements MovedByEPOSController, ControlledBySensors {

    public static final String ONLINE_CLAMP_CONFIG_CATEGORY = "onlineClampCurrent";
    public static final String CURRENT_MONITOR_TASK_NAME = "-monitorCurrent";

    public static final int TIMEOUT_FOR_CLOSING = 15000;
    public static final int TIMEOUT_FOR_OPENING = 15000;

    @LookupField(strategy = Strategy.TREE)
    private MainModule mainModule;

    @LookupField(strategy = Strategy.TREE)
    private Autochanger autochanger;
    private final EPOSControllerWithBrake controller;

    /* couple of close sensors*/
    private final ComplementarySensors closeSensors;

    /* couple of open sensors*/
    private final ComplementarySensors openSensors;
    
    @LookupField(strategy = Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;    

    /**
     * currentToClamp is different for each clamp so the value is initialized
     * from configuration system.
     */
    @ConfigurationParameter(description = "current to clamp ONLINE clamp in mA", category = ONLINE_CLAMP_CONFIG_CATEGORY)
    private int currentToClamp = 0;

    @ConfigurationParameter(description = "current to open ONLINE clamp in mA", category = ONLINE_CLAMP_CONFIG_CATEGORY)
    private int currentToOpen = 0;

    /**
     * sentCurrent is not really a ConfigurationParameter but we want to save
     * this value of current between two instances of running FCS and each time
     * we sent current to the controller.
     *
     */
    @ConfigurationParameter(description = "current sent to controller in mA", category = ONLINE_CLAMP_CONFIG_CATEGORY)
    private int sentCurrent = 0;
    
    /**
     * current read on the controller
     */
    private int current;

    /**
     * increment of current in a current ramp, in mA
     */
    @ConfigurationParameter(description = "increment of current to lock in a current ramp in mA", category = ONLINE_CLAMP_CONFIG_CATEGORY)
    private int incrementCurrentToClamp = 250;

    /**
     * increment of current to close a clamp in a current ramp, in mA
     */
    @ConfigurationParameter(description = "increment of current to close clamp in a current ramp in mA", category = ONLINE_CLAMP_CONFIG_CATEGORY)
    private int incrementCurrentToClose = 250;

    /**
     * increment of current to open a clamp in a current ramp, in mA
     */
    @ConfigurationParameter(description = "increment of current to open clamp in a current ramp in mA", category = ONLINE_CLAMP_CONFIG_CATEGORY)
    private int incrementCurrentToOpen = 250;

    /**
     * initial current to close a clamp in a current ramp, in mA
     */
    @ConfigurationParameter(description = "initial current to close clamp in a current ramp in mA",
            category = ONLINE_CLAMP_CONFIG_CATEGORY)
    private int initialCurrentToClose = 0;

    /**
     * initial current to close a clamp in a current ramp, in mA
     */
    @ConfigurationParameter(description = "final current to close clamp in a current ramp in mA",
            category = ONLINE_CLAMP_CONFIG_CATEGORY)
    private int finalCurrentToClose = 0;

    /**
     * wait time after sending initialCurrentToClose before doing current ramp
     * when closing a clamp, in milliseconds :
     */
    @ConfigurationParameter(description = "wait time for current ramp when closing a clamp, in milliseconds",
            category = ONLINE_CLAMP_CONFIG_CATEGORY)
    private int waitTimeToClose = 1000;

    /**
     * wait time after sending currentToOpen before doing current ramp when
     * opening a clamp, in milliseconds :
     */
    @ConfigurationParameter(description = "wait time for current ramp when opening a clamp, in milliseconds",
            category = ONLINE_CLAMP_CONFIG_CATEGORY)
    private int waitTimeToOpen = 1000;

    /**
     * max time for current ramp when closing a clamp, in milliseconds
     */
    @ConfigurationParameter(description = "max time for current ramp when closing a clamp, in milliseconds",
            category = ONLINE_CLAMP_CONFIG_CATEGORY)
    private int maxTimeToClose = 2000;

    /**
     * max time for current ramp when opening a clamp, in milliseconds
     */
    @ConfigurationParameter(description = "max time for current ramp when opening a clamp, in milliseconds",
            category = ONLINE_CLAMP_CONFIG_CATEGORY)
    private int maxTimeToOpen = 2000;

    /**
     * max time for current ramp when locking a clamp, in milliseconds
     */
    @ConfigurationParameter(description = "max time for current ramp when locking a clamp, in milliseconds",
            category = ONLINE_CLAMP_CONFIG_CATEGORY)
    private int maxTimeToLock = 2000;

    /**
     * max time for current ramp when unlocking a clamp, in milliseconds
     */
    @ConfigurationParameter(description = "max time for current ramp when unlocking a clamp, in milliseconds",
            category = ONLINE_CLAMP_CONFIG_CATEGORY)
    private int maxTimeToUnlock = 2000;

    /**
     * For the onlineClamp the lockStatus is OPENED, CLOSED, or LOCKED. Sensors
     * say if clamp is CLOSED or OPENED. To know if a clamp is LOCKED, we must
     * know the value of the current that was sent to the controller to close
     * the clamp. (field sentCurrent).
     */
    private FcsEnumerations.LockStatus lockStatus = LockStatus.UNKNOWN;

    private volatile boolean initialized;
    private boolean controllerConfigured = false;

    private final Condition stateUpdated = lock.newCondition();

    /* This is used when we update the latch state with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;

    /* Tools for current ramp*/
    private ScheduledFuture<?> currentRampHandle;

    /* for action SENT_CURRENT_WAIT_AND_RELEASE_BRAKE*/
    private long waitBeginTime = 0;
    private long waitDuration = 0;
    private long TEMPO_TO_RELEASE_BRAKE = 500;

    /**
     * Builds an AutochangerOnlineClampModule with a controller, 4 sensors and 2
     * maps of parameters for the controller.
     *
     * @param controller
     * @param closeSensors
     * @param openSensors
     */
    public AutochangerOnlineClamp(
            EPOSControllerWithBrake controller,
            ComplementarySensors closeSensors,
            ComplementarySensors openSensors
    ) {
        this.controller = controller;
        this.closeSensors = closeSensors;
        this.openSensors = openSensors;
    }

    /* for simulation and tests */
    @ConfigurationParameterChanger
    public void setCurrentToClamp(int currentToClamp) {
        this.currentToClamp = currentToClamp;
    }

    /* for simulation & tests */
    @ConfigurationParameterChanger
    public void setCurrentToOpen(int currentToOpen) {
        this.currentToOpen = currentToOpen;
    }

    @ConfigurationParameterChanger
    public void setInitialCurrentToClose(int initialCurrentToClose) {
        this.initialCurrentToClose = initialCurrentToClose;
    }

    @ConfigurationParameterChanger
    public void setFinalCurrentToClose(int finalCurrentToClose) {
        this.finalCurrentToClose = finalCurrentToClose;
    }

    /**
     *
     * @param sentCurrent
     */
    @ConfigurationParameterChanger
    public void setSentCurrent(int sentCurrent) {
        this.sentCurrent = sentCurrent;
    }

    /**
     *
     * @param stepHeightAbsValue
     */
    @ConfigurationParameterChanger
    public void setIncrementCurrentToClamp(int stepHeightAbsValue) {
        FcsUtils.checkPositive(stepHeightAbsValue);
        this.incrementCurrentToClamp = stepHeightAbsValue;
    }

    @ConfigurationParameterChanger
    public void setIncrementCurrentToOpen(int incrementCurrentToOpen) {
        FcsUtils.checkPositive(incrementCurrentToOpen);
        this.incrementCurrentToOpen = incrementCurrentToOpen;
    }

    @ConfigurationParameterChanger
    public void setIncrementCurrentToClose(int incrementCurrentToClose) {
        FcsUtils.checkPositive(incrementCurrentToClose);
        this.incrementCurrentToClose = incrementCurrentToClose;
    }

    @ConfigurationParameterChanger
    public void setWaitTimeToClose(int waitTimeToClose) {
        FcsUtils.checkPositive(waitTimeToClose);
        this.waitTimeToClose = waitTimeToClose;
    }

    @ConfigurationParameterChanger
    public void setWaitTimeToOpen(int waitTimeToOpen) {
        FcsUtils.checkPositive(waitTimeToOpen);
        this.waitTimeToOpen = waitTimeToOpen;
    }

    @ConfigurationParameterChanger
    public void setMaxTimeToClose(int maxTimeToClose) {
        FcsUtils.checkPositive(maxTimeToClose);
        this.maxTimeToClose = maxTimeToClose;
    }

    @ConfigurationParameterChanger
    public void setMaxTimeToOpen(int maxTimeToOpen) {
        FcsUtils.checkPositive(maxTimeToOpen);
        this.maxTimeToOpen = maxTimeToOpen;
    }

    @ConfigurationParameterChanger
    public void setMaxTimeToLock(int maxTimeToLock) {
        FcsUtils.checkPositive(maxTimeToLock);
        this.maxTimeToLock = maxTimeToLock;
    }

    @ConfigurationParameterChanger
    public void setMaxTimeToUnlock(int maxTimeToUnlock) {
        FcsUtils.checkPositive(maxTimeToUnlock);
        this.maxTimeToUnlock = maxTimeToUnlock;
    }

    public int getInitialCurrentToClose() {
        return initialCurrentToClose;
    }

    public int getFinalCurrentToClose() {
        return finalCurrentToClose;
    }

    public int getIncrementCurrentToOpen() {
        return incrementCurrentToClose;
    }

    public int getIncrementCurrentToClamp() {
        return incrementCurrentToClamp;
    }

    public int getIncrementCurrentToClose() {
        return incrementCurrentToClose;
    }

    public int getMaxTimeToClose() {
        return maxTimeToClose;
    }

    public int getWaitTimeToClose() {
        return waitTimeToClose;
    }

    public int getMaxTimeToOpen() {
        return maxTimeToOpen;
    }

    public int getMaxTimeToLock() {
        return maxTimeToLock;
    }

    public int getMaxTimeToUnlock() {
        return maxTimeToUnlock;
    }

    public int getWaitTimeToOpen() {
        return waitTimeToOpen;
    }

    /**
     * returns last value of current sent to controller
     *
     * @return
     */
    public int getSentCurrent() {
        return sentCurrent;
    }

    /**
     * Return ONLINE clamp controller
     *
     * @return
     */
    public EPOSControllerWithBrake getController() {
        return controller;
    }

    /**
     * For simulator
     *
     * @return
     */
    public int getCurrentToOpen() {
        return currentToOpen;
    }

    /**
     * For simulator
     *
     * @return
     */
    public int getCurrentToClamp() {
        return currentToClamp;
    }

    /**
     * returns closeSensors for tests
     *
     * @return
     */
    public ComplementarySensors getCloseSensors() {
        return closeSensors;
    }

    /**
     * returns openSensors for tests
     *
     * @return
     */
    public ComplementarySensors getOpenSensors() {
        return openSensors;
    }

    /**
     * Returns true if closeSensor and closeSensorC return the same value.
     * Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if lockSensor and lockSensorC retun the same value. "
            + "Doesn't read again sensors.")
    public boolean isCloseSensorsInError() {
        return closeSensors.isInError();
    }

    /**
     * Returns true if openSensor and openSensorC retun the same value. Doesn't
     * read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if unlockSensor and unlockSensorC retun the same value. "
            + "Doesn't read again sensors.")
    public boolean isOpenSensorsInError() {
        return openSensors.isInError();
    }

    /**
     * Returns true if LockStatus=ERROR, this means closeSensor or openSensor is
     * in ERROR or openSensor and closeSensor return non consistant values.
     * Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=ERROR, this means that "
            + "closeSensor or openSensor is in ERROR or that"
            + "openSensor and closeSensor return non consistant values. Doesn't read again sensors.")
    @Override
    public boolean isInError() {
        return lockStatus == FcsEnumerations.LockStatus.ERROR;
    }

    /**
     * Returns true if LockStatus=CLOSED. Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=CLOSED. Doesn't read again sensors.")
    public boolean isClosed() {
        return lockStatus == FcsEnumerations.LockStatus.CLOSED;
    }

    /**
     * Returns true if LockStatus=LOCKED. Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=LOCKED. Doesn't read again sensors.")
    public boolean isLocked() {
        return lockStatus == FcsEnumerations.LockStatus.LOCKED;
    }

    /**
     * Returns true if LockStatus=OPENED. Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=OPENED. Doesn't read again sensors.")
    public boolean isOpened() {
        return lockStatus == FcsEnumerations.LockStatus.OPENED;
    }

    /**
     * Returns true if LockStatus=INTRAVEL. Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=INTRAVEL. Doesn't read again sensors.")
    public boolean isInTravel() {
        return lockStatus == FcsEnumerations.LockStatus.INTRAVEL;
    }

    /**
     * return controller name
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns controller name.")
    @Override
    public String getControllerName() {
        return controller.getName();
    }

    /**
     * Returns true if clamp is initialized : hardware is ready (at least
     * booted) and clamp controller is initialized (parameters in the controller
     * CPU have been opened and controller has been configured).
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if clamp is initialized : controller is booted, parameters in the controller "
            + "CPU have been opened and controller is configured.")
    public boolean isInitialized() {
        return initialized;
    }

    /**
     * *** lifecycle methods *************************************************
     */
    /**
     * 
     */
    @Override
    public void build() {
        /**
         * Monitor current read on controller.
         */
        periodicTaskService.scheduleAgentPeriodicTask(
                new AgentPeriodicTask(name+CURRENT_MONITOR_TASK_NAME,this::monitorCurrent)
                        .withIsFixedRate(true)
                        .withLogLevel(Level.WARNING)
                        .withPeriod(Duration.ofSeconds(60)));
    }
    /**
     * ***********************************************************************
     */    
    
    public void increaseCurrentMonitoringSpeed() {
        periodicTaskService.setPeriodicTaskPeriod(name+CURRENT_MONITOR_TASK_NAME, Duration.ofSeconds(30));
    }
    
    public void decreaseCurrentMonitoringSpeed() {
        periodicTaskService.setPeriodicTaskPeriod(name+CURRENT_MONITOR_TASK_NAME, Duration.ofSeconds(60));
    }    

    /**
     * Return true when ONLINE clamp hardware is ready. That is when controller
     * is initialized and configured.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if controller is initialized and configured.")
    @Override
    //tested on CPPM testbench in september 2015
    public boolean myDevicesReady() {
        return mainModule.allDevicesBooted()
                && controller.isInitialized() && controllerConfigured;
    }

    /**
     * Check if it's safe to open the clamp.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the onlineClamp can be locked.")
    public void checkConditionsForOpening() {
        autochanger.checkConditionsForActioningOnlineClamps();

        if (!autochanger.isHoldingFilter()) {
            throw new RejectedCommandException(name + " can't be OPENED if autochanger is not HOLDING filter.");
        }
    }

    /**
     * Check if it's safe to close the clamp.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the onlineClamp can be locked.")
    public void checkConditionsForClosing() {
        autochanger.checkConditionsForActioningOnlineClamps();
    }

    /**
     * This method is called during INITIALIZATION phase. It configures the
     * controller for the digital inputs and outputs. It changes it to CURRENT
     * mode if it's not in this mode and it writes in the CPU of the controller
     * the values of configuration for CURRENT mode parameters.
     */
    @Override
    public void postStart() {
        FCSLOG.fine(name + " BEGIN postStart.");
        if (controller.isBooted()) {
            initializeController();
            controller.enable();
        }
        FCSLOG.fine(name + " END postStart.");
    }

    /**
     * check that controllers are correctly configured.
     */
    private void initializeController() {
        try {
            /* check that parameters on CPU are those on configuration
            */
            controller.initializeAndCheckHardware();
            this.initialized = true;

        } catch (FcsHardwareException | FailedCommandException ex) {
            this.raiseAlarm(FcsAlert.HARDWARE_ERROR, name + " couldn't initialize controller", 
                    controller.getName(), ex);
        }
    }

    /**
     * Enables controller, changes mode into CURRENT, Reads parameters for mode
     * CURRENT on controller CPU. If a parameter has a different value than in
//     * configuration, throws an exception.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read in the CPU of the controller parameters for mode CURRENT."
            + "If a parameter has a different value than in configuration, "
            + "throws an exception.")
    public void enableAndCheckControllerBeforeAction() {
        controller.switchOnEnableOperation();
        controller.changeMode(EposMode.CURRENT);
        //TODO refactorize that.
        try {
            //over engineering ? do not this tests in Normal mode for every action
            //do it when going from engineering mode to normal mode.
            controller.checkParameters(EposMode.CURRENT);
            controller.checkFault();
        } catch (Exception ex) {
            String msg = name + " error in parameters ";
            FCSLOG.error(msg + ex);
            throw new FcsHardwareException(msg, ex);
        }
        //TODO to be done in checkPArameters
        if (!controller.isParametersOK()) {
            String msg = name + " Some parameter values are not"
                    + " the same in CPU and configuration system.";
            FCSLOG.error(msg);
            throw new FcsHardwareException(msg);
        }
    }

    /**
     * Configure ONLINE clamp controller.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Configure controller.")
    //tested on CPPM testbench in september 2015
    public void configureController() {
        try {
            controllerConfigured = false;
            this.configureDigitalInputOfOnlineClamps();
            this.configureDigitalOutputOfOnlineClamps();
            controllerConfigured = true;
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(name, ex);
        }
    }
    
    /**
     * read actual current on controller, update field current and publish it on STATUS bus.
     */
    @Override
    public void updateCurrent() {
        this.current = controller.readCurrent();
    }

    /**
     * reads sensors, computes and state and raises an ALERT if sensors are in
     * error.
     *
     */
    public void updateStateAndCheckSensors() {
        autochanger.updateStateWithSensors();
        checkSensors(AC_SENSOR_ERROR, name);
    }

    /**
     * This methods updates lockStatus from the values return by the sensors.
     * This values are given in an array of hexa values as arguments of the
     * method.
     *
     * @param hexaValues
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Update state in reading sensors.")
    public void updateStateWithSensors(int[] hexaValues) {
        lock.lock();
        try {
            updatingState = true;

            this.closeSensors.updateValue(hexaValues);
            this.openSensors.updateValue(hexaValues);

            boolean closed = this.closeSensors.isOn();
            boolean opened = this.openSensors.isOn();
            boolean someSensorsInError = closeSensors.isInError() || openSensors.isInError();
            boolean inError = someSensorsInError || (closed && opened);

            if (inError) {
                lockStatus = FcsEnumerations.LockStatus.ERROR;
            } else if (closed) {
                lockStatus = FcsEnumerations.LockStatus.CLOSED;
            } else if (opened) {
                lockStatus = FcsEnumerations.LockStatus.OPENED;
            } else {
                lockStatus = FcsEnumerations.LockStatus.INTRAVEL;
            }
            /* if the clamp is closed and a currentToClamp has been previously sent to controller
             then clamp is LOCKED*/
            if (closed && this.sentCurrent == this.currentToClamp) {
                lockStatus = FcsEnumerations.LockStatus.LOCKED;
            }

        } finally {

            updatingState = false;
            stateUpdated.signalAll();
            lock.unlock();
            this.publishData();
        }

    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        autochanger.updateStateWithSensors();
        /* for action SEND_CURRENT_AND_WAIT*/
        if (this.currentAction == MobileItemAction.SEND_CURRENT_AND_WAIT) {
            waitDuration = System.currentTimeMillis() - this.waitBeginTime;
        }
    }

    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        if (action == MobileItemAction.CLOSE_ONLINECLAMP
                || action == MobileItemAction.UNLOCK_ONLINECLAMP) {
            return sentCurrent == finalCurrentToClose;

        } else if (action == MobileItemAction.OPEN_ONLINECLAMP) {
            return isOpened();

        } else if (action == MobileItemAction.LOCK_ONLINECLAMP) {
            return isLocked();

        } else if (action == MobileItemAction.SEND_CURRENT_AND_WAIT) {
            return waitDuration > TEMPO_TO_RELEASE_BRAKE;

        } else {
            throw new IllegalArgumentException(name + " invalid action for ONLINE clamp:" + action);
        }
    }

    /**
     * Closes the ONLINE clamp.
     */
    /*Tested in december 2016*/
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Closes the ONLINE clamp.", timeout = TIMEOUT_FOR_CLOSING)
    public void close() {
        updateStateAndCheckSensors();
        if (this.isOpened() || isInTravel()) {
            checkConditionsForClosing();
            this.executeAction(MobileItemAction.CLOSE_ONLINECLAMP, TIMEOUT_FOR_CLOSING);

        } else if (this.isClosed()) {
            FCSLOG.info(name + " is already CLOSED. Nothing to do.");

        } else {
            throw new RejectedCommandException(name + " has to be OPENED before a close action.");
        }
    }

    /**
     * close softly with a ramp of current from initialCurrent to finalCurrent
     * with an increment incrementCurrent, after first step wait during
     * waitTime. for tests in september 2017
     *
     * @param initialCurrent
     * @param finalCurrent
     * @param incrementCurrent
     * @param waitTime
     * @param maxTime
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Closes the ONLINE clamp softly with new parameters and saves the new parameters.",
            timeout = TIMEOUT_FOR_CLOSING)
    public void closeSoftly(int initialCurrent, int finalCurrent, int incrementCurrent,
            int waitTime, int maxTime) {
        this.enableAndCheckControllerBeforeAction();
        autochanger.checkConditionsForActioningOnlineClamps();
        checkArguments(initialCurrent, finalCurrent, incrementCurrent, maxTime, "closeSoftly");
        this.setInitialCurrentToClose(initialCurrent);
        this.setFinalCurrentToClose(finalCurrent);
        this.setIncrementCurrentToClose(incrementCurrent);
        this.setWaitTimeToClose(waitTime);
        this.setMaxTimeToClose(maxTime);
        controller.doReleaseBrake();
        this.sendCurrentToControllerAndSaveValue(initialCurrent);
        executeCurrentRamp(initialCurrent, finalCurrent, incrementCurrent, maxTime, waitTime);

        s.getScheduler().schedule(() -> {
            controller.activateBrake();
            controller.writeCurrent((short)0);
        }, 300, TimeUnit.MILLISECONDS);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Open the ONLINE clamp softly with new parameters. Does not save the parameters."
            + " To save parameters use change command.", timeout = TIMEOUT_FOR_OPENING)
    public void openSoftly(int initialCurrent, int finalCurrent, int incrementCurrent,
            int waitTime, int maxTime) {
        this.enableAndCheckControllerBeforeAction();
        autochanger.checkConditionsForActioningOnlineClamps();
        checkArguments(initialCurrent, finalCurrent, incrementCurrent, maxTime, "openSoftly");
        sendCurrentToControllerAndSaveValue(finalCurrent);
        controller.doReleaseBrake();
        executeCurrentRamp(initialCurrent, finalCurrent, incrementCurrent, maxTime, 0);
        /* we wait waitTime after sending finalCurrentToOpen, before activate brake and send current 0*/
        s.getScheduler().schedule(() -> {
            controller.activateBrake();
            controller.writeCurrent((short)0);
        }, waitTime, TimeUnit.MILLISECONDS);
    }

    private static int computePeriod(int initialCurrent, int finalCurrent, int incrementCurrent, int maxTime) {
        int nbStep = Math.abs((finalCurrent - initialCurrent) / incrementCurrent);
        return maxTime / nbStep;
    }

    private void checkArguments(int initialCurrent, int finalCurrent, int incrementCurrent, int maxTime, String actionName) {
        checkCurrentParametersValidity(initialCurrent, finalCurrent, incrementCurrent);
        int nbStep = Math.abs((finalCurrent - initialCurrent) / incrementCurrent);
        if (maxTime % nbStep != 0) {
            throw new IllegalArgumentException(maxTime + " :maxTime has to be a multiple of nbStep. Action= " + actionName
                    + " Clamp= " + name
                    + " nbStep= " + nbStep + " = |(finalCurrent - initialCurrent) / incrementCurrent|");
        }
        int period = computePeriod(initialCurrent, finalCurrent, incrementCurrent, maxTime);
        if (period < 200) {
            int newIncr = Math.abs((200 * (finalCurrent - initialCurrent))/maxTime);
            int newMaxTime = Math.abs(200 * (finalCurrent - initialCurrent)/incrementCurrent);
            throw new IllegalArgumentException(incrementCurrent + " bad value for incrementCurrent => period computed too low= " + period
                    + " incrementCurrent should be > " + newIncr + " or maxTime > " + newMaxTime);
        }
        
    }

    private static void checkCurrentParametersValidity(int initialCurrent, int finalCurrent, int incrementCurrent) {
        if ((finalCurrent - initialCurrent) % incrementCurrent != 0) {
            throw new IllegalArgumentException("finalCurrent=" + finalCurrent + " initialCurrent=" + initialCurrent
                    + " incrementCurrent=" + incrementCurrent
                    + " (finalCurrent - initialCurrent) has to be a "
                    + "multiple of incrementCurrent");
        }
    }

    /**
     * Opens the ONLINE clamp.
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Opens the ONLINE clamp.", timeout = TIMEOUT_FOR_OPENING)
    public void open() {
        updateStateAndCheckSensors();
        if (this.isClosed() || this.isInTravel()) {
            checkConditionsForOpening();
            this.executeAction(MobileItemAction.OPEN_ONLINECLAMP, TIMEOUT_FOR_OPENING);

        } else if (this.isOpened()) {
            FCSLOG.info(name + " is already OPENED. Nothing to do.");

        } else {
            throw new RejectedCommandException(name + " has to be CLOSED before an open action.");
        }
    }

    /**
     * Locks the ONLINE clamp : sends currentToClamp to the controller with a
     * ramp of current from currentToClose to currentToRamp. The clamp has to be
     * CLOSED before this action. At the end of this action, the clamp is
     * LOCKED.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Locks the ONLINE clamp : sends currentToClamp to the controller with a ramp of current"
            + "from currentToClose to currentToRamp.", timeout = TIMEOUT_FOR_CLOSING)
    public void lock() {
        //TODO check conditions for locking
        updateStateAndCheckSensors();
        if (this.isClosed() || this.isInTravel()) {
            this.executeAction(MobileItemAction.LOCK_ONLINECLAMP, TIMEOUT_FOR_CLOSING);

        } else if (this.isLocked()) {
            FCSLOG.info(name + " is already LOCKED. Nothing to do.");

        } else {
            throw new RejectedCommandException(name + " has to be CLOSED before lock action.");
        }
    }

    /**
     * Unlocks the ONLINE clamp : sends currentToClamp to the controller with a
     * ramp of current from currentToClamp to currentToClose. The clamp has to
     * be LOCKED before this action. At the end of this action, the clamp is
     * CLOSED. The goal if this action is to slow down the pressure exerted by
     * the clamp on the filter. If we decrease too quickly the pressure the
     * clamp could be ejected out by elastic return.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Unlocks the ONLINE clamp : sends currentToClamp to the controller "
            + " with a ramp of current from currentToClamp to currentToClose.",
            timeout = TIMEOUT_FOR_OPENING)

    public void unlock() {
        updateStateAndCheckSensors();
        //TODO check conditions for unlocking
        if (this.isLocked()) {
            this.executeAction(MobileItemAction.UNLOCK_ONLINECLAMP, TIMEOUT_FOR_CLOSING);

        } else if (this.isClosed()) {
            FCSLOG.info(name + " is already CLOSED. Nothing to do.");

        } else {
            throw new RejectedCommandException(name + " has to be LOCKED before unlock action.");
        }
    }

    /**
     * Start action of open or close or lock or unlock.
     *
     * @param action
     * @throws FcsHardwareException
     */
    @Override
    public void startAction(FcsEnumerations.MobileItemAction action) {
        this.enableAndCheckControllerBeforeAction();
        autochanger.checkConditionsForActioningOnlineClamps();
        this.increaseCurrentMonitoringSpeed();
        
        if (action == MobileItemAction.CLOSE_ONLINECLAMP) {
            checkArguments(initialCurrentToClose, finalCurrentToClose, incrementCurrentToClose, maxTimeToClose, action.name());
            controller.doReleaseBrake();
            this.sendCurrentToControllerAndSaveValue(initialCurrentToClose);
            executeCurrentRamp(initialCurrentToClose, finalCurrentToClose, incrementCurrentToClose, maxTimeToClose, waitTimeToClose);

        } else if (action == MobileItemAction.OPEN_ONLINECLAMP) {
            checkArguments(finalCurrentToClose, currentToOpen, incrementCurrentToOpen, maxTimeToOpen, action.name());
            sendCurrentToControllerAndSaveValue(finalCurrentToClose);
            FcsUtils.sleep(500, name);
            controller.doReleaseBrake();
            this.executeCurrentRamp(finalCurrentToClose, currentToOpen, incrementCurrentToOpen, maxTimeToOpen, 0);

        } else if (action == MobileItemAction.LOCK_ONLINECLAMP) {
            checkArguments(this.finalCurrentToClose, this.currentToClamp, incrementCurrentToClamp, maxTimeToLock, action.name());
            sendCurrentToControllerAndSaveValue(finalCurrentToClose);
            FcsUtils.sleep(500, name);
            controller.doReleaseBrake();
            /* write a ramp from  finalCurrentToClose to currentToClamp within maxTimeToLock*/
            this.executeCurrentRamp(this.finalCurrentToClose, this.currentToClamp, incrementCurrentToClamp, maxTimeToLock, 0);

        } else if (action == MobileItemAction.UNLOCK_ONLINECLAMP) {
            /* the clamp is LOCKED so we send currentToClamp before releasing brake to avoid an elastic
             return of the clamp */
            checkArguments(this.currentToClamp, this.finalCurrentToClose, incrementCurrentToClamp, maxTimeToUnlock, action.name());
            sendCurrentToControllerAndSaveValue(this.currentToClamp);
            FcsUtils.sleep(500, name);
            controller.doReleaseBrake();
            /* write a ramp from  currentToClamp to finalCurrentToClose within maxTimeToUnlock*/
            this.executeCurrentRamp(this.currentToClamp, this.finalCurrentToClose, incrementCurrentToClamp, maxTimeToUnlock, 0);

        } else {
            throw new IllegalArgumentException(name + " invalid action for ONLINE clamp:" + action);
        }
    }

    /**
     * Writes current to controller and saves the current value in the
     * ConfigurationParameter sentCurrent.
     *
     * @param current to send to controller
     */
    @Command(type = Command.CommandType.ACTION, description = "sent current to control and save this value")
    public void sendCurrentToControllerAndSaveValue(int current) {
        FCSLOG.finest(name + " current to write to controller " + getControllerName() + ":" + current);
        controller.writeCurrent((short)current);
        s.getConfigurationService().getComponentConfigurationEnvironment(name).change("sentCurrent", current);
        autochanger.saveCurrentSentToOnlineClampsControllers();
        this.publishData();
    }

    /**
     * What to do to abort an action.
     *
     * @param action
     * @param delay
     * @throws FcsHardwareException
     */
    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay) {
        controller.activateBrake();
        controller.writeCurrent((short)0);
        this.decreaseCurrentMonitoringSpeed();
    }

    /**
     * What to be done after the action is completed.
     *
     * @param action
     * @throws FcsHardwareException
     */
    @Override
    public void postAction(FcsEnumerations.MobileItemAction action) {
        controller.activateBrake();
        controller.writeCurrent((short)0);
        updateStateAndCheckSensors();
        if (action == MobileItemAction.CLOSE_ONLINECLAMP) {
            if (isClosed()) {
                FCSLOG.info(name + " is CLOSED");
            } else {
                throw new FailedCommandException(name + " is not CLOSED after close command.");
            }
        } else if (action == MobileItemAction.UNLOCK_ONLINECLAMP) {
            if (isClosed()) {
                FCSLOG.info(name + " has been UNLOCKED and is now CLOSED");
            } else {
                throw new FailedCommandException(name + " is not CLOSED after unlock command.");
            }
        }
    }

    /**
     * This method is used to configure the controllers of the autochanger
     * online clamps. ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ Digital
     * Inputs activated for OnlineClamps controllers are:
     * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ QuickStop Input=01
     * Mask=Enabled Polarity=HighActive HE_Sensor General_A Input=02
     * Mask=Enabled Polarity=HighActive HE_Sensor General_B Input=03
     * Mask=Enabled Polarity=HighActive Check Brake Status
     * ******************************************** 1 - Configure the Digital
     * Input Functionnality index 0x2070 sindex input_ID Functionnality tab
     * Value = 15 General Purpose A Value = 14 General Purpose B Value = 13
     * General Purpose C Value = 12 General Purpose D Value = 11 General Purpose
     * E Value = 10 General Purpose F Value = 9 General Purpose G Value = 8
     * General Purpose H Value = 7 General Purpose I Value = 6 General Purpose J
     * Value = 5 QuickStop Value = 4 Drive enable Value = 3 Position marker
     * Value = 2 Home switch Value = 1 Positive Limit switch Value = 0 Negative
     * Limit switch
     * *************************************************************** 2 -
     * Configure the Digital Input Polarity index 0x2071 subindex 0x03 Value = 0
     * High Active Value = 1 Low Active
     * **************************************************************** 3 -
     * Configure the Digital Input Mask index 0x2071 subindex 0x02 Value = 0
     * Functionnality state will not be displayed Value = 1 Functionnality state
     * will be displayed Digital Input Functionnality tab value Bit_15 General
     * Purpose A 1 Bit_14 General Purpose B 1 Bit_13 General Purpose C 0 Bit_12
     * General Purpose D 0 Bit_11 General Purpose E 0 Bit_10 General Purpose F 0
     * Bit_9 General Purpose G 0 Bit_8 General Purpose H 0 Bit_7 General Purpose
     * I 0 Bit_6 General Purpose J 0 Bit_5 QuickStop 1 Bit_4 Drive enable 0
     * Bit_3 Position marker 0 Bit_2 Home switch 0 Bit_1 Positive Limit switch 0
     * Bit_0 Negative Limit switch 0
     * ********************************************************************* 4 -
     * Configure the Digital Input Execution Mask index 0x2071 subindex 0x04
     * Digital Input Excecution Mask Value Bit_15 to Bit_6 reserved 0 Bit_5
     * QuickStop 1 Bit_4 Drive enable 1 Bit_3 Position marker 0 Bit_2 Home
     * switch 0 Bit_1 Positive Limit switch 0 Bit_0 Negative Limit switch 0
     * **********************************************************************
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "To configure the online clamps controllers.")
    public void configureDigitalInputOfOnlineClamps() {
        //1-Configure Digital Input Fonctionnality
        //QuickStop (2070,01)
        controller.writeParameter(EPOSEnumerations.Parameter.ConfigurationOfDigitalInput1, 0x5);

        //2-Configure the Digital Input Polarity (2071,03)
        controller.writeParameter(EPOSEnumerations.Parameter.DigitalInputFonctionnalityPolarity, 0);

        //3-Configure the Digital Input Mask 8C20:1000110000100000 (2071,02)
        controller.writeParameter(EPOSEnumerations.Parameter.DigitalInputFonctionnalityMask, 0x8C20);

        //4 - Configure the Digital Input Execution Mask 30:0011 0000 (2071,04)
        controller.writeParameter(EPOSEnumerations.Parameter.DigitalInputFonctionnalityExecutionMask, 0x30);
    }

    /**
     * msg = 'Set DigIn_1 Quick Stop,wsdo,%s,2070,01,02,0005\n' % (str(cobID))
     * self.com_hal.executeOperation(client,msg) msg = 'Set DigIn_2
     * General_A,wsdo,%s,2070,02,02,8000\n' % (str(cobID))
     * self.com_hal.executeOperation(client,msg) msg = 'Set DigIn_3
     * Genral_B,wsdo,%s,2070,02,02,4000\n' % (str(cobID))
     */

    /**
     * This method is used to configure the autochanger onlineClamps EPOS
     * controllers. Digital Outputs activated for OnlineClamps controller are:
     * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ Ready/Fault
     * Output=01 Mask=Enabled Polarity=HighActive General_A Output=0
     * Mask=Enabled Polarity=HighActive
     * ******************************************* 1 - Configure the Digital
     * Output Functionnality index 0x2079 subindex input_ID Functionnality tab :
     * Value = 15 General Purpose Out_A Value = 14 General Purpose Out_B Value =
     * 13 General Purpose Out_C Value = 12 General Purpose Out_D Value = 11
     * General Purpose Out_E Value = 10..8 not used Value = 7..3 reserved Value
     * = 2 Holding Brake Value = 1 Position compare Value = 0 Ready/Fault
     * ********************************************************************************
     * 2 - Configure the Digital Output Functionnality Mask index 0x2078
     * subindex 0x02 Value = 0 functionnality not activatd Value = 1
     * functionnality activated
     * ********************************************************************************
     * Digital Output Functionnality Mask Value Bit_15 General Purpose Out_A 1
     * Bit_14 General Purpose Out_B 0 Bit_13 General Purpose Out_C 0 Bit_12
     * General Purpose Out_D 0 Bit_11 General Purpose Out_E 0 Bit_10 General
     * Purpose Out_F 0 Bit_9 General Purpose Out_G 0 Bit_8 General Purpose Out_H
     * 0 Bit_7..3 reserved 0 Bit_2 Holding Brake 0 (read only) Bit_1 Position
     * Compare (read only) Bit_0 Ready/Fault (read only)
     * ********************************************************************************
     * 3 - Configure the Digital Output Functionnality Polarity index 0x2078
     * subindex 0x03 Value = 0 associated output not change => HighActive Value
     * = 1 associated output inverted => LowActive
     * ********************************************************************************
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "To configure the online clamps controllers.")
    public void configureDigitalOutputOfOnlineClamps() {
        //1 - Configure the Digital Output Functionnality
        //Ready/Fault Active on Device Ready / Inactive on Fault
        controller.writeParameter(EPOSEnumerations.Parameter.ConfigurationOfDigitalOutput1, 0);
        //General_A
        controller.writeParameter(EPOSEnumerations.Parameter.ConfigurationOfDigitalOutput4, 0xF); //pas bon

        //2 - Configure the Digital Output Functionnality Mask 8001=1000 0000 0000 0001 
        controller.writeParameter(EPOSEnumerations.Parameter.DigitalOutputFonctionnalityMask, 0x8001);

        //3 - Configure the Digital Output Functionnality Polarity 8001=1000 0000 0000 0001 
        controller.writeParameter(EPOSEnumerations.Parameter.DigitalOutputFonctionnalityPolarity, 0);
    }

    /**
     * Creates and returns the object to be published on the STATUS bus.
     *
     * @return
     */
    public StatusDataPublishedByAutochangerOnlineClamp createStatusDataPublishedByOnlineClamp() {
        StatusDataPublishedByAutochangerOnlineClamp status = new StatusDataPublishedByAutochangerOnlineClamp();
        status.setName(name);
        status.setLockSensorValue(closeSensors.isOn());
        status.setUnlockSensorValue(openSensors.isOn());
        status.setLockStatus(lockStatus);
        status.setLockSensorInError(closeSensors.isInError());
        status.setUnlockSensorInError(openSensors.isInError());
        status.setInError(lockStatus == FcsEnumerations.LockStatus.ERROR);
        status.setControllerBooted(controller.isBooted());
        status.setControllerInFault(controller.isInError());
        status.setSentCurrent(sentCurrent);
        status.setCurrent(current);
        return status;
    }
    
    @Override
    public void publishData() {
        s.publishSubsystemDataOnStatusBus(
                new KeyValueData(name, createStatusDataPublishedByOnlineClamp()));
    }

    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay) {
        controller.quickStop();
    }

    /**
     * **********************************************************
     * Methods to write current to the controller with a ramp
     * /************************************************************
     */
    
    /**
     * Cancel current ramp
    */
    private void cancelCurrentRamp() {
        this.currentRampHandle.cancel(true);
        FCSLOG.debug(" => current ramp ended");
    }

    /**
     * This method waits until the newCurrentValue ramp is completed.
     */
    private void waitForEndOfCurrentRamp(int timeout) {
        FCSLOG.info(name + " BEGIN WAITING FOR END OF CURRENT RAMP");
        try {
            currentRampHandle.get(timeout, TimeUnit.SECONDS);
        } catch (CancellationException ex) {
            if (currentRampHandle.isDone()) {
                FCSLOG.info(name + " CURRENT RAMP ENDED");
            } else {
                FCSLOG.error(name + " during action " + this.currentAction 
                        + " waiting was cancelled before END of CURRENT RAMP "+ ex);
            }
        } catch (TimeoutException ex) {
            FCSLOG.error(name + " CURRENT RAMP IS DURING TOO LONG - timeout=" + timeout, ex);
            cancelCurrentRamp();

        } catch (InterruptedException ex) {
            FCSLOG.error(name + " interrupted while waiting end of current ramp during action=" + currentAction, ex);
            throw new FcsHardwareException(name + " interrupted while waiting end of current ramp during action=" + currentAction, ex);

        } catch (ExecutionException ex) {
            FCSLOG.error(name + " error during action=" + currentAction, ex);
            throw new FcsHardwareException(name + " error during action=" + currentAction, ex);

        } finally {
            FCSLOG.info(name + " STOP WAITING FOR END OF CURRENT RAMP");
        }
    }

    /**
     * Sends current value to controller with a current ramp from initialValue
     * to finalValue. Uses a scheduler to schedule the task.
     *
     * @param initialValue first value to send to controller
     * @param finalValue last value to send to controller
     * @param nbStep number of steps between initialValue to finalValue.
     * @param period period of time between 2 instructions writeCurrent.
     */
    private void writeCurrentRamp(final int initialValue, final int finalValue, final int increment,
            final long period, final int waitTime) {
        FCSLOG.debug("############################");
        FCSLOG.debug(name + "writeCurrentRamp");
        FCSLOG.debug("initialValue=" + initialValue);
        FCSLOG.debug("finalValue=" + finalValue);
        FCSLOG.debug("|increment|=" + increment);
        FCSLOG.debug("period=" + period);
        FCSLOG.debug("waitTime=" + waitTime);
        FCSLOG.debug("############################");

        final Runnable currentRamp = new Runnable() {
            private final int adjustedStepHeight = FcsUtils.getSignedStepHeight(initialValue, finalValue,
                    increment);
            private int newCurrentValue = initialValue + adjustedStepHeight;

            public boolean finalValueReached() {
                boolean reachedUp = adjustedStepHeight > 0 && newCurrentValue > finalValue;
                boolean reachedDown = adjustedStepHeight < 0 && newCurrentValue < finalValue;
                return reachedUp || reachedDown;
            }

            @Override
            public void run() {
                if (finalValueReached()) {
                    cancelCurrentRamp();
                } else {
                    sendCurrentToControllerAndSaveValue(newCurrentValue);
                    newCurrentValue = newCurrentValue + adjustedStepHeight;
                }
            }
        };
        this.currentRampHandle = scheduler.scheduleAtFixedRate(currentRamp, waitTime, period, TimeUnit.MILLISECONDS);
    }

    /**
     * Sends current to controller with a ramp : from initialCurrent to
     * finalCurrent in a given number of steps (nbStep) and waits until the
     * final value is reached. It uses method writeCurrentRamp which uses a
     * scheduler.
     *
     * @param initialCurrent first value to be sent to controller
     * @param finalCurrent last value to send to controller
     * @param incrementCurrent
     * @param maxTime
     * @param waitTime time to wait after sending initialCurrent to controller.
     */
    public void executeCurrentRamp(int initialCurrent, int finalCurrent, int incrementCurrent,
            int maxTime, int waitTime) {
        int period = computePeriod(initialCurrent, finalCurrent, incrementCurrent, maxTime);
        writeCurrentRamp(initialCurrent, finalCurrent, incrementCurrent, period, waitTime);
        waitForEndOfCurrentRamp(TIMEOUT_FOR_OPENING);
    }

    /**
     * **************************************************************
     * end of Methods to write wurrent to the controller with a ramp
     * /****************************************************************
     */
}
