
package org.lsst.ccs.subsystems.fcs.utils;

import static java.lang.Math.abs;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations;

/**
 * This class gathers static final variables and static methods. 
 * 
 * By example, to creates Objects for GUI and Status Bus.
 * @author virieux
 */
public final class FcsUtils {


    
    /**
     * The maximum value which a number coded on 2 bytes can reach.
     */
    public static final int MAX_VALUE_2BYTES = 65535;
    
    /**
     * The maximum number of CANopen devices which can be on a CANbus.
     * Also the maximum CANopen node ID a device can have.
     */
    public static final int MAX_NODE_ID = 127;
    
    /**
     * A value for all fcs Modules tickmillis.
     */
    public static final int TICKMILLIS = 3000;
    
    private static final String SOCKET_NAME= "socket";
    
    private static final String[] CAN_OPEN_COMMANDS 
            = new String[] {"sync", "scan", "rsdo", "wsdo", "info", "quit", "srtr", "reset"};
    
    
    /**
     * 
     * @param name 
     */
    public static void checkSocketName(String name) {
        String errorMsg = ": invalid socket name - Socket name must be \""+SOCKET_NAME+'X' + "\" where 0 " + '<' + "= X " + '<' + "= 5";
        
        /*Socket Name can be an empty String or */
        /*Socket Name can be "AC" if filter is on AUTOCHANGER.*/
        if (name.isEmpty() || name.contentEquals("AC")) return;
        
        /*Socket Name must start with "socket".*/
        if (!name.startsWith(SOCKET_NAME)) throw new IllegalArgumentException(name + errorMsg);
        
        /*Socket Name must end with 1, 2, 3, 4 or 5.*/
        String end = name.substring(6);
        try {
            int ix = Integer.parseInt(end);
            if (ix < 1 || ix > 5) throw new IllegalArgumentException(name + errorMsg);
            
        } catch (NumberFormatException ex) {
            throw new IllegalArgumentException(name + errorMsg,ex);
        }        
    }


    /**
     *
     * @param bin
     * @return
     */
    public static String binaryToHex(String bin) {
        int binaryToInt = Integer.parseInt(bin, 2);
        return Integer.toHexString(binaryToInt);
    }

    /**
     * The String argument represents an integer in hexa. This methods returns a
     * string representation of this integer argument as an unsigned integer in
     * base 2.
     *
     * @param Hex
     * @return the binary form of this integer with 8 digits examples :
     * HexToBinary(4) returns 00000100 HexToBinary(AA) returns 10101010
     * HexToBinary(FF) returns 11111111
     */
    public static String hexToBinary(String Hex) {
        int integer = Integer.parseInt(Hex, 16);
        String bin = Integer.toBinaryString(integer);
        return String.format("%08d", Integer.parseInt(bin));
    }

    /**
     * The String argument represents an integer in hexa. This methods returns a
     * string representation of this integer argument as an unsigned integer in
     * base 2.
     *
     * @param twoBytesInHexa
     * @return the binary form of this integer with 16 digits examples :
     * twoBytesInHexaToBinary("4") returns 0000000000000100
     * twoBytesInHexaToBinary("AA") returns 0000000010101010
     * twoBytesInHexaToBinary("FFFF") returns 1111111111111111
     */
    public static String twoBytesInHexaToBinary(String twoBytesInHexa) {
        int integer = Integer.parseInt(twoBytesInHexa, 16);
        String bin = Integer.toBinaryString(integer);
        return String.format("%016d", Long.parseLong(bin));
    }


    /**
     * Computes a new hexa value given as argument in replacing bit which is at position 
     * given as argument by 0.
     *
     * @param hexaVal value to transform
     * @param bit bit numero to be replaced
     * @return new hexa value
     * @throws IllegalArgumentException if bit <0 or bit >15
     */
    public static String force2zero(String hexaVal, int bit) {
        if (bit < 0 || bit > 15) {
            throw new IllegalArgumentException(String.format("bad value for bit=%s - must be : >0 and <15", bit));
        }
        int mask = 0b1111111111111111;
        mask = mask - (int) Math.pow(2, bit);
        int val = Integer.parseInt(hexaVal,16);
        return Integer.toHexString(val & mask);
    }

    /**
     * Transforms a numericonfigonfig value given in hexa to binary and replaconfigonfige the digit in position bit given as argument by 1.
     *
     * @param hexaVal
     * @param bit
     * @return
     */
    public static String force2one(String hexaVal, int bit) {
        if (bit < 0 || bit > 15) {
            throw new IllegalArgumentException(String.format("bad value for bit=%s - must be : >0 and <15", bit));
        }
        int mask = (int) Math.pow(2, bit);
        int intVal = Integer.parseInt(hexaVal,16);
        return Integer.toHexString(intVal | mask);
    }

    /**
     * Transform an integer given in its decimal format to its binary
     * format, reverse it and returns the reversed binary at an array of int.
     * format.
     *
     * @param decimal
     * @return an array of int
     */
    public static int[] toReverseBinary(int decimal) {
        int[] tab = new int[16];

        for (int i = 0; i < 16; i++) {
            int bit = (decimal >> i) & 0x1;
            tab[i] = bit;
        }
        return tab;
    }

    
    /**
     * build a Can Open wsdo Command that can understand the C wrapper. 
     * exemple : wsdo,2,6411,01,2,3000
     * @param nodeID
     * @param index
     * @param subindex
     * @param size
     * @param data
     * @return 
     */
    public static String buildWsdoCommand(String nodeID, String index, String subindex, String size, String data) {
        char sep = ',';
        StringBuilder sb = new StringBuilder("wsdo").append(sep);
        sb.append(nodeID).append(sep);
        sb.append(index).append(sep);
        sb.append(subindex).append(sep);
        sb.append(size).append(sep);
        sb.append(data);
        return sb.toString();
    }

    /**
     * build a Can Open rsdo Command that can understand the C wrapper. exemple
     * : rsdo,1,1018,0
     * @param nodeID
     * @param index
     * @param subindex
     * @return 
     */
    public static String buildRsdoCommand(String nodeID, String index, String subindex) {
        char sep = ',';
        StringBuilder sb = new StringBuilder("rsdo").append(sep);
        sb.append(nodeID).append(sep);
        sb.append(index).append(sep);
        sb.append(subindex);
        return sb.toString();
    }
    
    /**
     * returns true if the command given as argument is an available CANopen command.
     * @param command
     * @return 
     */
    public static boolean isValidCommandWord(String command) {
        for(String s: CAN_OPEN_COMMANDS){
            if(s.equals(command))
                return true;
        }
        return false;
    }
    
    /**
     * @param command 
     * @throws IllegalArgumentException if command is not a valid command.
     */
    public static void checkCommand(String command) {
        if (command == null) throw new IllegalArgumentException(" null command");
        String[] words = command.split(",");
        if (words.length == 0 || !isValidCommandWord(words[0]))  {
            throw new IllegalArgumentException(command+" invalid command");
        }
    }




    /**
     * Change an int represented by an UNSIGNED16 to an INTEGER16.
     * 
     * UNSIGNED16=0 =>INTEGER16=0
     * UNSIGNED16=2356 =>INTEGER16=2356
     * UNSIGNED16=65342 =>INTEGER16=-193
     * UNSIGNED16=32767 =>INTEGER16=32767
     * UNSIGNED16=32768 =>INTEGER16=-32767
     * UNSIGNED16=65535 =>INTEGER16=0
     * UNSIGNED16=-400 =>INTEGER16=-400
     *
     * @param unsigned16
     * @return
     */
    public static int convertToInteger16(int unsigned16) {
        if (unsigned16 < 32768) {
            return unsigned16;
        } else {
            return unsigned16 - 65535;
        }
    }

    /**
     * Displays the list of parameters that we have to define for a given mode.
     *
     * @param modeInString
     * @return
     */
    public static String displayListParameters(String modeInString) {
        StringBuilder sb = new StringBuilder("List of parameters for mode: ");
        sb.append(modeInString);
        sb.append("\n");
        EPOSEnumerations.Parameter[] params = EPOSEnumerations.EposMode.valueOf(modeInString).getParameters();
        for (EPOSEnumerations.Parameter param : params) {
            sb.append(param.display());
            sb.append("\n");
        }
        return sb.toString();
    }

    /**
     * Displays the list of parameters.
     *
     * @return
     */
    public static String displayListParameters() {
        StringBuilder sb = new StringBuilder("List of parameters : ");
        sb.append("\n");
        EPOSEnumerations.Parameter[] params = EPOSEnumerations.Parameter.values();
        for (EPOSEnumerations.Parameter param : params) {
            sb.append(param.display());
            sb.append("\n");
        }
        return sb.toString();
    }
    
    /**
     * Add 2 angles : an angle is a int which value is between -360 and +360
     *
     * @param angle1
     * @param angle2
     * @return result of the angles addition. Always >= 0 and always < 360
     */
    public static int addAngle(int angle1, int angle2) {

        int angle = angle1 + angle2;
        int result = angle % 360;
        if (result < 0) {
            result += 360;
        }
        return result;
    }
    
    /**
     * This method computes the shortest way for a rotation angle given as argument and returns it.
     * @param angle
     * @return
     */
    public static int computeRotationShortestWay(int angle) {
        if (abs(angle) > 360) {
            throw new IllegalArgumentException(angle + " invalid value");
        } else if (abs(angle) < 180) {
            return angle;
        } else if (angle > 0) {
            return angle - 360;
        } else {
            return angle + 360;
        } 
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        System.out.println(FcsUtils.displayListParameters("HOMING"));
        System.out.println(FcsUtils.displayListParameters("CURRENT"));
        System.out.println(FcsUtils.displayListParameters("PROFILE_POSITION"));
        System.out.println(FcsUtils.displayListParameters());
    }





}
