
package org.lsst.ccs.subsystems.fcs.simulation;

import java.util.Map;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenEPOS;

/**
 * This class is to simulate an EPOS Controller.
 *
 * @author virieux
 */
public class SimuEPOSControllerModule extends CanOpenEPOS {

    private boolean enabled;
    protected int targetPosition;
    protected int position;
    protected int ssiPosition;
    protected int targetCurrent;
    protected int actualCurrent;

    /**
     * Build a new SimuEPOSControllerModule
     * @param nodeID
     * @param serialNB
     * @param paramsForCurrent
     * @param paramsForProfilePosition
     * @param paramsForHoming 
     */
    public SimuEPOSControllerModule(
            String nodeID, String serialNB,
            Map<String, Integer> paramsForCurrent,
            Map<String, Integer> paramsForProfilePosition,
            Map<String, Integer> paramsForHoming
    ) {
        super(nodeID, serialNB,paramsForCurrent,paramsForProfilePosition,paramsForHoming);
        this.position = 0;
        this.mode = EposMode.PROFILE_POSITION;
        this.enabledToPublish = true;
    }
    
    
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "For simulator only : Update position with a position given as argument.")
    public void setPosition(int actualPosition) {
        this.position = actualPosition;
    }

    /**
     * We assum that simulated hardware's controller has always correct parameters.
     * @return 
     */
    @Override
    public boolean isParametersOK() {
        return true;
    }

    @Override
    public void enableAndWriteRelativePosition(int pos) {
        enable();
        changeMode(EPOSEnumerations.EposMode.PROFILE_POSITION);
        writeTargetPosition(this.position + pos);
        writeControlWord("7F");
    } 

    @Override
    public void defineAbsolutePosition(int position)  {
        this.position = position;
    }

    @Override
    public void disable()  {
        this.enabled = false;
        publishData();
    }

    @Override
    public void writeParameters(EposMode mode) {
    }

    @Override
    public void writeParameterInHexa(EPOSEnumerations.Parameter parameter, String string) {
        FCSLOG.info(getName() + " " + parameter.toString() + " set to value in HEXA=" + string);
    }
    
    @Override
    public void writeParameter(EPOSEnumerations.Parameter param, int value) {
        FCSLOG.debug(getName() + "writeParameter");
    }

    @Override
    public void writeTargetPosition(int position)  {
        this.position = position;
    }

    @Override
    public void writeParameters(Map<String, Integer> paramMap) {
    }

    @Override
    public void writeCurrent(int aValue)  {
        this.turnedOff = false;
        this.actualCurrent = aValue;
    }


    
    @Override
    public void off()  {
        this.turnedOff = true;
    }


    @Override
    public int readPosition()  {
        return this.position;
    }

    /**
     * Read the position returned by the absolute encoder (single serial data).
     *
     * @return
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     */
    @Override
    public int readSSIPosition()  {
        return this.ssiPosition;
    }

    @Override
    public void enable()  {
        enabled = true;
        publishData();
    }

    @Override
    public int readCurrent()  {
        return actualCurrent;
    }

    @Override
    public EposMode readMode()  {
        return mode;
    }



    @Override
    public void quickStop()  {
        FCSLOG.finest(getName() + " quickStop done.");
    }

    @Override
    public int readNumberOfErrors()  {
        return 0;
    }
    
    @Override
    public String readErrorRegister() {
        return null;
    }
    
    @Override
    public String[] readErrorHistory() {
        return new String[0];
    }

    @Override
    public String displayErrorHistory()  {
        return "No ERROR";
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the Controller is in fault.")
    @Override
    public void checkFault()  {
    }
    
        /**
     * Returns true if this CANopen node is booted.
     * @return 
     */
    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns always true in simulation.")
    public boolean isBooted() {
        return true;
    }

    @Override
    public int readProfileVelocity() {
        //TODO return a value of current in random between min and max
        return 0;
    }

    @Override
    public String readParameters(EposMode mode)  {
        return getName() + ":no parameter to read.";
    }

    @Override
    public void checkParameters(EposMode aMode) {
        /*nothing to do because no controller CPU*/
    }
    
    @Override
    protected void checkCurrentValue(int aValue) {
        /*nothing to do because no controller CPU*/
    }

    @Override
    public void writeControlWord(String f)  {
        FCSLOG.debug(getName() + "writeControlWord :" + f);
        
    }

    @Override
    public void shutdownEPOS()  {
        FCSLOG.debug(getName() + " is SHUTDOWN");
    }

    @Override
    public boolean isEnabled()  {
        return enabled;
    }


    public void releaseBrake()  {
        FCSLOG.debug(getName() + " brake is released.");
    }
    

    public void activateBrake()  {
        FCSLOG.debug(getName() + " brake is activated.");
    }
}
