
package org.lsst.ccs.subsystems.fcs.simulation;

import java.util.Map;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystems.fcs.CarouselModule;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import static org.lsst.ccs.subsystems.fcs.utils.FcsUtils.addAngle;

/**
 * To simulate the controller which rotates carousel.
 * @author virieux
 */
public class SimuCarouselController extends SimuEPOSControllerModule {
    
    private CarouselModule carousel;

    /**
     * Build a new SimuCarouselController
     * @param nodeID
     * @param serialNB
     * @param paramsForCurrent
     * @param paramsForProfilePosition
     * @param paramsForHoming 
     */
    public SimuCarouselController(
            String nodeID, String serialNB, 
            Map<String, Integer> paramsForCurrent, 
            Map<String, Integer> paramsForProfilePosition, 
            Map<String, Integer> paramsForHoming) {
        super(nodeID, serialNB, 
                paramsForCurrent, paramsForProfilePosition, paramsForHoming);
    }
    
    @Override
    public void initModule() {
        super.initModule();
        carousel = (CarouselModule) getComponentLookup().getComponentByName("carousel");
    }
    
    /**
     * In the same way than for the real hardware with an EPOSController, this method starts actually 
     * the simulated motion.
     * @param controlWord 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "In PROFILE_POSITION mode this methods start the motion to the targetPosition.")
    @Override
    public void writeControlWord(String controlWord) {
        /* Absolute position*/
        if (controlWord.equals("3F")) {
            simulateWriteTargetPosition(this.targetPosition, this.position);
                    
        /* target position is a Relative position. It's the rotation angle.*/
        } else if (controlWord.equals("7F")) {
            int positionToReach = addAngle(this.targetPosition,this.position);
            int positionInitiale = this.position;
            /**
             * What follows is a hack (verrue) for the carousel simulator.
             * Because in some states the simulated carousel doesn't rotate in the shortest way.
             * It happens when the shortest way goes through the position 0.
             * For example if socket2 is at STANDBY position (carousel position is 288) and we want
             * to move socket1 at STANDBY position (target position is then 0). (clockwise way is the shortest way)
             * 
             * So the simulated carousel has to go from 288 to 0. ie from 288 to 360.
             * 
             * Another example is when socket5 is at STANDBY position
             */
            /* clockwise way*/
            if (0 <= this.targetPosition && this.targetPosition < 180 && positionToReach < position) {
                positionToReach += 360;
                
            /* counterclockwise */    
            } else if (-180 < this.targetPosition && this.targetPosition <= 0 && positionToReach > position) {
                positionInitiale += 360;
            }
            /* end of hack*/
            /**************/
            
            simulateWriteTargetPosition(positionToReach,positionInitiale);
        }
    }
    
    /**
     * This methods does nothing but save the position given as argument.
     * @param position 
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "In PROFILE_POSITION mode this methods set the target position.")
    @Override
    public void writeTargetPosition(int position)throws FcsHardwareException {
        this.targetPosition = position;
    }
    
    /**
     * This simulates a rotation of carousel to reach a position given as argument.
     * (absolute position)
     * @param positionToReach can be < -360 or > 360
     * @param positionInitiale
     */
    public void simulateWriteTargetPosition(int positionToReach, int positionInitiale) 
             {
        FCSLOG.fine(getName() + " positionToReach="+positionToReach);
        this.targetPosition = positionToReach;
        //int positionInitiale = this.position;

        FCSLOG.fine(getName() + "=> carousel is rotating.");
        FCSLOG.fine(getName() + "=> position initiale=" + this.position);
        FCSLOG.fine(getName() + "=> target position=" + this.targetPosition);
        int stepsNB = 10;
        int step = (targetPosition - positionInitiale) / stepsNB;
        try {
            for (int i = 1; i < stepsNB; i++) {
                this.position = positionInitiale + (i * step);
                carousel.updatePosition();
                FCSLOG.fine(getName() + " i=" + i + ",position=" + position);
                try {
                    Thread.sleep(500);
                    if (carousel.getHaltRequired().get()) {
                        FCSLOG.fine(getName() + " STOP simulated carousel motion.");
                        return;
                    }
                } catch (InterruptedException ex) {
                    throw new FcsHardwareException(getName() + " sleep was interrupted.",ex);
                }
            }
            this.position = targetPosition % 360;
            carousel.updatePosition();
        } catch (FcsHardwareException ex) {
            FCSLOG.error(getName()+" should not raise an Exception:",ex);
        }
    }
    
}
