
package org.lsst.ccs.subsystems.fcs.simulation;

import java.util.Observable;
import org.lsst.ccs.bus.states.PhaseState;
import org.lsst.ccs.command.annotations.Command;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import static org.lsst.ccs.subsystems.fcs.FCSCst.SENSOR14BITS_MIN;

/**
 * To simulate an analogic sensor which indicates if a Carousel Clamp is LOCKED or not.
 * For the carousel, we read the new values of the lock sensors by PDO.
 * This class provides methods to simulate that clamp is locked, or clamp is unlocked.
 * @author virieux
 */
public class SimuCarouselClampLockSensor extends SimuCanOpenSensor14bits {
    
    int acTrucksStandbyPosition = 992000;
    
    /**
     * Buyild a new SimuCarouselClampLockSensor
     * @param adcName
     * @param numOfAnalogInput 
     */
    public SimuCarouselClampLockSensor( 
            String adcName, int numOfAnalogInput) {
        super(adcName, numOfAnalogInput);
    }
    
    @Override
    public void initModule() {
        super.initModule();
        //works only for whole fcs
//        ComponentLookup lookup = getComponentLookup();
//        acTrucksStandbyPosition = ((AutoChangerTwoTrucksModule) lookup.getComponentByName("autochangerTrucks")).getStandbyPosition();
        simulateClampIsLOCKED();
    }
    
        
    /**
     * We observe SimuAutochangerPlutoGateway.
     * When it changes trucks position, we are notified, we can then simulate that carousel is LOCKED or not.
     * In phase OPERATIONAL, when this sensor is at STANDBY and autochanger trucks position has changed, 
     * and autochanger trucks are holding a filter, then the state of this simulated clamp lock sensor changes.
     * @param source
     * @param v 
     */
    @Override
    public void processUpdate(Observable source, ValueUpdate v) {
        FCSLOG.finest(getName()+" :old Value=" + this.printValue());
        if (this.isAtStandby()) {
            FCSLOG.finest(getName()+ " is AT STANDBY.");
        }
        if (this.getSubsystem().isInState(PhaseState.OPERATIONAL) && this.isAtStandby() 
                && autochanger.isHoldingFilter()) {
            
            if ((Integer) v.getValue() < acTrucksStandbyPosition - 500) {
                simulateClampIsUNLOCKED();
                
            } else if ((Integer) v.getValue() == acTrucksStandbyPosition) {
                simulateClampIsLOCKED();
            }           
        }
        FCSLOG.finest(getName()+" :new Value=" + this.printValue());
    }
    
    
    /**
     * Simulate that the clamp is locked. 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateClampIsLOCKED() {
        this.setNewRange(clamp.getLockSensorValueB(), clamp.getLockSensorValueC());
        updateFakePDOStorage();
    }

    /**
     * Simulate that the clamp is unlocked. 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateClampIsUNLOCKED() {
        this.setNewRange(SENSOR14BITS_MIN, clamp.getLockSensorValueA());
        updateFakePDOStorage();
    }

}
