
package org.lsst.ccs.subsystems.fcs.simulation;

import java.util.List;
import java.util.Map;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.subsystems.fcs.AutoChangerModule;
import org.lsst.ccs.subsystems.fcs.AutoChangerTwoTrucksModule;
import org.lsst.ccs.subsystems.fcs.AutochangerOnlineClampModule;
import org.lsst.ccs.subsystems.fcs.DigitalSensor;
import org.lsst.ccs.subsystems.fcs.CarouselModule;
import org.lsst.ccs.subsystems.fcs.ComplementarySensors;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.Filter;
import org.lsst.ccs.subsystems.fcs.FilterIdentificator;
import org.lsst.ccs.subsystems.fcs.FilterManager;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;

/**
 * To simulate the plutoGateway on the autochanger.
 * @author virieux
 */
public class SimuAutochangerPlutoGateway extends SimuPlutoGateway implements SimuFilterIDPlutoGatewayInterface {
    
    /* To read filterID hall effect sensors.*/
    private FilterIdentificator filterIdentificator;
    private FilterHolder carousel;

    private AutoChangerModule autochanger;
    //trucks controllers
    private AutoChangerTwoTrucksModule trucks;
    private SimuAutochangerLinearRailController linearRailMasterController; //(X-)
    private SimuAutochangerLinearRailController linearRailSlaveController; //(X+)
    
    //online clamps
    private AutochangerOnlineClampModule onlineClampXminus;
    private AutochangerOnlineClampModule onlineClampXplus;
    private AutochangerOnlineClampModule onlineClampYminus;


    
    //loader connected sensors 
    private ComplementarySensors loaderConnectedSensors;

    //latches sensors
    private ComplementarySensors lockSensorsLatchXminus;
    private ComplementarySensors lockSensorsLatchXplus;
    private ComplementarySensors unlockSensorsLatchXminus;
    private ComplementarySensors unlockSensorsLatchXplus;
    private ComplementarySensors filterEngagedSensorsLatchXminus;
    private ComplementarySensors filterEngagedSensorsLatchXplus;


    
    //trucks position sensors
    private ComplementarySensors handoffPositionSensorsXminus;
    private ComplementarySensors onlinePositionSensorsXminus;
    private ComplementarySensors standbyPositionSensorsXminus;
    private ComplementarySensors handoffPositionSensorsXplus;
    private ComplementarySensors onlinePositionSensorsXplus;
    private ComplementarySensors standbyPositionSensorsXplus;    

    /**
     * Build a new SimuAutochangerPlutoGateway
     * @param nodeID
     * @param serialNB 
     */
    public SimuAutochangerPlutoGateway(
            String nodeID, String serialNB) {
        super(nodeID, serialNB);
    }
    
    @Override
    public List<DigitalSensor> getFilterIDSensors() {
        return filterIdentificator.getFilterIDSensors();
    }

    @Override
    public void initModule() {
        super.initModule();
        ComponentLookup lookup = getComponentLookup();
        filterIdentificator = (FilterIdentificator) lookup.getComponentByName("filterIdentificator");
        carousel = (FilterHolder) lookup.getComponentByName("carousel");
        //trucks
        this.loaderConnectedSensors = (ComplementarySensors) lookup.getComponentByName("loaderConnectedSensors");
        this.linearRailMasterController = (SimuAutochangerLinearRailController) lookup.getComponentByName("linearRailMasterController");
        this.linearRailSlaveController = (SimuAutochangerLinearRailController) lookup.getComponentByName("linearRailSlaveController");
        this.trucks = (AutoChangerTwoTrucksModule) lookup.getComponentByName("autochangerTrucks");

        //latches
        this.lockSensorsLatchXminus = (ComplementarySensors) lookup.getComponentByName("lockSensorsLatchXminus");
        this.unlockSensorsLatchXminus = (ComplementarySensors) lookup.getComponentByName("unlockSensorsLatchXminus");
        this.filterEngagedSensorsLatchXminus = (ComplementarySensors) lookup.getComponentByName("filterEngagedSensorsLatchXminus");
        this.lockSensorsLatchXplus = (ComplementarySensors) lookup.getComponentByName("lockSensorsLatchXplus");
        this.unlockSensorsLatchXplus = (ComplementarySensors) lookup.getComponentByName("unlockSensorsLatchXplus");
        this.filterEngagedSensorsLatchXplus = (ComplementarySensors) lookup.getComponentByName("filterEngagedSensorsLatchXplus");
        


        //ONLINE clamps
        this.onlineClampXminus = (AutochangerOnlineClampModule) 
                lookup.getComponentByName("onlineClampXminus");    
        this.onlineClampXplus = (AutochangerOnlineClampModule) 
                lookup.getComponentByName("onlineClampXplus"); 
        this.onlineClampYminus = (AutochangerOnlineClampModule) 
                lookup.getComponentByName("onlineClampYminus");        

        //Trucks
        this.handoffPositionSensorsXminus = (ComplementarySensors)
                lookup.getComponentByName("handoffPositionSensorsXminus");
        this.handoffPositionSensorsXplus = (ComplementarySensors)
                lookup.getComponentByName("handoffPositionSensorsXplus");  
        this.onlinePositionSensorsXminus = (ComplementarySensors)
                lookup.getComponentByName("onlinePositionSensorsXminus");        
        this.onlinePositionSensorsXplus = (ComplementarySensors)
                lookup.getComponentByName("onlinePositionSensorsXplus"); 
        this.standbyPositionSensorsXminus = (ComplementarySensors)
                lookup.getComponentByName("standbyPositionSensorsXminus");  
        this.standbyPositionSensorsXplus = (ComplementarySensors)
                lookup.getComponentByName("standbyPositionSensorsXplus");  
        
        simulateAutochangerTrucksAreAtHandoff();
        simulateAutochangerIsEmpty();
        simulateAutochangerLatchesAreUnlocked();
        simulateAutochangerOnlineClampsAreUnlocked();
    }
    
    protected void simuComplementarySensorsOn(ComplementarySensors sensors) {
        replaceHexaValues((DigitalSensor) sensors.getSensor(), true);
        replaceHexaValues((DigitalSensor) sensors.getSensorC(), false);
    }
    
    protected void simuComplementarySensorsNotOn(ComplementarySensors sensors) {
        replaceHexaValues((DigitalSensor) sensors.getSensor(), false);
        replaceHexaValues((DigitalSensor) sensors.getSensorC(), true);
    }
    
    protected void simuComplementarySensorsInError(ComplementarySensors sensors) {
        replaceHexaValues((DigitalSensor) sensors.getSensor(), true);
        replaceHexaValues((DigitalSensor) sensors.getSensorC(), true);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerTrucksAreAtHandoff() {
        FCSLOG.info("simulateAutochangerTrucksAreAtHandoff");
        simuComplementarySensorsOn(handoffPositionSensorsXminus);
        simuComplementarySensorsOn(handoffPositionSensorsXplus);
        ComplementarySensors[] sensors = new ComplementarySensors[] {standbyPositionSensorsXminus, 
            standbyPositionSensorsXplus, onlinePositionSensorsXminus, onlinePositionSensorsXplus};
        for (ComplementarySensors coupleSensors : sensors ) {
            simuComplementarySensorsNotOn(coupleSensors);
        }
        setTrucksPosition(trucks.getHandoffPosition());
        setChanged();
        this.notifyObservers(new ValueUpdate(getName(),trucks.getHandoffPosition()));
        FCSLOG.info("end of simulateAutochangerTrucksAreAtHandoff");
    }
    

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerTrucksAreOnline() {
        FCSLOG.info("simulateAutochangerTrucksAreOnline");
        simuComplementarySensorsOn(onlinePositionSensorsXminus);
        simuComplementarySensorsOn(onlinePositionSensorsXplus);
        ComplementarySensors[] sensors = new ComplementarySensors[] {standbyPositionSensorsXminus, 
            standbyPositionSensorsXplus, handoffPositionSensorsXminus, handoffPositionSensorsXplus};
        for (ComplementarySensors coupleSensors : sensors ) {
            simuComplementarySensorsNotOn(coupleSensors);
        }              
        setTrucksPosition(trucks.getOnlinePosition());  
        setChanged();
        this.notifyObservers(new ValueUpdate(getName(),trucks.getOnlinePosition()));
        FCSLOG.info("end of simulateAutochangerTrucksAreOnline");
    }
    

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerTrucksAreAtStandby() {
        FCSLOG.info("simulateAutochangerTrucksAreAtStandby");
        simuComplementarySensorsOn(standbyPositionSensorsXminus);
        simuComplementarySensorsOn(standbyPositionSensorsXplus);
        ComplementarySensors[] sensors = new ComplementarySensors[] {onlinePositionSensorsXminus, 
            onlinePositionSensorsXplus, handoffPositionSensorsXminus, handoffPositionSensorsXplus};
        for (ComplementarySensors coupleSensors : sensors ) {
            simuComplementarySensorsNotOn(coupleSensors);
        }
        setTrucksPosition(trucks.getStandbyPosition());      
        setChanged();
        this.notifyObservers(new ValueUpdate(getName(),trucks.getStandbyPosition()));
        simulateFilterIDAtStandby();
        FCSLOG.info("end of simulateAutochangerTrucksAreAtStandby");
    }
    
    private void setTrucksPosition(int pos) {
        linearRailMasterController.setPosition(pos);
        linearRailMasterController.setSsiPosition(linearRailMasterController.getEncoderRibbonMinValue() + pos);
        linearRailSlaveController.setPosition(pos);
        linearRailSlaveController.setSsiPosition(linearRailSlaveController.getEncoderRibbonMinValue() + pos);
    }
    

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerTrucksAreInTravel() {
        ComplementarySensors[] sensors = new ComplementarySensors[] {onlinePositionSensorsXminus, 
            onlinePositionSensorsXplus, handoffPositionSensorsXminus, handoffPositionSensorsXplus,
            standbyPositionSensorsXminus, standbyPositionSensorsXplus};
        for (ComplementarySensors coupleSensors : sensors ) {
            simuComplementarySensorsNotOn(coupleSensors);
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderIsConnected() {
        simuComplementarySensorsOn(loaderConnectedSensors);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderIsDisconnected() {
        simuComplementarySensorsNotOn(loaderConnectedSensors);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateFilterIsOnAutochanger() {
        simuComplementarySensorsOn(filterEngagedSensorsLatchXminus);
        simuComplementarySensorsOn(filterEngagedSensorsLatchXplus);        
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerIsEmpty() {
        simuComplementarySensorsNotOn(filterEngagedSensorsLatchXminus);
        simuComplementarySensorsNotOn(filterEngagedSensorsLatchXplus);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerLatchesAreLocked() {
        simuComplementarySensorsOn(lockSensorsLatchXminus);
        simuComplementarySensorsOn(lockSensorsLatchXplus);
        simuComplementarySensorsNotOn(unlockSensorsLatchXminus);
        simuComplementarySensorsNotOn(unlockSensorsLatchXplus);
    }
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerLatchesAreUnlocked() {
        simuComplementarySensorsNotOn(lockSensorsLatchXminus);
        simuComplementarySensorsNotOn(lockSensorsLatchXplus);
        simuComplementarySensorsOn(unlockSensorsLatchXminus);
        simuComplementarySensorsOn(unlockSensorsLatchXplus);        
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerLatchIsLocked(String latchName) {
        if (latchName.contains("Xminus")) {
            simuComplementarySensorsOn(lockSensorsLatchXminus);
            simuComplementarySensorsNotOn(unlockSensorsLatchXminus);
        } else if (latchName.contains("Xplus")) {
            simuComplementarySensorsOn(lockSensorsLatchXplus);
            simuComplementarySensorsNotOn(unlockSensorsLatchXplus);            
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerLatchIsUnlocked(String latchName) {
        if (latchName.contains("Xminus")) {
            simuComplementarySensorsNotOn(lockSensorsLatchXminus);
            simuComplementarySensorsOn(unlockSensorsLatchXminus);
        } else if (latchName.contains("Xplus")) {
            simuComplementarySensorsNotOn(lockSensorsLatchXplus);
            simuComplementarySensorsOn(unlockSensorsLatchXplus);           
        }
    }
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLatchLockSensorsInError(String latchName) {
        if (latchName.contains("Xminus")) {
            simuComplementarySensorsInError(lockSensorsLatchXminus);
        } else if (latchName.contains("Xplus")) {
            simuComplementarySensorsInError(lockSensorsLatchXplus);
        }
    }
    
    /********************************************************************************/
    /***                       onlineClamps                                       ***/
    /********************************************************************************/

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampsAreLocked() {
        simulateAutochangerOnlineClampIsLocked(onlineClampXminus);
        simulateAutochangerOnlineClampIsLocked(onlineClampXplus);
        simulateAutochangerOnlineClampIsLocked(onlineClampYminus);
    }
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampsAreClosed() {
        simulateAutochangerOnlineClampIsClosed(onlineClampXminus);
        simulateAutochangerOnlineClampIsClosed(onlineClampXplus);
        simulateAutochangerOnlineClampIsClosed(onlineClampYminus);
    }    

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampsAreUnlocked() {
        simulateAutochangerOnlineClampIsUnlocked(onlineClampXminus);
        simulateAutochangerOnlineClampIsUnlocked(onlineClampXplus);
        simulateAutochangerOnlineClampIsUnlocked(onlineClampYminus);
    }
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampsAreOpened() {
        simulateAutochangerOnlineClampIsOpened(onlineClampXminus);
        simulateAutochangerOnlineClampIsOpened(onlineClampXplus);
        simulateAutochangerOnlineClampIsOpened(onlineClampYminus);
    }    
    
   
    
    /**
     * simulates onlineClamp is CLOSED : CLOSED and sentCurrent is currentToClamp.
     * @param clamp 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampIsClosed(AutochangerOnlineClampModule clamp) {
        simuComplementarySensorsOn(clamp.getCloseSensors());
        simuComplementarySensorsNotOn(clamp.getOpenSensors());
        clamp.setSentCurrent(clamp.getCurrentToClose());
    }

    /**
     * simulates onlineClamp is LOCKED : CLOSED and sentCurrent is currentToClamp.
     * @param clamp 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampIsLocked(AutochangerOnlineClampModule clamp) {
        simuComplementarySensorsOn(clamp.getCloseSensors());
        simuComplementarySensorsNotOn(clamp.getOpenSensors());
        clamp.setSentCurrent(clamp.getCurrentToClamp());
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampIsUnlocked(AutochangerOnlineClampModule clamp) {
        simuComplementarySensorsOn(clamp.getCloseSensors());
        simuComplementarySensorsNotOn(clamp.getOpenSensors());
        clamp.setSentCurrent(clamp.getCurrentToClose());
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampIsOpened(AutochangerOnlineClampModule clamp) {
        simuComplementarySensorsOn(clamp.getOpenSensors());
        simuComplementarySensorsNotOn(clamp.getCloseSensors());
        clamp.setSentCurrent(clamp.getCurrentToOpen());
    }
    /********************************************************************************/
    /***                 end of onlineClamps                                      ***/
    /********************************************************************************/



    
    /**
     * Changes the values of the gateway in order to simulate that the filter read by autochanger 
     * at STANDBY is the filterID of the filter on carousel at STANDBY.
     * This has to be done only for wholeFCS.
     * In autochanger-standalone mode this is not needed.
     */
    private void simulateFilterIDAtStandby() {
        if ((carousel instanceof CarouselModule) && ((CarouselModule)carousel).getFilterAtStandby() != null) {
                simulateFilterID(((CarouselModule)carousel).getFilterAtStandby().getFilterID());
        }
    }
    
    /**
     * Here we want to simulate the autochanger sensors to be consistent with the filter location.
     * If a filter is on autochanger, we suppose that it's at ONLINE position and onlineClamps are LOCKED,
     * and latches are LOCKED, and filter presence sensors detect a filter.
     */
    @Override
    public void start() {
        boolean aFilterIsOnTrucks = false;
        autochanger = (AutoChangerModule) getComponentLookup().getComponentByName("autochanger");
        FilterManager filterManager = (FilterManager) getComponentLookup().getComponentByName("filterManager");
        Map<String, Filter> filtersMap = filterManager.getFiltersMapByName();
        for (Map.Entry<String, Filter> entry : filtersMap.entrySet()) {
            Filter filter = entry.getValue();
            
            if (filter.isOnAutoChanger()) {
                FCSLOG.fine(filter.getName() + " is on autochanger trucks.");
                aFilterIsOnTrucks = true;
                autochanger.setFilterOnTrucksID(filter.getFilterID());
            } 
        }
        if (aFilterIsOnTrucks) {
                simulateAutochangerTrucksAreOnline();
                simulateFilterIsOnAutochanger();
                simulateAutochangerLatchesAreLocked();
                simulateAutochangerOnlineClampsAreLocked();
                
        } else {
                simulateAutochangerTrucksAreAtHandoff();
                simulateAutochangerIsEmpty();
                simulateAutochangerLatchesAreUnlocked();
                simulateAutochangerOnlineClampsAreUnlocked();
        }
    }
}
