
package org.lsst.ccs.subsystems.fcs.drivers;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByPlutoGateway;
import org.lsst.ccs.subsystems.fcs.common.PlutoGatewayInterface;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;

/**
 * This represents the gateway to the pluto PLC involved in the camera
 * protection system. The sensors are read through this gateway. We read the
 * data from the gateway byte by byte. Usually we read 2 bytes at one time. This
 * class has to be extended to fit the real hardware control software needs and
 * the simulator.
 *
 * @author virieux
 */
public abstract class PlutoGatewayModule extends CanOpenDevice implements PlutoGatewayInterface {

    protected String[] hexaValues;
    protected int[] intValues;

   //updatingValues is true while we are reading the value sent by the Sensor
    //this could take some time
    protected volatile boolean updatingValues = false;
    protected final Lock lock = new ReentrantLock();
    protected final Condition valueUpdated = lock.newCondition();

    /**
     * Build a new PlutoGatewayModule.
     * @param nodeID
     * @param serialNB 
     */
    public PlutoGatewayModule(String nodeID, String serialNB) {
        super(nodeID, serialNB);
        this.hexaValues = new String[16];
        for (int i=0; i<hexaValues.length; i++) {
            this.hexaValues[i] = "00";
        }
        this.intValues = new int[16];
        for (int i=0; i<intValues.length; i++) {
            this.intValues[i] = 0;
        }
    }

    /**
     * return 
     * @return intValues
     */
    public int[] getIntValues() {
        return intValues;
    }
    
    

    /**
     * @return the hexa values.
     */
    @Override
    public String[] getHexaValues() {
        lock.lock();
        try {
            while (updatingValues) {
                try {
                    //TODO put a timeout
                    valueUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(getName() + ex.getMessage());

                }
            }
            return this.hexaValues;
        } finally {
            lock.unlock();
        }
    }
    
    


    /**
     * Read new values on the device plutoGateway and updates field values.
     * @throws FcsHardwareException 
     */
    @Override
    public void updateValues()  {
        lock.lock();

        try {
            updatingValues = true;
            this.hexaValues = readNewHexaValues();

        } finally {

            updatingValues = false;
            valueUpdated.signalAll();
            lock.unlock();

        }
    }


    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public String toString() {
        StringBuilder sb = new StringBuilder(super.toString());
        for (int i=0; i<hexaValues.length; i++) {
            sb.append("/byte");
            sb.append(i);
            sb.append('=');
            sb.append(String.valueOf(this.hexaValues[i]));
        }
        return sb.toString();
    }
    
    /**
     * Create an object with status data to be published on the status bus.
     * @return 
     */
    public StatusDataPublishedByPlutoGateway createStatusDataPublishedByPlutoGatewayModule() {
        return new StatusDataPublishedByPlutoGateway(getName(), isBooted(), isInitialized(), intValues);
    }
    
    @Override
    public void publishData() {
        this.getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData(getName(), 
                createStatusDataPublishedByPlutoGatewayModule()));
    }


}
