
package org.lsst.ccs.subsystems.fcs.drivers;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.framework.Module;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.SDO_ERROR;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;
import org.lsst.ccs.subsystems.fcs.errors.CWrapperNotConnected;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenCallTimeoutException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import static org.lsst.ccs.subsystems.fcs.utils.FcsUtils.TICKMILLIS;
import org.lsst.ccs.utilities.beanutils.WrappedException;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * This Modules : - start a tcpip server on a portNumber, - waits for the
 * connection of the client (which is supposed to be a Can OPen C wrapper), -
 * send commands received by the call method to the client and - send back the
 * client response. - it handles the asynchronous messages coming from the
 * client and for that it starts a thread which listen to the client.
 *
 * @author virieux
 */
abstract class FcsTcpProxy extends Module implements AlertRaiser {
    
     /**
    * A Logger for CommandDispenser.
    */
    public static final Logger COMMAND_LOG = Logger.getLogger("org.lsst.ccs.subsystems.fcs.CommandDispenser");

    @ConfigurationParameter(isFinal=true,range="1024..99000",
        description="tcpip port number on which this tcpip server starts on.")
    protected int portNumber;

    @ConfigurationParameter(description="Timeout in Millisecond. If a command sent to the CANbus doesn't "
            + "respond during this amount of time, we considere that they could be an issue on the hardware.")    
    protected int fieldBusTimeout;
    
    private ServerSocket serverSock;
    private Thread readerThread;
    protected volatile boolean stopped = true;
    protected volatile boolean stopping = false;
    protected volatile boolean tcpServerStarted = false;
    protected ClientContext clientContext;

    private CommandDispenser commandDispenser;

    protected final Lock lock = new ReentrantLock();
    protected final Condition hardwareBooted = lock.newCondition();

    /**
     * A class to handle the tcpip connexion.
     */
    protected static class ClientContext {

        private String clientName;
        private BufferedReader reader;
        private BufferedWriter writer;
        protected Socket socket;

        ClientContext(String name, Socket socket, BufferedReader reader, OutputStream os) {
            try {
                this.clientName = name;
                this.reader = reader;
                this.socket = socket;
                writer = new BufferedWriter(new OutputStreamWriter(os, "ISO-8859-1"), 256);
            } catch (UnsupportedEncodingException e) {
                /*. SHOULD NOT HAPPEN */
                FCSLOG.error(name + ":context not started", e);
                throw new Error(e);
            }
        }
    }

    

    /**
     * Build a new FcsTcpProxy with a tcpip port number and a tickMillis value 3000.
     * And a fieldBusTimeout value 6000;
     * @param portNumber 
     */
    public FcsTcpProxy(int portNumber, int fieldBusTimeout) {
        super(TICKMILLIS);
        this.portNumber = portNumber;
        this.fieldBusTimeout = fieldBusTimeout;
    }


    

    @Override
    public void initModule() {
        stopped = false;
        stopping = false;
        tcpServerStarted = false;
        commandDispenser = new CommandDispenser();
    }

    @Override
    public void start() {
        startServer();
        startThreadReader();
    }

    /**
     * Starts the server tcp on the port portNumber. And waits for a client
     * connection.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, 
            description = "Starts the tcp server.")
    public void startServer() {
        try {
            serverSock = new ServerSocket(portNumber);
            FCSLOG.info(getName() + ":SERVER STARTED ON PORT:" + portNumber);
        } catch (IOException e) {
            FCSLOG.error(getName() + ":server not started", e);
            throw new WrappedException(e);
        }

        try {
            FCSLOG.info(getName() + ":WAITING FOR C-WRAPPER CLIENT...");
            this.getSubsystem().updateAgentState(FcsEnumerations.FilterState.WAITING_FOR_CWRAPPER, 
                    FcsEnumerations.FilterReadinessState.NOT_READY);
            FCSLOG.debug(getName() + ":DEBUGGING MODE");
            Socket sock = serverSock.accept();
            FCSLOG.info(getName() + ":socket accept on " + portNumber);
            BufferedReader reader = new BufferedReader(
                    new InputStreamReader(sock.getInputStream(), "ISO-8859-1"), 256);

            String nameAndProtocol = reader.readLine();
            if (nameAndProtocol == null) {
                FCSLOG.error(getName() + ": nameAndProtocol is null : that should not happen.");
            } else {
                String[] words = nameAndProtocol.split(" ");
                String cname = words[0];

                clientContext = new ClientContext(cname, sock, reader, sock.getOutputStream());
                FCSLOG.info(getName() + ":REGISTERED : " + cname);
                tcpServerStarted = true;
                stopped = false;
            }

        } catch (IOException e) {
            // LOG
            FCSLOG.error(getName() + " unexpected ", e);
        }
    }

    /**
     * Starts a thread which read on the tcp socket, waiting for messages coming
     * from tcp proxy. If the message has a known token (already registred in
     * commandDispenser) it is a response to a can open command that we had sent
     * to the proxy. If the token of the message is null, that means that is's
     * an unsynchronous message coming from the can open stack and transmitted
     * directly from the tcp proxy. That can be : - a boot message :
     * "boot,nodeID" - an emergency message : "emcy,80+NodeID"
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, 
            description = "Starts to listen to the tcp client.")
    public void startThreadReader() {
        FCSLOG.info(getName() + "startThreadReader");
        if (!tcpServerStarted) {
            String errorMsg = getName() + " tcp server has to be started before startThreadReader.";
            throw new FcsHardwareException(errorMsg);
        }
        /*We start thread to read from tcpProxy*/
        Runnable readingFromTcpProxy;
        readingFromTcpProxy = new Runnable() {
            @Override
            public void run() {
                while (!stopped) {
                    try {
                        String readline = clientContext.reader.readLine();
                        if (readline == null) {
                            if (stopping) {
                                FCSLOG.info(getName() + " : CWrapper is stopped.");
                            } else {
                                FCSLOG.error(getName() + " :nameAndProtocol is null : that should not happen.");
                            }
                        } else {
                            processCommand(readline);
                        }

                    } catch (IOException ex) {
                        FCSLOG.error(ex);
                        FCSLOG.error(getName() + ": Error in Thread reading from the tcp client.");
                    }
                }
            }
        };
        FCSLOG.info(getName() + ":STARTING Thread reader");
        readerThread = new Thread(readingFromTcpProxy);
        readerThread.start();
    }
    
    /**
     * Processes a command received from the tcp client.
     * 
     */
    private void processCommand(String command) {
        FCSLOG.finest(getName() + ":message read from socket=" + command);

        if (command.startsWith("emcy")) {
            processEmcyMessage(command);

        } else if (command.startsWith("Unknown")) {
            processUnknownCommand(command);

        } else {
            String[] words = command.split(",");
            String commandWord = words[0];
            String nodeID = words[1];
            if ("boot".equals(commandWord)) {
                FCSLOG.info(getName() + ":boot message received for node: " + nodeID);
                processBootMessage(nodeID);
            } else {
                String token;
                if (command.startsWith("sync")) {
                    token = "sync";
                } else {
                    token = commandWord + nodeID;
                }
                COMMAND_LOG.finest("Command token=" + token);
                if (commandDispenser.isTokenUsed(token)) {
                    COMMAND_LOG.finest("Response to a registred command:" + command);
                    try {
                        commandDispenser.registerResponse(token, command);
                    } catch (Exception ex) {
                        FCSLOG.error(ex);
                        FCSLOG.error(getName() + ":Error in registering response to command :" + command, ex.getCause());
                    }
                } else {
                    FCSLOG.warning(getName() + ":Unsynchronous message read from CANbus=" + command);
                }
            }
        }
    }

    @Override
    public void shutdownNow() {
        super.shutdownNow();
        this.stopServer();
        this.getSubsystem().updateAgentState(FcsEnumerations.FilterState.OFF_LINE, 
            FcsEnumerations.FilterReadinessState.NOT_READY);
    }

    /**
     * Stops the tcp server.
     */
    public void stopServer() {
        if (!tcpServerStarted) {
            FCSLOG.warning(getName() + " is stopped; nothing to stop.");
            return;
        }
        FCSLOG.info(getName() + ": ABOUT TO STOP TCP server.....");
        FCSLOG.finest("clientName=" + clientContext.clientName);
        stopping = true;
        try {
            call(clientContext.clientName, "quit");

        } catch (CanOpenCallTimeoutException ex) {
            String msg = getName() + ": could not stop Cwrapper.";
            throw new FcsHardwareException(msg + ex);
            
        } catch (CWrapperNotConnected ex) {
            String msg = getName() + ": could not stop because CWrapper is not connected.";
            throw new FcsHardwareException(msg + ex);
        }

        stopped = true;
        readerThread.interrupt();

        try {
            serverSock.close();
            this.tcpServerStarted = false;
            FCSLOG.info(getName() + ": Subsystem " + this.getSubsystem().getName() + " TCPIP SERVER STOPPED:");
        } catch (IOException ex) {
            FCSLOG.error(ex);
            FCSLOG.error(getName() + " : stop method does not work properly when closing socket");
        }

        try {
            clientContext.reader.close();
            clientContext.writer.close();
            clientContext.socket.close();

        } catch (IOException ex) {
            FCSLOG.error(ex);
            FCSLOG.error(getName() + " : stop method does not work properly");
        }

    }

    public synchronized boolean isReady(String clientName) {
        if (clientName == null) {
            throw new IllegalArgumentException("Client name can't be null.");
        }
        if (clientContext == null || clientContext.clientName == null) {
            return false;
        }
        if (stopped) {
            return false;
        }
        if (!tcpServerStarted) {
            return false;
        }
        FCSLOG.debug(getName() + "client name:= " + clientContext.clientName + "#");
        return clientContext.clientName.equals(clientName);
    }

    /**
     * This methods send a command to the tcp client. The command is a String
     * and can be understood by the client. Example for a Can Open command :
     * rsdo,1,1018,0 The command is stored in the command dispenser with a token
     * and command corelation. When the response comes back from the proxy we
     * can retrieve it from the dispenser with the call token.
     *
     * @param clientName
     * @param command
     * @return result of the call or null if the call failed
     */
    public Object call(String clientName, String command) {

        if (clientContext == null) {
            throw new CWrapperNotConnected(this.portNumber, clientName, 
                    " could not send command:"+command);
        }
        if (stopped) {
            throw new IllegalStateException(getName() + ":textTCP module stopped");
        }

        try {

            String sentCommand = command + "\r\0\n";
            FCSLOG.finest(getName() + ":Sent command=" + command);
            clientContext.writer.write(sentCommand, 0, sentCommand.length());
            clientContext.writer.flush();

            //if we want to wait for the response we have to store this command.
            if (command.startsWith("rsdo")
                    || command.startsWith("wsdo")
                    || command.startsWith("info")
                    || command.startsWith("sync")
                    || command.startsWith("srtr")) {
                String commandToken = commandDispenser.register(command);
                try {
                    return commandDispenser.getCommandResponse(commandToken, fieldBusTimeout);
                } finally {
                    commandDispenser.remove(commandToken);
                }
            } else {
                return "asynchronous command sent:" + command;
            }

        } catch (IOException ex) {
            FCSLOG.error(getName() + ":ERROR in tcpProxy call method");
            FCSLOG.error(ex);
            //TODO change this : we shouldn't use a WrappedException
            throw new WrappedException(getName() + ":ERROR in tcpProxy call method " + clientName, ex);
        }

    }

    /**
     * Process a boot message coming from the can open stack.
     *
     * @param nodeID
     */
    abstract void processBootMessage(String nodeID);

    /**
     * Process an emergency message coming from the can open stack.
     *
     * @param nodeID
     */
    abstract void processEmcyMessage(String message);

    /**
     * Process a message received from the CAN bus when this message is an unknown message.
     * Just raise a Warning Alarm.
     *
     * @param nodeID
     */
    protected void processUnknownCommand(String message) {
        this.raiseWarning(SDO_ERROR,getName()+  ": Unknown command received by CWrapper:" + message);
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(String.valueOf(getName()));
        sb.append("/port=");
        sb.append(this.portNumber);
        return sb.toString();
    }

}
