
package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import java.util.Map;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.CAN_BUS_TIMEOUT;
import org.lsst.ccs.subsystems.fcs.common.PDOStorage;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenCallTimeoutException;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * A class to represent the CANopen pluto gateway which is a device where we can read sensors.
 * Its raises an alert :
 * - FCS004 in readNewHexaValues
 * 
 * @author virieux
 */
public class CanOpenPlutoGateway extends PlutoGatewayModule {
    
   /**
    * A Logger for CommandDispenser.
    */
    public static final Logger LOCAL_LOG = Logger.getLogger("org.lsst.ccs.subsystems.fcs.CanOpenPlutoGateway");
    


    /**
     * build a new CanOpenPlutoGateway
     * @param nodeID
     * @param serialNB
     */
    public CanOpenPlutoGateway(String nodeID, String serialNB) {
        super(nodeID, serialNB);
    }



    @Override
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Initialize the Gateway.")
    public void initializeAndCheckHardware()  {
        checkBooted();
        try {
            FCSLOG.info(getName() + " BEGIN initializeHardware");
            this.configurePlutoGateway();
            this.initialized = true;
            FCSLOG.info(getName() + " is INITIALIZED.");
            //publishData for the LoaderHardwareGUI
            this.tcpProxy.publishHardwareData(this);

        } catch (SDORequestException | FailedCommandException ex) {
            throw new FcsHardwareException(getName(),ex);
        }
    }

    @Override
    public void initModule() {
        super.initModule();
        Map.Entry<String, Object> entry = getComponentLookup().getParent(getName());
        BridgeToHardware bridge = (BridgeToHardware) entry.getValue();
        tcpProxy = (CanOpenProxy) bridge;
    }
    
    /**
     * Configures the gateway.
     * Configuration depends on specific needs for each subsystems.
     * This methods switches to the right configuration in reading the name of the gateway.
     * 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
        description = "Configures this gateway for subsystem needs. "
                + "If gateway's name contains loader it executes configureLoaderPlutoGateway"
                + "else it executes configureAutochangerPlutoGateway.")
    public void configurePlutoGateway() {
        if (getName().contains("loader")) {
            configureLoaderPlutoGateway();
        } else if (getName().startsWith("acSensors")){
            configureAutochangerPlutoGateway();
        }
    }

    /**
     * Configures this gateway to receive 3 data blocs from Pluto.
     * Used for autochanger pluto gateway.
     * Assuming Pluto has 3 fonctions Gateway_to_User_C First one =>
     * Fonction_ID=01 Second one => Fonction_ID=02 Third one => Fonction_ID=03
     *
     * @throws SDORequestException
     */
    private void configureAutochangerPlutoGateway()  {
        
        FCSLOG.debug(getName()+ " configureAutochangerGateway");

        //Clear config 
        this.tcpProxy.writeSDO(getNodeID(), "2010", "01", "01", "00");

        //Set Additionnal data TPDO1 => define Area_0 (doc p:96)
        this.tcpProxy.writeSDO(getNodeID(), "2011", "01", "02", "17");

        //Set Additionnal data TPDO1 => define Area_1 (doc p:96)
        this.tcpProxy.writeSDO(getNodeID(), "2011", "02", "02", "4");

        //Set Additionnal data TPDO2 => define Aera_2  (doc p:96)
        this.tcpProxy.writeSDO(getNodeID(), "2012", "01", "02", "117");
        
        //Set Additionnal data TPDO2 => define Aera_3  (doc p:96)
        this.tcpProxy.writeSDO(getNodeID(), "2012", "02", "02", "143");
    }
    
    /**
     * Configures plutoGateway for loader needs.
     */
    private void configureLoaderPlutoGateway() {
        FCSLOG.debug(getName()+ " configureLoaderPlutoGateway");
        
        //Clear additional data configuration.
        this.tcpProxy.writeSDO(getNodeID(), "2010", "01", "01", "01");

        //Set Additionnal data TPDO1 => define Area_0 (doc p:96)
        this.tcpProxy.writeSDO(getNodeID(), "2011", "01", "02", "01");

        //Set Additionnal data TPDO1 => define Area_1 (doc p:96)
        this.tcpProxy.writeSDO(getNodeID(), "2011", "02", "02", "02");

        //Set Additionnal data TPDO2 => define Aera_2  (doc p:96)
        this.tcpProxy.writeSDO(getNodeID(), "2012", "01", "02", "03");
        
        //Activate additional data for Area_0, Area_1 and Area_2 (Enable Data To Pluto Areas 0-3)
        this.tcpProxy.writeSDO(getNodeID(), "2002", "01", "01", "07");
    }

    /**
     * Write a transmission type in the CPU of the pluto gateway.
     * @param transmissionType
     * @throws FcsHardwareException
     * @throws FailedCommandException
     */
    public void writeTransmissionType(String transmissionType)  {
        if (transmissionType == null) {
            throw new IllegalArgumentException(getName() + ": Can't write transmission type to device because "
                    + "transmission type is null.");
        }
        String ack = this.tcpProxy.writeSDO(getNodeID(), "2005", "1", "1", transmissionType);
        String goodAck = "OK";
        if (!ack.equals(goodAck)) {
            throw new FailedCommandException(getName() + " :bad returned ack for wsdo, index 1801");
        }
    }

    /* 
     * @return values in hexa read on the device
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return values in hexa read on the device.")
    @Override
    public String[] readNewHexaValues()  {

        checkBooted();
        try {            
            /*Index 6000*/
            hexaValues[0] = tcpProxy.readSDO(getNodeID(), "6000", "01");
            LOCAL_LOG.finest(String.format("%s : first byte read in hexaValue= %s", getName(), hexaValues[0]));
            hexaValues[1] = tcpProxy.readSDO(getNodeID(), "6000", "02");
            LOCAL_LOG.finest(String.format("%s : second byte read in hexaValue= %s", getName(), hexaValues[1]));            
            hexaValues[2] = tcpProxy.readSDO(getNodeID(), "6000", "03");
            LOCAL_LOG.finest(String.format("%s : third byte read in hexaValue= %s", getName(), hexaValues[2]));
            hexaValues[3] = tcpProxy.readSDO(getNodeID(), "6000", "04");
            LOCAL_LOG.finest(String.format("%s : 4th byte read in hexaValue= %s", getName(), hexaValues[3]));
            
            /*Index 6001*/
            hexaValues[4] = tcpProxy.readSDO(getNodeID(), "6001", "01");
            LOCAL_LOG.finest(String.format("%s : 5th byte read in hexaValue= %s", getName(), hexaValues[4]));
            hexaValues[5] = tcpProxy.readSDO(getNodeID(), "6001", "02");
            LOCAL_LOG.finest(String.format("%s : 6th byte read in hexaValue= %s", getName(), hexaValues[5]));            
            hexaValues[6] = tcpProxy.readSDO(getNodeID(), "6001", "03");
            LOCAL_LOG.finest(String.format("%s : 7th byte read in hexaValue= %s", getName(), hexaValues[6]));
            hexaValues[7] = tcpProxy.readSDO(getNodeID(), "6001", "04");
            LOCAL_LOG.finest(String.format("%s : 8th byte read in hexaValue= %s", getName(), hexaValues[7])); 
            
            /*Index 6002*/
            hexaValues[8] = tcpProxy.readSDO(getNodeID(), "6002", "01");
            LOCAL_LOG.finest(String.format("%s : 9th byte read in hexaValue= %s", getName(), hexaValues[8]));
            hexaValues[9] = tcpProxy.readSDO(getNodeID(), "6002", "02");
            LOCAL_LOG.finest(String.format("%s : 10th byte read in hexaValue= %s", getName(), hexaValues[9]));            
            hexaValues[10] = tcpProxy.readSDO(getNodeID(), "6002", "03");
            LOCAL_LOG.finest(String.format("%s : 11th byte read in hexaValue= %s", getName(), hexaValues[10]));
            hexaValues[11] = tcpProxy.readSDO(getNodeID(), "6002", "04");
            LOCAL_LOG.finest(String.format("%s : 12th byte read in hexaValue= %s", getName(), hexaValues[11]));  
            
            /*Index 6003*/
            hexaValues[12] = tcpProxy.readSDO(getNodeID(), "6003", "01");
            LOCAL_LOG.finest(String.format("%s : 13th byte read in hexaValue= %s", getName(), hexaValues[12]));
            hexaValues[13] = tcpProxy.readSDO(getNodeID(), "6003", "02");
            LOCAL_LOG.finest(String.format("%s : 14th byte read in hexaValue= %s", getName(), hexaValues[13]));            
            hexaValues[14] = tcpProxy.readSDO(getNodeID(), "6003", "03");
            LOCAL_LOG.finest(String.format("%s : 15th byte read in hexaValue= %s", getName(), hexaValues[14]));
            hexaValues[15] = tcpProxy.readSDO(getNodeID(), "6003", "04");
            LOCAL_LOG.finest(String.format("%s : 16th byte read in hexaValue= %s", getName(), hexaValues[15]));             
            
            
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(getName() + "=> ERROR IN READING SENSOR:" + ex);

        } catch (SDORequestException ex) {
            String message = getName() + "=> ERROR IN READING SENSOR:";
            this.raiseAlarm("FCS004:"+getName(), message, ex);
            
        } catch (CanOpenCallTimeoutException ex) {
            String msg = getName() + ":" + CAN_BUS_TIMEOUT.getLongDescription()
                    + " to command: readNewHexaValues - POWER FAILURE ? ";
            this.raiseWarning(getName() + ":" + CAN_BUS_TIMEOUT, CAN_BUS_TIMEOUT.getLongDescription(),msg + ex);
        }
        return hexaValues;
    }
    
    /**
     * Read new values for sensors connected to this gateway from a PDO request. 
     * @return an Array of 16 bytes in hexadecimal format
     */
    @Override
    public String[] readHexaValuesFromPDO() {
        String[] values = new String[16];
        PDOStorage pdos = tcpProxy.readPDOs();
        for (int i = 0; i < 16; i++) {
            values[i] = pdos.getVarValue(nodeID, Integer.toString(i));
        }
        return values;
    }


}
