
package org.lsst.ccs.subsystems.fcs.drivers;

import java.math.BigInteger;
import java.util.Map;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.errors.EPOSConfigurationException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.PROFILE_POSITION;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.CAN_BUS_TIMEOUT;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.PARAMETER_ERROR;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenCallTimeoutException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * A general model for a controller EPOS.
 *
 * @author virieux 
 */
public class CanOpenEPOS extends CanOpenDevice implements EPOSController {
    
    
    protected boolean turnedOff;

    /**
     * used in checkParameters and initializeAndCheckHardware methods
     */
    protected boolean parametersOK;

    /* Mode of Operation */
    protected EposMode mode;

    /* Parameters */
    @ConfigurationParameter
    private Map<String, Integer> paramsForHoming;
    
    @ConfigurationParameter
    private Map<String, Integer> paramsForProfilePosition;
    
    @ConfigurationParameter
    private Map<String, Integer> paramsForCurrent;

    /**
     * Used to publish on the status bus for the GUI*
     */
    protected boolean enabledToPublish;


    /**
     * Construction of a CanOpenEPOS.
     * @param nodeID
     * @param serialNB
     * @param paramsForCurrent
     * @param paramsForProfilePosition
     * @param paramsForHoming 
     */
    public CanOpenEPOS(
            String nodeID, String serialNB,
            Map<String, Integer> paramsForCurrent,
            Map<String, Integer> paramsForProfilePosition,
            Map<String, Integer> paramsForHoming) {
        super(nodeID, serialNB);
        this.paramsForCurrent = paramsForCurrent;
        this.paramsForProfilePosition = paramsForProfilePosition;
        this.paramsForHoming = paramsForHoming;
        turnedOff = false;
        parametersOK = true;
        enabledToPublish = false;
    }

    @Override
    public boolean isEnabledToPublish() {
        return enabledToPublish;
    }


    @Override
    public boolean isTurnedOff() {
        return turnedOff;
    }
    
    /**
     * 
     * @return true if parameters in the CPU of the controllers
     * have the same values than in configuration system.
     */
    @Override
    public boolean isParametersOK() {
        return parametersOK;
    }
    
    /**
     * Return Epos mode for GUI.
     * @return 
     */
    @Override
    public EposMode getMode() {
        return mode;
    }
    
    /**
     * return true if this controller is in the mode given as argument.
     * @param aMode
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, 
            description="return true if this controller is in the mode given as argument.")
    @Override
    public boolean isInMode(EposMode aMode) {
        return aMode.equals(readMode());
    }

    
    
    /**
     * Changes an EPOS parameter for this controller. Writes the new value on the CPU of the controller.
     * Changes the new value in the configuration.
     * But doesn't save the new parameter in the CPU. To save it, use command saveAllChanges.
     * 
     * @param key
     * @param value
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, 
            description="Changes an EPOS parameter for this controller. "
                    + "Writes the new value on the CPU of the controller."
            + "Changes the new value in the configuration. "
            + "But doesn't save the new parameter in the CPU. To save in CPU controller, "
                    + "use command saveParameters and saveAllChanges."
            + "To save configuration, use command saveAllChanges.")    
    public void changeEPOSParameter(String key, int value) {
        Parameter parameter = Parameter.valueOf(key);
        this.writeParam(key, value);
        EposMode eposMode = EposMode.getModeForParameter(parameter);
        Map<String, Integer> paramsToChange;
        String paramsToChangeName;
        if (EposMode.CURRENT.equals(eposMode)) {
            paramsToChange = paramsForCurrent;
            paramsToChangeName = "paramsForCurrent";
        } else if (EposMode.PROFILE_POSITION.equals(eposMode)) {
            paramsToChange = paramsForProfilePosition;
            paramsToChangeName = "paramsForProfilePosition";
        } else if (EposMode.HOMING.equals(eposMode)) {
            paramsToChange = paramsForHoming;
            paramsToChangeName = "paramsForHoming";           
        } else throw new IllegalArgumentException(key+ " no parameter list is defined for mode: " + eposMode);
        paramsToChange.put(key, value);
        getComponentConfigurationEnvironment().change(paramsToChangeName,paramsToChange);
        FCSLOG.info(getName()+ ":changed EPOS parameter " + key + "for mode "+ eposMode);
    }    


    /**
     * Reads the EPOS mode in the CPU of the EPOS device, updates the field mode
     * and returns the mode as a String.
     *
     * @return
     * @throws SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Reads the EPOS mode in the CPU of the EPOS device and "
            + "updates the field mode.")
    @Override
    public EposMode readMode()  {
        try {
            String modeInHexa;
            modeInHexa = this.readSDO("6061", "0").trim().toUpperCase();
            EposMode readMode = EposMode.getMode(modeInHexa);
            this.mode = readMode;
            return readMode;
        } catch (CanOpenCallTimeoutException ex) {
            String msg = getName() + ":" + CAN_BUS_TIMEOUT.getLongDescription()
                    + " to command: readMode - POWER FAILURE ? ";
            this.raiseWarning(getName() + ":" + CAN_BUS_TIMEOUT,CAN_BUS_TIMEOUT.getLongDescription(),msg + ex);
            return mode;
        }
    }


    /**
     * This methods changes the mode to the new mode given as an argument. It
     * writes the new mode in the CPU of the EPOS controller and updates the
     * field mode.
     *
     * @param newMode
     * @throws SDORequestException
     */
    @Override
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Changes EPOS mode in the CPU of the EPOS device and "
            + "updates the field mode.")
    public void changeMode(EposMode newMode)  {
        //TODO check if the mode is already newMode don't change anything.
        this.writeSDO("6060", "0", "1", newMode.getModeInHexa());
        this.mode = newMode;
        publishData();
    }


    /**
     * This methods reads in the CPU of the EPOS the values of the parameters
     * for a given mode.
     *
     * @param modeInString
     * @return
     * @throws SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "This methods reads in the CPU of the EPOS the values "
                    + "of the parameters for a given mode.")
    public String readParameters(String modeInString)  {
        return readParameters(EposMode.valueOf(modeInString));
    }

    @Override
    public String readParameters(EposMode aMode)  {
        String modeIS = aMode.toString();
        StringBuilder sb = new StringBuilder(getName() + " parameters VALUES in decimal format for mode ");
        sb.append(modeIS);
        sb.append("\n");
        Parameter[] params = aMode.getParameters();
        for (Parameter param : params) {
            sb.append(param.toString());
            sb.append("=");
            int valueInt = readParameter(param);
            sb.append(valueInt);
            sb.append("\n");
        }
        FCSLOG.debug(getName() + sb.toString());
        return sb.toString();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read the parameters for the actual mode.")
    public String readParameters()  {
        return readParameters(readMode().toString());
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "This methods writes in the CPU of the EPOS device the values "
            + "of the parameters set for the mode.")
    public void writeParameters()  {
        readMode();
        writeParameters(this.mode);
    }


    /**
     * This methods writes in the CPU of the EPOS devis the values of the
     * parameters set for the mode. Usually the values of the parameters are
     * given by the configuration system.
     *
     * @param mode
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "This methods writes in the CPU of the EPOS device the values "
            + "of the parameters set for the mode given as argument")
    @Override
    public void writeParameters(EposMode mode)  {
        String modeIS = mode.toString();
        Map<String, Integer> paramsMap = null;
        switch (mode) {
            case HOMING:
                paramsMap = this.paramsForHoming;
                break;
                
            case PROFILE_POSITION:
                paramsMap = this.paramsForProfilePosition;
                break;

            case CURRENT:
                paramsMap = this.paramsForCurrent;
                break;
            
            default:
                assert false : mode;
        }
        if (paramsMap == null) {
            throw new EPOSConfigurationException(getName()
                    + "parameters for mode :" + modeIS + "are NULL");
        }
        if (paramsMap.isEmpty()) {
            throw new EPOSConfigurationException(getName()
                    + "parameters for mode :" + modeIS + "are not defined.");
        }
        writeParameters(paramsMap);

    }

    /**
     * Write to the CPU of the EPOS a map of parameters.
     *
     * @param paramMap
     * @throws SDORequestException
     */
    @Override
    public void writeParameters(Map<String, Integer> paramMap) {
        for (Map.Entry<String, Integer> entry : paramMap.entrySet()) {
            String paramName = entry.getKey();
            int value = entry.getValue();
            CanOpenEPOS.this.writeParam(paramName, value);
        }
    }

    /**
     * This method writes a parameter in the CPU of the EPOS.
     * For end user at the console
     * @param parameterName the name of the parameter
     * @param value in decimal
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Set EPOS parameter with the value given as argument.")
    public String writeParameter(String parameterName, int value) {
        Parameter param = Parameter.valueOf(parameterName);
        writeSDO(param.getIndex(), param.getSubindex(), param.getSize(), 
                Integer.toHexString(value));
        return String.format("%s has been written to the CanEpos CPU, value=%d", 
                param.toString(), value);
    }
    
    /**
     * Set the parameter given as argument by its name with the decimal value given as
     * argument and write it in the controller CPU.
     * @param parameterName
     * @param value
     * @return 
     */
    private void writeParam(String parameterName, int value) {
        writeParameter(Parameter.valueOf(parameterName),value);
    }

    /**
     * Set the parameter given as argument with the decimal value given as
     * argument and write it in the controller CPU.
     *
     * @param param parameter to be set
     * @param value decimal value
     * @throws SDORequestException
     */
    @Override
    public  void writeParameter(Parameter param, int value) {
        writeSDO(param.getIndex(), param.getSubindex(), param.getSize(), Integer.toHexString(value));
    }

    /**
     * This method writes a parameter in the CPU of the EPOS.
     *
     * @param parameterName the name of the parameter
     * @param hexaValue value of the parameter given in hexa
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Set EPOS parameter with the hexadecimal value given as argument.")
    public void writeParameterInHexa(String parameterName, String hexaValue) {
        Parameter param = Parameter.valueOf(parameterName);
        this.writeParameterInHexa(param, hexaValue);
    }

    /**
     * Set the parameter given as argument with the hexadecimal value given as
     * argument and write it in the controller CPU.
     *
     * @param param
     * @param hexaValue
     * @throws SDORequestException
     */
    @Override
    public void writeParameterInHexa(Parameter param, String hexaValue) {
        writeSDO(param.getIndex(), param.getSubindex(), param.getSize(), hexaValue);
    }

    /**
     * Reads in the EPOS CPU the value of the Parameter which parameter name is
     * given as argument.
     *
     * @param parameterName name of the parameter to read
     * @return value of the parameter in decimal format.
     * @throws SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Reads in the EPOS CPU the decimal value of the Parameter "
            + "which parameter name is given as argument.")
    public int readParameter(String parameterName)  {
        Parameter param = Parameter.valueOf(parameterName);
        return readParameter(param);
    }

    /**
     * Reads in the EPOS CPU the value of the Parameter.
     *
     * @param param
     * @return value of the parameter in decimal format.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public int readParameter(Parameter param)  {
        try {
            String valueInHexa = readSDO(param.getIndex(), param.getSubindex());
            int valueInDecimal = new BigInteger(valueInHexa, 16).intValue();
            FCSLOG.debug(getName() + " readParameter:" + param.name() + "=" + valueInDecimal
                    + " in decimal format  (valueInHexa=" + valueInHexa + ")");
            return valueInDecimal;
        } catch (SDORequestException | ShortResponseToSDORequestException ex) {
            String msg = getName() + " Error in reading Parameter: "+param;
            this.raiseWarning(getName(), msg, ex);
        } 
        return 0;
    }

    /**
     * Reads in the EPOS CPU the hexadecimal value of the Parameter.
     *
     * @param parameterName
     * @return value of the parameter in hexa format.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Reads in the EPOS CPU the hexadecimal value of the Parameter "
            + "which parameter name is given as argument.")
    protected String readParameterInHexa(String parameterName)  {
        return readParameterInHexa(Parameter.valueOf(parameterName));
    }

    /**
     * Reads in the EPOS CPU the value of the Parameter.
     *
     * @param param
     * @return value of the parameter in hexa format.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    protected String readParameterInHexa(Parameter param)  {
        return readSDO(param.getIndex(), param.getSubindex());
    }


    /**
     * Check if the parameters in the cPU of the controller have the right values.
     * And if the controller is not in fault.
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read controller's parameter in the CPU for each mode and "
            + "warn if a parameter has not the same value as in the configuration."
            + "Then publish data.")
    @Override
    public void initializeAndCheckHardware()  {
        checkBooted();
        this.mode = readMode();
        this.parametersOK = true;
        checkParameters(EposMode.CURRENT);
        checkParameters(EposMode.PROFILE_POSITION);
        checkParameters(EposMode.HOMING);
        checkFault();
        this.initialized = true;
        if (!parametersOK) {
            FCSLOG.warning(getName() + " Some parameter values are not"
                    + " the same in CPU and configuration system.");
        }
        //publishData for the GUI
        this.tcpProxy.publishHardwareData(this);
        this.publishData();
        FCSLOG.info(getName() + ": is INITIALIZED.");
    }

    /**
     * Returns true if the controller is enabled.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if the controller is enabled.")
    @Override
    public boolean isEnabled()  {
        try {
            String statusWordInHexa = readStatusWord();
            int statusWord = Integer.parseInt(statusWordInHexa, 16);
            int[] statusWordInBinary = FcsUtils.toReverseBinary(statusWord);
            FCSLOG.debug(getName() + ":statusWordInBinary[0]=" + statusWordInBinary[0]);
            FCSLOG.debug(getName() + ":statusWordInBinary[1]=" + statusWordInBinary[1]);
            FCSLOG.debug(getName() + ":statusWordInBinary[2]=" + statusWordInBinary[2]);
            FCSLOG.debug(getName() + ":statusWordInBinary[3]=" + statusWordInBinary[3]);
            return (statusWordInBinary[0] == 1)
                    && (statusWordInBinary[1] == 1)
                    && (statusWordInBinary[2] == 1)
                    && (statusWordInBinary[3] == 0);

        } catch (ShortResponseToSDORequestException ex) {
            String msg = getName() + " couldn't read status word because ShortResponseToSDORequestException :" + ex.getMessage();
            FCSLOG.error(msg);
            FCSLOG.error(ex);
            throw new FcsHardwareException(msg,ex);
        }
    }



    /**
     * This methods enables the controller : i.e. this makes it able to receive
     * commands.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "This methods enables the controller : i.e. this makes it able to receive commands.")
    @Override
    public void enable()  {
        shutdownEPOS();
        switchOnEnableOperation();
        checkEnabled();
    }
    
    
    private void checkEnabled() {
        long timeout = 500;
        final long timeStart = System.currentTimeMillis();
        long duration = 0;
        while (!isEnabled() && duration <= timeout) {
            FCSLOG.info(getName() + " is not enabled yet.");
            duration = timeStart - System.currentTimeMillis();
        }
        if (!isEnabled()) {
            String statusWord = readStatusWord();
            FCSLOG.info(getName() + " NOT ENABLED" + " StatusWord=" + statusWord);
           throw new FcsHardwareException(getName() + " couldn't be enabled"); 
        }
    }
    
    public void checkTargetReached() {
        long timeout = 500;
        final long timeStart = System.currentTimeMillis();
        long duration = 0;
        while (!isTargetReached() && duration <= timeout) {
            FCSLOG.info(getName() + " target is not yet reached.");
            duration = timeStart - System.currentTimeMillis();
        }
        if (!isTargetReached()) {
           String msg = getName() + " couldn't reach target.";
           FCSLOG.info(msg);
           throw new FcsHardwareException(msg); 
        }
    }

    /**
     * Shutdown the controller.
     *
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Shutdown the controller.")
    @Override
    public void shutdownEPOS()  {
        writeControlWord("6");
        enabledToPublish = false;
        //TODO test the return code for the command writeSDO
        FCSLOG.debug(getName() + ": is SHUTDOWN.");
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "writeControlWord(\"F\")")
    @Override
    public void switchOnEnableOperation()  {
        writeControlWord("F");
    }   


    /**
     * Restore default parameters.
     *
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Restore default parameters. AT YOUR OWN RISK.")
    public void restoreParameters()  {
        //TODO if is Power Disabled
        this.writeSDO("1011", "1", "4", "64616F6C");
    }

    /**
     * Write a value in hexa in the control world (index=6040,
     * subindex=0,size=2)
     *
     * @param value in hexa
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Write a value in hexa in the control world (index=6040, subindex=0,size=2)")
    @Override
    public void writeControlWord(String value)  {
        this.writeSDO("6040", "0", "2", value);
    }

    /**
     * Read the control world (index=6040, subindex=0,size=2)
     *
     * @return value in hexa
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read the control world (index=6040, subindex=0,size=2)")
    public String readControlWord()  {
        return this.readSDO("6040", "0");
    }

    /**
     * Read the status world (index=6041, subindex=0,size=2)
     *
     * @return value in hexa
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read the status world (index=6041, subindex=0,size=2)")
    @Override
    public String readStatusWord()  {
        return this.readSDO("6041", "0");
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "QuickStop.")
    @Override
    public void quickStop()  {

        FCSLOG.debug(getName() + " running QUICKSTOP command.");
        
        if (this.mode.equals(EposMode.HOMING)||this.mode.equals(EposMode.PROFILE_POSITION)
                ||this.mode.equals(EposMode.VELOCITY))
            writeControlWord("B");
        else if (this.mode.equals(EposMode.CURRENT)){
            writeControlWord("2");
        } else throw new IllegalArgumentException(getName() + " has invalid Epos mode:"+this.mode);
    }



    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Turn OFF controller (in mode CURRENT, stop CURRENT) and shutdown.")
    @Override
    public void off()  {
        switch (this.mode) {
            case HOMING:
                this.writeControlWord("11F");
                break;
            case PROFILE_POSITION:
                this.writeControlWord("10F");
                break;
            case CURRENT:
                stopCurrent();
                break;
            default:
                assert false: mode;
        }
        shutdownEPOS();
        this.turnedOff = true;
        FCSLOG.info(getName() + " is TURNED OFF");
    }



    /*Methods available in CURRENT mode*/
    /**
     * In current mode this methods send a current to the motor. This make run
     * the motor.
     *
     * @param aValue UNIT=mA / FORMAT=decimal the value of the current to be
     * sent.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Sent current to controller the EPOS CPU. UNIT=mA / FORMAT=decimal")    
    @Override
    public void writeCurrent(int aValue)  {
        if (!EposMode.CURRENT.equals(mode)) {
            throw new RejectedCommandException(getName() + " is not in CURRENT mode.");
        }
        writeSDO("2030", "0", "2", Integer.toHexString(aValue));
        this.turnedOff = false;
        FCSLOG.info(getName() + ": sent current to controller=" + aValue);
    }
    
    /**
     * Enables controller and sent current value to controller.
     * For end user at console.
     * @param aValue 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Enables controller and sets the current value in the EPOS CPU.")    
    @Override
    public void enableAndWriteCurrent(int aValue)  {
        enable(); 
        checkCurrentValue(aValue);
        changeMode(EposMode.CURRENT);
        writeCurrent(aValue);
    }
    
    /**
     * Checks a current value typed by a user at the console.
     * @param aValue 
     */
    protected void checkCurrentValue(int aValue) {
        if (aValue > this.readParameter("ContinuousCurrentLimit")) {
            throw new IllegalArgumentException(aValue + " is greater than ContinuousCurrentLimit.");
        } 
    }

    /**
     * In Current Mode this methods returns the current actualy received by the motor.
     *
     * @return the current UNIT=mA / FORMAT=decimal
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, alias = "readCurrentActualValue",
            description = "In Current Mode this methods returns the current actualy received by the motor.")
    public int readCurrent()  {

        String currentInHexa = readSDO("6078", "0");
        int uns16 = Integer.parseInt(currentInHexa, 16);
        FCSLOG.debug(getName() + ":readCurrent="
                + FcsUtils.convertToInteger16(uns16) + " Unsigned value read=" + uns16);
        return FcsUtils.convertToInteger16(uns16);
    }

    /**
     * In Current Mode this methods returns the average of the current received
     * by the motor.
     *
     * @return the current UNIT=mA / FORMAT=decimal
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, alias = "readCurrentAverage",
            description = "In Current Mode this methods returns the average of the current  received by the motor.")
    public int readCurrentAverageValue()  {

        String currentInHexa = readSDO("2027", "0");
        int uns16 = Integer.parseInt(currentInHexa, 16);
        FCSLOG.debug(getName() + ":readAverageCurrent=" + FcsUtils.convertToInteger16(uns16) + " Unsigned value read=" + uns16);
        return FcsUtils.convertToInteger16(uns16);
    }

    /**
     * In current mode this methods set to zero the value of the current sent to
     * the motor. This stops the motor.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    public void stopCurrent()  {
        if (mode.equals(EposMode.CURRENT)) {
            writeSDO("2030", "00", "2", "0");
        } else {
            throw new RejectedCommandException(getName() + "is not in CURRENT mode");
        }

    }

    /**
     * Reads in the CPU the value of the parameter PositionActualValue and
     * returns it in a decimal format.
     *
     * @return value in decimal format
     * @throws SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            alias = "readPositionActualValue",
            description = "In PROFILE_POSITION Mode this methods returns the actual position.")
    @Override
    public int readPosition()  {
        int positionActualValue = this.readParameter(Parameter.PositionActualValue);
        FCSLOG.debug(getName() + ":readPosition=" + positionActualValue);

        return positionActualValue;
    }

    /**
     * Read the position returned by the absolute encoder (single serial data).
     *
     * @return
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read the position returned by the absolute encoder (single serial data).")
    @Override
    public int readSSIPosition()  {
        return this.readParameter(Parameter.SSIEncoderActualPosition);
    }

    /**
     * In PROFILE_POSITION returns the value of the parameter ProfileVelocity.
     *
     * @return
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "In PROFILE_POSITION returns the value of the parameter ProfileVelocity.")
    @Override
    public int readProfileVelocity()  {
        int profileVelocity = this.readParameter(Parameter.ProfileVelocity);
        FCSLOG.debug(getName() + ":readProfileVelocity=" + profileVelocity);
        return profileVelocity;
    }

    /**
     * Defines the actual position as the absolute position which value is given
     * as an argument.
     * Writes value 35 in hexa to set the Homing method as Actual (See EPOS
     * documentation)
     *
     * @throws SDORequestException
     * @throws RejectedCommandException
     */
    @Override
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Define the actual position as position given as argument.")
    public void defineAbsolutePosition(int position)  {
        
        FCSLOG.debug(getName() + " Defining Absolute Position:" + position);
        this.changeMode(EposMode.HOMING);
        this.writeParameters(EposMode.HOMING);
        this.writeParameter(Parameter.HomePosition, position);
        this.writeParameter(Parameter.HomingMethod, 35);
        this.enable();
        this.writeControlWord("3F");
    }

    /**
     * In PROFILE_POSITION mode this methods set the target position. This make
     * run the motor.
     *
     * @param aValue UNIT=mA / FORMAT=decimal the value of the current to be
     * sent.
     * @throws RejectedCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "In PROFILE_POSITION mode this methods set the target position."
            + " Target position has to be given in microns.")
    @Override
    public void writeTargetPosition(int aValue)  {
        if (!mode.equals(EposMode.PROFILE_POSITION)) {
            throw new RejectedCommandException(getName() + " is not in PROFILE_POSITION mode");
        }

//        if (!isEnabled()) {
//            throw new RejectedCommandException(getName() + " is not ENABLED");
//        }

        String targetPositionInHexa = Integer.toHexString(aValue);
        writeSDO("607A", "0", "4", targetPositionInHexa);
        //TODO check the return code of writeSDO
    }


    /**
     * In HOMING mode and PROFILE_POSITION mode this indicates that the position
     * is reached.
     *
     * @return true if the position is reached.
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "In HOMING mode and PROFILE_POSITION mode this indicates that the position is reached.")
    @Override
    public boolean isTargetReached()  {
        String statusWordInHexa = readStatusWord();
        int statusWord = Integer.parseInt(statusWordInHexa, 16);
        int[] statusWordInBinary = FcsUtils.toReverseBinary(statusWord);
        return statusWordInBinary[10] == 1;
    }

    /**
     * Check if the Controller is in fault.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the Controller is in fault.")
    @Override
    public void checkFault()  {
        checkBooted();
        if (readNumberOfErrors() > 0) {
            this.inError = true;
            String errorInHexa = this.readErrorRegister();
            this.errorRegister = CanOpenErrorsTable.getErrorRegisterNameByCode(errorInHexa);
            this.errorHistory = this.readErrorHistoryNames();
            setChanged();
            this.notifyObservers(new ValueUpdate(getName(), "inFault"));
            publishData();
            String errorHistoryString = displayErrorHistory();
            throw new FcsHardwareException(getName()
                    + " is in fault. " + errorHistoryString);
        } else {
            resetErrorAndNotifyObservers();
        }

    }
    
    /**
     * Clear faults on the controller.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Clear faults on the controller.", alias="resetFault")
    @Override
    public void faultReset()  {
        writeControlWord("80");
        resetErrorAndNotifyObservers();
    }
    
    private void resetErrorAndNotifyObservers() {
        this.resetError();
        setChanged();
        this.notifyObservers(new ValueUpdate(getName(), "faultReset"));
        publishData();
    }

    /**
     * For every parameter to be defined for this mode this method compares the
     * values in configuration with the value stored in controller CPU and
     * throws a HardwareException if these values are not equal. values stored
     * in the controller CPU.
     *
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "For every parameter to be defined for this mode this method "
            + "compares the values in configuration with the value stored in controller CPU"
            + "and throws a HardwareException if these values are not equal.")
    @Override
    public void checkParameters(EposMode aMode)   {
        Map<String, Integer> paramsMap = null;
        switch (aMode) {
            case HOMING:
                paramsMap = this.paramsForHoming;
                break;
                
            case PROFILE_POSITION:
                paramsMap = this.paramsForProfilePosition;
                break;

            case CURRENT:
                paramsMap = this.paramsForCurrent;
                break;
                
            default:
                assert false;
        }
        if (paramsMap == null) {
            throw new FcsHardwareException(getName()
                    + "parameters for mode :" + aMode.toString() + "are NULL in configuration system.");
        }
        if (paramsMap.isEmpty()) {
            throw new FcsHardwareException(getName()
                    + "parameters for mode :" + aMode.toString() + "are not defined in configuration system.");
        }
        checkParameters(paramsMap);
    }

    /**
     * Compare values of parameters in the map with the values of parameters in
     * the controller CPU.
     *
     * @param paramsMap
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     */
    public void checkParameters(Map<String, Integer> paramsMap)  {
        boolean errorFound = false;
        for (Map.Entry<String, Integer> entry : paramsMap.entrySet()) {
            String paramName = entry.getKey();
            int configValue = entry.getValue();
            int controllerValue = 0;
            try {
                controllerValue = readParameter(paramName);
 
            } catch (CanOpenCallTimeoutException ex) {
                String cause = getName() + ":" + CAN_BUS_TIMEOUT.getLongDescription()
                        + " to command: checkParameters - POWER FAILURE ? ";
                this.raiseWarning(getName() + ":" + CAN_BUS_TIMEOUT,CAN_BUS_TIMEOUT.getLongDescription(),cause + ex);
            }

            if (configValue != controllerValue) {
                String cause = paramName + " value found in configuration=" + configValue
                        + " value read in CPU=" + controllerValue + "\n";
                this.raiseWarning(getName()+":"+PARAMETER_ERROR,PARAMETER_ERROR.getLongDescription(), cause);
                errorFound = true;
                //commented out for tests on autochanger in january 2017
//                CanOpenEPOS.this.writeParam(paramName,configValue);
            }

        }
        parametersOK = !errorFound;
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Change mode to CURRENT mode and initialize the"
            + "CURRENT mode parameters from configuration values.")
    public void activateCurrentMode()  {
        this.changeMode(EposMode.CURRENT);
        this.writeParameters(this.paramsForCurrent);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Change mode to HOMING mode and initialize the"
            + "HOMING mode parameters from configuration values.")
    public void activateHomingMode()  {
        this.changeMode(EposMode.HOMING);
        this.writeParameters(this.paramsForHoming);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Change mode to PROFILE_POSITION mode and initialize the"
            + "PROFILE_POSITION mode parameters from configuration values.")
    public void activateProfilePositionMode()  {
        this.changeMode(EposMode.PROFILE_POSITION);
        this.writeParameters(this.paramsForProfilePosition);
    }
    

    /**
     * shutdowns controller
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Shutdowns the controller.")
    @Override
    public void disable()  {
        shutdownEPOS();
        publishData();
    }


    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            alias = "refreshGUI",
            description = "Publish data for the controller on the status bus.")
    @Override
    public void publishData() {
        super.publishData();
        this.getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData(getName(), 
                createStatusDataPublishedByEPOSController()));
    }
}
