
package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.subsystems.fcs.common.ADCInterface;
import org.lsst.ccs.subsystems.fcs.errors.HardwareNotDetectedException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

/**
 *
 * This is the driver class for the ADC we use in the FCS test bench at APC and in
 * SINGLE FILTER TEST CAN-CBX-AI814.
 *
 * @author virieux
 */
public class CanOpenADC extends CanOpenDevice implements ADCInterface {

    /**
     * One bit read on the input of the ADC represents 0.3125 mV
     */
    public static final double adcVoltsPerBit = 0.3125;
    public static final double waterFreezeTemp = 273.15;

    /**
     * The transmission types for the analogue inputs are described in the
     * following: acyclic, synchronous: The transmission is initiated if a
     * SYNC-message has been received (PDO- transmission type 0) and data has
     * changed. cyclic, synchronous: The transmission is initiated if a defined
     * number of SYNC-messages have been received (PDO-transmission type
     * 1...240). synchronous, remote request: The state of the inputs is latched
     * with each SYNC-message and is transmitted after the reception of a
     * RTR-frame (PDO-transmission type 252). asynchronous, remote request:
     * After the reception of a RTR-frame the last latched state of the inputs
     * is transmitted (PDO-transmission type 253). event controlled,
     * asynchronous: The transmission is initiated if the state of selected
     * inputs has changed (PDO-transmission type 254, 255).
     */
    @ConfigurationParameter
    protected String transmissionType;

    /**
     * Build a CanOpenADC with a CANopen node ID, a serial number and a transmission type.
     * @param nodeID
     * @param serialNB
     * @param transmissionType 
     */
    public CanOpenADC(String nodeID, String serialNB, 
            String transmissionType) {
        super(nodeID, serialNB);
        this.transmissionType = transmissionType;
    }



    /**
     * Read the analog input whom number is given as a parameter
     *
     * @param inputNB
     * @return the value read in hexa (on 16bits)
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws org.lsst.ccs.subsystems.fcs.errors.HardwareNotDetectedException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Override
    @Command
    public String readAnalogInput(int inputNB)  {

        if (!tcpProxy.isCWrapperConnected()) {
            throw new HardwareNotDetectedException(getName() + ": hardware not connected.");
        }

        if (isBooted()) {
            if ((inputNB < 1) || (inputNB > 8)) {
                throw new IllegalArgumentException("CAN-CBX-AI814 : input has to be 1..8");
            }
            String input = Integer.toHexString(inputNB);
            return tcpProxy.readSDO(nodeID, "6401", input);

        } else {
            String message = String.format("Module %s / nodeID=%s is UNREACHABLE",
                    getName(), nodeID);
            FCSLOG.error(message);

            throw new HardwareNotDetectedException("CanOpenADC MISSING",
                    this.getName(), this.nodeID, this.serialNB);
        }
    }

    /**
     * Read the voltage at the ADC input inputNB on the ADC and returns the
     * value in Volt
     *
     * @param inputNB : integer from 1 to 8
     * @return the value read in hexa (on 16bits)
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws org.lsst.ccs.subsystems.fcs.errors.HardwareNotDetectedException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Override
    public double readVoltage(int inputNB)  {

        String hexa = readAnalogInput(inputNB);
        int valueInt = Integer.parseInt(hexa, 16); //voltage in V
        return valueInt * CanOpenADC.adcVoltsPerBit;
    }

    @Override
    @Command
    public void initializeAndCheckHardware()  {
        super.initializeAndCheckHardware();
        this.writeTransmissionTypeToDevice();
        this.initialized = true;
    }

    /**
     * This methods writes the transmission type given by the configuration to
     * the device.
     *
     * @return a message
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command
    protected String writeTransmissionTypeToDevice()  {
        if (this.transmissionType == null) {
            throw new IllegalArgumentException(getName() 
                    + ": Can't write transmission type to device because transmaission type is null.");
        }
        this.tcpProxy.writeSDO(this.nodeID, "1801", "2", "1", this.transmissionType);


        this.tcpProxy.writeSDO(this.nodeID, "1802", "2", "1", this.transmissionType);

        return String.format("Transmission type of device %s is now %s", getName(), this.transmissionType);
    }

}
