
package org.lsst.ccs.subsystems.fcs.common;

import org.lsst.ccs.UsesSubsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByHardware;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;

/**
 * This is an interface for a piece of hardware on a can open network. //TODO :
 * should be renamed CanOpenDevice
 *
 * @author virieux
 */
public interface PieceOfHardware extends UsesSubsystem {

    String getSerialNB();

    String getNodeID();

    /**
     * @return true If this piece of hardware is booted and the nodeID on the
     * hardware is the same than the one in the Configuration DB.
     */
    boolean isBooted();


    /**
     * @return true if the hardware has been initialized with success, false
     * otherwise.
     */
    boolean isInitialized();

    /**
     * This methods write some configuration data on the hardware CPU.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    void initializeAndCheckHardware();
    
    default void checkBooted() {
        if (!isBooted()) {
            throw new FcsHardwareException(getName() + ": not booted");
        }
    }
    
    default void checkInitialized() {
        if (!isInitialized()) {
            throw new FcsHardwareException(getName() + ": not initialized");
        }
    }
    
    /**
     * Creates an object to be published on the STATUS bus by a PieceOfHardware.
     * @return
     */
    default StatusDataPublishedByHardware createStatusDataPublishedByHardware() {
        return new StatusDataPublishedByHardware(getName(), isBooted(), isInitialized());
    }
    
    /**
     * Publish Data on status bus for trending data base and GUIs.
     *
     */
    default void publishData() {
        StatusDataPublishedByHardware status
                = createStatusDataPublishedByHardware();
        FCSLOG.info(getName() + ":publishHardwareData is publishing:" + status);
        this.getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData(getName(), status));
    }
    
    default String printState() {
        StringBuilder sb = new StringBuilder("name="+getName());
        sb.append(",nodeID=");
        sb.append(getNodeID());
        sb.append(",serial number=");
        sb.append(getSerialNB());
        if (isBooted()) {
            sb.append(" is BOOTED");
        } else {
            sb.append(" is NOT BOOTED");
        }
        if (isInitialized()) {
            sb.append(" and INITIALIZED");
        } else {
            sb.append(" and NOT INITIALIZED");
        }        
        return sb.toString();
    }
    
}
