
package org.lsst.ccs.subsystems.fcs.common;

import java.io.Serializable;
import java.util.HashMap;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;


/**
 * To store the values returned by a PDO.
 * The PDO request is a way to read all the values returned by a CANopen devices in the same
 * request.
 * 
 * For example, if we have an ADC which has 8 physical entries, 8 sensors can be physically 
 * connected to this ADC.
 * So to read the value sent by the sensor we need to know the nodeID of the ADC 
 * and on which entry the sensor is connected to (inputNB).
 * 
 * A PDO request is a CANopen command "sync".
 * 
 * The response to a PDO request is a String which is formated like :
 * "ADC1_1=10,ADC1_2=F53B, ...."
 * That means that the sensor connected to the entry 1 of ADC which nodeID is 1, has the value 10.
 * And that the sensor connected to the entry 2 of ADC which nodeID is 1, has the value F53B.
 * 
 * All the values are in hexadecimal format.
 * 
 * The goal of this class is to provide a method (updatePDOs) to extract the values sent by sensors 
 * from this response and to store this values in a hasp map. 
 *
 */
public class PDOStorage implements Serializable {
    private static final long serialVersionUID = 2487036147845829605L;
    private static final String SEP = "_";
    /* For single-filter-test :*/
//    private static final String VAR_SUFFIXE = "ADC";
    
    /* For autochanger tests*/
    private static final String VAR_SUFFIXE = "v";
    
    /**
     * A hash map for the storage of the values sent from the PDO. 
     * key = VAR_SUFFIXE + nodeID + SEP + input numero : Example : ADC23_5 or v3a_1
     * value = value in hexa
     */
    private final HashMap<String, String> mapSensorsValues = new HashMap<>();

    /**
     * update the PDO storage in reading a PDO line response.
     *
     * @param pdoLine something like : "ADC1_1=10,ADC1_2=F53B, ...."
     */
    public void updatePDOs(String pdoLine) {
        FCSLOG.finest(": pdoLine=" + pdoLine);
        String[] words = pdoLine.split(",");
        FCSLOG.finest(": words.length=" + words.length);
        //for single-filter-test
//        for (int ix = 1; ix < words.length; ix++) {

        for (int ix = 0; ix < words.length; ix++) {
            //example toStore= ADC1_1=10
            FCSLOG.finest("ix=" + ix + "words[ix]=" + words[ix]);
            String[] toStore = words[ix].split("=");
            //example varName= ADC1_1
            String varName = toStore[0];
            FCSLOG.finest("varName=" + varName);
            String varValue = toStore[1];
            String[] mots = varName.split(SEP);
            String nodeID = mots[0].substring(3);
            FCSLOG.finest(String.format("%nwords(%s):%s nodeID=%s", ix, words[ix], nodeID));
            mapSensorsValues.put(varName, varValue);
        }
    }

    /**
     * Returns the value stored for a CANopen device identified by its nodeID and for an input 
     * number on this CANopen device.
     * 
     * 
     * @param nodeID
     * @param inputNB
     * @return 
     */
    public String getVarValue(String nodeID, String inputNB) {
        String token = VAR_SUFFIXE + nodeID + SEP + inputNB;
        return mapSensorsValues.get(token);
    }
    
    /**
     * For the simulation ONLY.
     * This set a new hexa value in the PDO Storage in order to simulate a sensor.
     * @param nodeID
     * @param inputNB
     * @param varValue
     */
    public void setVarValue(String nodeID, String inputNB, String varValue) {
        String token = VAR_SUFFIXE + nodeID + SEP + inputNB;
        mapSensorsValues.put(token, varValue);
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("PDO STORAGE:");
        sb.append("(SIZE=").append(mapSensorsValues.size()).append(")");
        sb.append(mapSensorsValues.toString());
        return sb.toString();
    }
    
}
