
package org.lsst.ccs.subsystems.fcs.common;

import org.lsst.ccs.UsesSubsystem;
import org.lsst.ccs.framework.Module;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

/**
 * This interface is to be implemented by every class which is a model for a hardware 
 * that is moved by an EPOSController.
 * The goal of this interface is to factorise methods to process emergency messages
 * coming from the CANbus either when a controller is in fault or when a command faultReset 
 * has been sent to the controller.
 * 
 * 
 * @author virieux
 */
public interface MovedByEPOSController extends UsesSubsystem {
    
    /**
     * Return the name of the controller.
     * @return 
     */
    String getControllerName();
    
    /**
     * Return true if the controller is in fault (an EmergencyMessage has been previously received.)
     * @return 
     */
    boolean isControllerInFault();
    
    /**
     * 
     * @param controllerInFault 
     */
    void setControllerInFault(boolean controllerInFault);
    

    
    /**
     * publishes data on the STATUS bus
     */
    void publishData();
    
    /**
     * Updates field controllerInFault when the controller notifies its observers and sends new values.
     * Publishes data when controller is in fault or after a faultReset. 
     * The class which implements this interface must observe its controller.
     * Needed to update the GUI.
     * @param controllerName
     * @param v 
     */
    default void updateControllerInFault(String controllerName, Module.ValueUpdate v) {
        FCSLOG.finest(getName() + ":processUpdate from controller=" + controllerName
        + " ValueUpdate=" + v.getName());
        if (v.getValue() instanceof EmergencyMessage) {
            EmergencyMessage emcyMsg = (EmergencyMessage) v.getValue();
            FCSLOG.finest(getName() + ":EmergencyMessage received from CanOpenProxy=" + emcyMsg.toString());
            processEmergencyMessage(emcyMsg);

        } else if (v.getValue() instanceof String) { 
            if ("faultReset".equals((String) v.getValue())) {
                processFaultReset();
            } else if ("inFault".equals((String) v.getValue())) {
                this.setControllerInFault(true);
            }
        }
        this.publishData();
    }
    
    
    /**
     * This method is to be used when receiving an EmergencyMessage from the CANbus.
     * 
     * An emergency message for a controller can be received from the CANbus in 2 cases:
     * - a faultReset has been previously sent to the controller, in this case the error code
     * of the emergency message is "00",
     * - or when a new error is detected by the controller device, in this case the error code 
     * is different from "00".
     * 
     * This method updates the boolean controllerInFault and publishes data.
     * It's useful for the GUI and the monitoring of the hardware.
     * 
     * @param emcyMsg the emergency message to be processed.
     */
    default void processEmergencyMessage(EmergencyMessage emcyMsg) {
            
        String errCode = emcyMsg.getDeviceErrorCode();
        if ("00".equals(errCode)) {
            FCSLOG.finest(getName() + ": faultReset has been done on controller. " + emcyMsg.toString());
            setControllerInFault(false);
        } else {
            FCSLOG.finest(getName() + ":EmergencyMessage received for "
                        + "my controller from CanOpenProxy=" + emcyMsg.toString());
            setControllerInFault(true);
        }
        this.publishData();
    }
    
    /**
     * What to do when a faultReset has been done on the controller.
     */
    default void processFaultReset() {
        this.setControllerInFault(false);
    }
    
}
