
package org.lsst.ccs.subsystems.fcs.common;

import java.util.Map;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations;
import org.lsst.ccs.subsystems.fcs.errors.EPOSConfigurationException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;

/**
 *
 * @author virieux
 */
public interface EPOSController extends PieceOfHardware {
    
    /**
     * Return true if the controller is turned OFF;
     * @return 
     */
    boolean isTurnedOff();
    
    /**
     * Turn off controller.
     * @throws FcsHardwareException 
     */
    void off();


    
    /**
     * This methods returns true if the values of parameters stored in CPU are
     * the same than those stored in the configuration system.
     * @return 
     */
    boolean isParametersOK();
    

    /**
     * This methods changes the mode to the new mode given as an argument. It
     * writes the new mode in the CPU of the EPOS controller and updates the
     * field mode.
     *
     * @param newMode
     * @throws SDORequestException
     */
    void changeMode(EposMode newMode);


    /**
     * Defines the actual position as the absolute position which value is given
     * as an argument.
     *
     * @param position
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Defines the actual position as the absolute position "
                    + "which value is given as an argument.")
    void defineAbsolutePosition(int position);

    /**
     * Return the number of errors that occured on this controller.
     * @return
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     * @throws FcsHardwareException 
     */
    int readNumberOfErrors();

    /**
     * Return a printed list of errors that occured on this controller.
     * @return
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     * @throws FcsHardwareException 
     */
    String displayErrorHistory();

    /**
     * Check if the controller is in fault.
     * Throw an Exception if the controller is in fault.
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     * @throws FcsHardwareException 
     */
    void checkFault();
    
    void faultReset();


    /**
     * This methods writes in the CPU of the EPOS devis the values of the
     * parameters set for the mode. Usually the values of the parameters are
     * given by the configuration system.
     *
     * @param mode
     * @throws SDORequestException
     */
    void writeParameters(EposMode mode);

    /**
     * In PROFILE_POSITION mode this methods set the target position. This make
     * run the motor.
     *
     * @param aValue UNIT=mA / FORMAT=decimal the value of the current to be
     * sent.
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "In PROFILE_POSITION mode this methods set the target position.")
    void writeTargetPosition(int aValue);

    /**
     * Write to the CPU of the EPOS a map of parameters.
     *
     * @param paramMap
     * @throws SDORequestException
     */
    void writeParameters(Map<String, Integer> paramMap);

    /*Methods available in CURRENT mode*/
    /**
     * In current mode this methods send a current to the motor. This make run
     * the motor.
     *
     * @param aValue UNIT=mA / FORMAT=decimal the value of the current to be
     * sent.
     * @throws org.lsst.ccs.subsystems.fcs.errors.EPOSConfigurationException
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "In current mode this methods send a current to the motor.")
    void writeCurrent(int aValue) throws EPOSConfigurationException;

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
        description = "Enable, set mode CURRENT and write current.")
    void enableAndWriteCurrent(int aValue);

    /**
     * Read actual position on the controller CPU.
     * @return
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     * @throws FcsHardwareException 
     */
    int readPosition();

    /**
     * Read the SSS position on the controller CPU.
     * @return
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     * @throws FcsHardwareException 
     */
    int readSSIPosition();

    /**
     * Read actual current on the controller CPU.
     * @return
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     * @throws FcsHardwareException 
     */
    int readCurrent();

    /**
     * Read all the value of parameters for a given EPOS mode and return a printed list of this values.
     * @param mode
     * @return
     * @throws FcsHardwareException 
     */
    String readParameters(EposMode mode);

    /**
     * Read ProfileVelocity on the controller CPU.
     * @return
     * @throws FcsHardwareException 
     */
    int readProfileVelocity();

    /**
     * enable the controller. (to be done before each action on the motor.)
     * @throws FcsHardwareException 
     */
    void enable();

    /**
     * Return true if the controller is enabled and so ready to receive commands.
     * @return
     * @throws FcsHardwareException 
     */
    boolean isEnabled();
    
    /**
     * read the EPOS mode on the controller CPU.
     * @return
     * @throws FcsHardwareException 
     */
    EposMode readMode();

    /**
     * Send a quickStop command to the controller.
     * @throws FcsHardwareException 
     */
    void quickStop();

    /**
     * Check parameters for a given EPOS mode : compare the values stored in the Configuration System and
     * the values stored in controller CPU.
     * Raise a FcsHardwareException if some values are not equals.
     * @param aMode
     * @throws HardwareException 
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException 
     */
    void checkParameters(EposMode aMode) throws HardwareException;

    /**
     * Write controlWord to the controller.
     * Use to send or start commands. See EPOS documentation.
     * @param f
     * @throws FcsHardwareException 
     */
    void writeControlWord(String f);

    /**
     * Shutdown the controller.
     * @throws FcsHardwareException 
     */
    void shutdownEPOS();

    /**
     * Throw an Exception if the target was not reached in the previous action.
     * @return
     * @throws FcsHardwareException 
     */
    boolean isTargetReached();

    /**
     * check if target is reached in HOMING or PROFILE_POSITION commands and 
     * throws an Exception if not.
     */
    void checkTargetReached();
    
    /**
     * SwitchOn and enable controller.
     * @throws FcsHardwareException 
     */
    void switchOnEnableOperation();

    
    /**
     * Return the EPOS mode stored in field mode. Doesn't read again controller CPU.
     * @return 
     */
    EposMode getMode();

    /**
     * For GUI. Doesn't read again controller CPU.
     * Return true if controller is enabled.
     * @return 
     */
    boolean isEnabledToPublish();

    /**
     * Return true if controller is enabled.
     * @return 
     */
    boolean isInError();

    /**
     * 
     * @return error register in printed format
     */
    String getErrorRegister();

    /**
     * 
     * @return a printed list of errors.
     */
    String[] getErrorHistory();

    /**
     * Write on the CPU of the controller a value given in hexadecimal format for a parameter.
     * @param parameter
     * @param string HEXADECIMAL format of the new value.
     * @throws FcsHardwareException 
     */
    void writeParameterInHexa(EPOSEnumerations.Parameter parameter, String string);

    /**
     * Disable controller.
     * @throws FcsHardwareException 
     */
    void disable();
   
    
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Enable controller and go to absolute position. Doesn't check condition. DANGER !!!! ")
    default void enableAndWriteAbsolutePosition(int pos) {
        enable();
        changeMode(EposMode.PROFILE_POSITION);
        writeTargetPosition(pos);
        writeControlWord("3F");
    }
    
    
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Enable controller and go to relative position. Doesn't check condition. DANGER !!!! ")
    default void enableAndWriteRelativePosition(int pos) {
        enable();
        changeMode(EposMode.PROFILE_POSITION);
        writeTargetPosition(pos);
        writeControlWord("7F");
    } 
    
    /**
     * Creates an object to be published on the STATUS bus by an EPOSController.
     * @return 
     */
    default StatusDataPublishedByEPOSController createStatusDataPublishedByEPOSController() {
        StatusDataPublishedByEPOSController status = new StatusDataPublishedByEPOSController(getName(),
                isBooted(),isInitialized());
        status.setInError(isInError());
        status.setErrorRegister(getErrorRegister());
        status.setErrorHistory(getErrorHistory());
        status.setEnabled(isEnabledToPublish());
        status.setMode(getMode());
        return status;
    }

    public void writeParameter(EPOSEnumerations.Parameter parameter, int position);

    public String readStatusWord();

    /**
     * return true if this controller is in the mode given as argument.
     * @param aMode
     * @return
     */
    boolean isInMode(EposMode aMode);

}

