
package org.lsst.ccs.subsystems.fcs.common;

import org.lsst.ccs.UsesSubsystem;
import org.lsst.ccs.bus.data.Alert;
import static org.lsst.ccs.bus.states.AlertState.ALARM;
import static org.lsst.ccs.bus.states.AlertState.WARNING;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert;

/**
 * This interface is implemented by each class that can raises an Alert.
 * @author virieux
 */
public interface AlertRaiser extends UsesSubsystem {
    
    /**
     * Raises an ALARM with a fcsErrorID, a message and a cause.
     * @param fcsErrorID
     * @param message description of the fcsErrorID
     * @param cause 
     */
    default void raiseAlarm(String fcsErrorID, String message, String cause) {
        FCSLOG.error(fcsErrorID+message+cause);
        this.getSubsystem().raiseAlert(new Alert(fcsErrorID,message), ALARM, cause);
    }    
    
    /**
     * Raises an ALARM with a FcsAlert and a cause.
     * @param fcsAlert
     * @param cause 
     */
    default void raiseAlarm(FcsAlert fcsAlert, String cause) {
        raiseAlarm(fcsAlert.name(), fcsAlert.getLongDescription(), cause);
    }
    
    /**
     * Raises an ALARM with a FcsAlert, a cause and a deviceName.
     * @param fcsAlert
     * @param cause
     * @param deviceName 
     */
    default void raiseAlarm(FcsAlert fcsAlert, String cause, String deviceName) {
        raiseAlarm(deviceName+'/'+fcsAlert.name(), fcsAlert.getLongDescription(), cause);
    }
    
    
    /**
     * Raises an ALARM with a fcsErrorID, a message and an exception.
     * @param fcsErrorID
     * @param message
     * @param ex 
     */
    default void raiseAlarm(String fcsErrorID, String message, Exception ex) {
        FCSLOG.error(message,ex);
        this.getSubsystem().raiseAlert(new Alert(fcsErrorID,message), ALARM, message + ex);
    }
    
    
    /**
     * Raises a WARNING alert with a fcsErrorID, a message and a cause.
     * @param fcsErrorID
     * @param message description of the fcsErrorID.
     * @param cause 
     */
    default void raiseWarning(String fcsErrorID, String message, String cause) {
        FCSLOG.warning(message);
        this.getSubsystem().raiseAlert(new Alert(fcsErrorID,message), WARNING, cause);
    }
    
    /**
     * Raises a WARNING with an fcsAlert and a cause.
     * @param fcsAlert
     * @param cause 
     */
    default void raiseWarning(FcsAlert fcsAlert, String cause) {
        raiseWarning(fcsAlert.name(),fcsAlert.getLongDescription(), cause);
    }
    
    /**
     * Raises an WARNING with a FcsAlert, a cause and a deviceName.
     * @param fcsAlert
     * @param cause
     * @param deviceName 
     */
    default void raiseWarning(FcsAlert fcsAlert, String cause, String deviceName) {
        raiseWarning(deviceName +':'+ fcsAlert.name(), fcsAlert.getLongDescription(), cause);
    }
    
    /**
     * Raises a WARNING with a fcsErrorID, a message and a cause.
     * @param fcsErrorID
     * @param message
     * @param ex 
     */
    default void raiseWarning(String fcsErrorID, String message, Exception ex) {
        FCSLOG.warning(message);
        this.getSubsystem().raiseAlert(new Alert(fcsErrorID,message), WARNING, message + ex);
    }    
  
}
