
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.framework.Module;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.PlutoGatewayInterface;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;

/**
 * This is a model for the Inclinometer used in CPPM test bench and on 
 * whole size prototype.
 * 
 * Because for mechanical raisons, the filter changer system X axis has to be horizontal
 * in order to be able to rotate carousel or move autochanger trucks along its rails.
 * So we have 2 inclinometers which give redondant information. We must read permanently
 * the 2 inclinometers to check that inclinaison is correct. 
 * We read the inclinaison on the plutoGateway.
 *
 * @author virieux
 */
public class InclinometerModule extends Module {
    
    @SuppressWarnings("FieldMayBeFinal")
    @ConfigurationParameter
    private final String plutoGatewayName;    

    @SuppressWarnings("FieldMayBeFinal")
    @ConfigurationParameter
    private final int byteNumero;
    
    private PlutoGatewayInterface plutoGateway;
    
    private double inclinaison;

    /**
     * Build an InclinometerModule with the numero of byte read on the device.
     * 
     * @param plutoGatewayName
     * @param byteNumero
     */
    public InclinometerModule(String plutoGatewayName, int byteNumero) {
        this.plutoGatewayName = plutoGatewayName;
        this.byteNumero = byteNumero;
    }
    
    /**
     * Returns inclinaison value in degrees.
     * Doesn't read again plutoGateway.
     *
     * @return inclinaison
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns inclinaison value in degrees. Doesn't read again CANopen device.")
    public double getInclinaison()  {
        return inclinaison;
    }
    
    @Override
    public void initModule() {
        FCSLOG.info(getName() + ": init MODULE");
        ComponentLookup lookup = getSubsystem().getComponentLookup();
        this.plutoGateway = (PlutoGatewayInterface) lookup.getComponentByName(plutoGatewayName);
    }

    /**
     * Updates inclinaison from array of hexaValues.
     * @param hexaValues
     * @throws FcsHardwareException 
     */
    public void updateInclinaison(String[] hexaValues)  {
        //the value read from the plutoGateway is in tenth of volts
        //so it has to be divided by ten to obtain the value in volt
        String valueInHexa = hexaValues[byteNumero];
        inclinaison = convertVoltage2Degree(Integer.parseInt(valueInHexa, 16) / 10);
    }

    /**
     * Read new value for inclinometer on plutoGateway and return inclinaison.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Reads new value for inclinometer and returns inclinaison.")    
    public double readInclinaison() {
        this.plutoGateway.readNewHexaValues();
        updateInclinaison(this.plutoGateway.getHexaValues());
        return inclinaison;
    }

    /**
     * Concert a voltage in volt between 0.5 and 4.5 volts.
     *
     * @param voltage
     * @return
     */
    private static double convertVoltage2Degree(int voltage) {
        if (voltage < 0 || voltage > 5) {
            throw new IllegalArgumentException(voltage + ": bad value for voltage. " 
                    + "Voltage has to be between 0 and 5");
        }
        //voltage is in volt
        double sin_angle = (voltage - 2.5) / 11.52;
        return Math.toDegrees(Math.asin(sin_angle));
    }

}
