package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.UsesSubsystem;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import static org.lsst.ccs.bus.states.AlertState.ALARM;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterLocation;
import static org.lsst.ccs.subsystems.fcs.utils.FcsUtils.checkSocketName;

/**
 * This is a model for an optical filter for the camera.
 *
 */
public class Filter implements UsesSubsystem {

    public static final String SOCKET_NAME = "socket";
    
    @ConfigurationParameter(isFinal = true, description="weight of this filter", range="0..100")
    private double weight;
    
    @ConfigurationParameter(description="Name of the socket where the filter is stored in.", category="filterLocation")
    private String socketName;
    
    @ConfigurationParameter(isFinal = true, description="FilterID which is coded on filter frame with 6 hall effect sensors.", range="1..63")
    private int filterID;
    
    private FilterLocation filterLocation = FilterLocation.UNKNOWN;
    

    
    /**
     * Build a filter with an ID and a weight and a socketName.
     * @param filterID
     * @param weight 
     * @param socketName 
     */    
    public Filter(int filterID, double weight, String socketName) {
        this.filterID = filterID;
        this.weight = weight;
        this.socketName = socketName;
    }
       

    /**
     * This method is called by FilterLocator during INITIALIZATION to update field filterLocation.
     * It's a bad idea to put this initialization in the constructor because if field socketName has changed 
     * it has to be read from configuration system and not from the description groovy file. 
     * But constructor reads only the description file.
     */
    public void initializeFilterLocation() {
                
        if (socketName.contains(SOCKET_NAME)) {
            this.filterLocation = FilterLocation.CAROUSEL;
            
        } else if (socketName.contains("AC")){
            /*The filter is in Autochanger or in Loader or Out of camera. We don't know yet.*/
            this.filterLocation = FilterLocation.AUTOCHANGER;
            
        } else if (socketName.contains("OUT")){
            /*The filter is Out of camera.*/
            this.filterLocation = FilterLocation.OUT;
            
        } else {
            /*The filter is in Loader or Out of camera. We don't know yet.*/
            this.filterLocation = FilterLocation.UNKNOWN;
        }
        
        
    }

    /**
     *
     * @return true if the filter is held by the carousel.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Returns true if filter is held by the carousel.")    
    public boolean isOnCarousel() {
        return this.filterLocation == FilterLocation.CAROUSEL;
    }

    /**
     *
     * @return true if the filter is held by the autochanger.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Returns true if filter is held by autochanger.")    
    public boolean isOnAutoChanger() {
        return this.filterLocation == FilterLocation.AUTOCHANGER;
    }

    /**
     *
     * @return true if the filter is out the camera.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Returns true if filter is out of camera.")    
    public boolean isOut() {
        return this.filterLocation == FilterLocation.OUT;
    }

    /**
     *
     * @return true if the location of the filter is unknown.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Returns true if filter location is UNKNOWN.")    
    public boolean isUnknown() {
        return this.filterLocation == FilterLocation.UNKNOWN;
    }

    /**
     * Returns filter location
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Returns filter location.")    
    public FilterLocation getFilterLocation() {
        return filterLocation;
    }

    /**
     * Set a new location and publishes data.
     * @param filterLocation 
     */
    public void setFilterLocation(FilterLocation filterLocation) {
        this.filterLocation = filterLocation;
        if (FilterLocation.AUTOCHANGER.equals(filterLocation)) {
            setSocketName("AC");
        } else if(FilterLocation.OUT.equals(filterLocation)) {
            setSocketName("");
        }
        this.publishData();
    }

    /**
     * Returns filterID as Integer
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Returns filter id.")    
    public int getFilterID() {
        return filterID;
    }
    
    

    /**
     * return filter weight
     * @return 
     */
    public double getWeight() {
        return weight;
    }


    public String getSocketName() {
        return socketName;
    }

    /**
     * To be used by end user to change the value stored in the Configuration system.
     * Tests validity of socket name.
     * @param socketName 
     */
    @ConfigurationParameterChanger
    public void setSocketName(String socketName) {
        checkSocketName(socketName);
        this.socketName = socketName;
    }
    

    /**
     * This filter is now on carousel in socket given by its name.
     * It publishes data to update the GUI.
     * 
     * @param socketName 
     */
    public void putFilterOnSocket(String socketName)  {
        
        try {
                synchronized (this) {
                    this.filterLocation = FilterLocation.CAROUSEL;
                    getComponentConfigurationEnvironment().change("socketName", socketName);
                }
        } catch (Exception ex) {
            String msg = getName()+" Software error : couln't update socket name:";
                FCSLOG.error(getName() + msg,ex);
                Alert alert = new Alert("FCS010"+getName(), msg + ex.getMessage());
                this.getSubsystem().raiseAlert(alert, ALARM, "cause");

        }
        publishData();
    }
    
    /**
     * This method is call by the CarouselSocket when the filter is not seen 
     * anymore by the filter presence sensor at STANDBY position.
     * When a filter is removed from CAROUSEL, the only place it can go it's the 
     * AUTOCHANGER.
     */
    public void removeFromCarousel() {

        try {
                synchronized (this) {
                    this.filterLocation = FilterLocation.AUTOCHANGER;
                    getComponentConfigurationEnvironment().change("socketName", "AC");
                }
        } catch (Exception ex) {
                String msg = " Software error : couln't remove filter from carousel:";
                FCSLOG.error(getName() + msg,ex);
                Alert alert = new Alert("FCS010"+getName()," Software error : couln't remove filter from carousel:"
                        + ex.getMessage());
                this.getSubsystem().raiseAlert(alert, ALARM, "cause");
        }
        FCSLOG.info(getName() + " removed from carousel");
        publishData();
    }
    
    
    
    /**
     * Creates an object StatusDataPublishedByFilter and  publishes it on the 
     * Status Bus for trending data base and GUIs.
     *
     */
    public void publishData() {
        getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData(getName(), 
                new StatusDataPublishedByFilter(filterLocation,socketName)));
    }
    

    @Command(alias = "printFilter",
            type = Command.CommandType.QUERY, 
            level = Command.NORMAL,
            description = "Return a printed filter.")
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(this.getName());
        sb.append(":filterID:");
        sb.append(filterID);       
        sb.append(",Weight:");
        sb.append(weight);
        sb.append(",FilterLocation:");
        sb.append(filterLocation);
        if (filterLocation.equals(FilterLocation.CAROUSEL)) {
            sb.append(",socket:");
            sb.append(socketName);
        } 
        return sb.toString();
    }




}
