
package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.framework.Module;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.PlutoGatewayInterface;

/**
 * This class is a model for the fakeAutochanger that we use on loader and carousel test bench 
 * in standalone mode.
 * 
 * As there is no autochanger, a numeric sensor is setup to simulate that the autochanger holds the filter
 * at STANDBY or HANDOFF position.
 * It's also a model for the fakeCarousel and the fakeLoader in autochanger-standalone subsystem.
 * 
 * FakeFilterHolder implements interface FilterHolder.
 * In the whole FCS, carousel, autochanger and loader implements FilterHolder too.
 * 
 * @author virieux
 */
public class FakeFilterHolder extends Module implements FilterHolder {
    
    @ConfigurationParameter(isFinal = true)
    private String plutoGatewayName;
    
    private PlutoGatewayInterface plutoGateway;
        
    private ComplementarySensors holdingFilterSensors;
    private DigitalSensor holdingFilterSensor;
    
    /**
     * holdingFilter is updated by method updateStateWithSensors
     */
    private boolean holdingFilter = false;
    
    /**
     * sensors which says if this object holds filter can be a couple of sensors (ComplementarySensors)
     * or an unique sensor.
     * If there is a couple of sensors, onlyOneSensor is false, else it's true. 
     */
    private final boolean onlyOneSensor;

    //Used because we have to wait for the update from the sensors
    private final Lock lock = new ReentrantLock();
    private final Condition stateUpdated = lock.newCondition();
    private volatile boolean updatingState = false;

    /**
     * Build a FakeFilterHolder with a pluto gateway name and a couple of digital sensors.
     * Used in 
     * @param plutoGatewayName 
     * @param holdingFilterSensors 
     */
    public FakeFilterHolder(String plutoGatewayName,
            ComplementarySensors holdingFilterSensors) {
        this.plutoGatewayName = plutoGatewayName;
        this.holdingFilterSensors = holdingFilterSensors;
        this.onlyOneSensor = false;
    }

    /**
     * A constructor with with a pluto gateway name and only one filter sensor.
     * Used for loader standalone module.
     * @param plutoGatewayName
     * @param holdingFilterSensor 
     */
    public FakeFilterHolder(String plutoGatewayName, 
            DigitalSensor holdingFilterSensor) {
        this.plutoGatewayName = plutoGatewayName;
        this.holdingFilterSensor = holdingFilterSensor;
        this.onlyOneSensor = true;
    }

    /**
     * initialize plutoGateway
     */
    @Override
    public void initModule() {
        this.plutoGateway = (PlutoGatewayInterface) getComponentLookup().getComponentByName(plutoGatewayName);
    }
    
    /**
     * Return true if the fake filter holder is holding the filter.
     * This command doesn't read again the sensor.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the fake autochanger is holding the filter. "
                    + "This command doesn't read again the sensor.")
    @Override
    public boolean isHoldingFilter() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(getName() + ": has been interrupted while waiting for end of update.");
                }

            }
            return holdingFilter;

        } finally {
            lock.unlock();
        }
    }
    
    
    /**
     * This methods updates the field holdingFilter in reading the holding filter sensor.
     * Then it published status data on the STATUS bus.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, 
            description = "Update clamp state in reading sensors.")
    @Override
    public void updateStateWithSensors()  {
        
        plutoGateway.checkBooted();
        plutoGateway.checkInitialized();
        
        lock.lock();
        try {
            updatingState = true;
            this.plutoGateway.updateValues();
            computeState(this.plutoGateway.getHexaValues());
        } finally {

            updatingState = false;
            stateUpdated.signal();
            lock.unlock();
            this.publishData();
        }
    }
    
    private void computeState(String[] hexaValues) {
        if (onlyOneSensor) {
            this.holdingFilterSensor.updateValue(hexaValues);
            this.holdingFilter = this.holdingFilterSensor.isOn();
        } else {
            this.holdingFilterSensors.updateValues(hexaValues);
            this.holdingFilter = holdingFilterSensors.isOn();
        }
    }
    
    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    public void publishData() {
        KeyValueData kvd;
        if (onlyOneSensor) {
            kvd = new KeyValueData(getName(), holdingFilterSensor.isOn());
        } else {
            kvd = new KeyValueData(getName(), holdingFilterSensors.getDigitalValue());
        }
        this.getSubsystem().publishSubsystemDataOnStatusBus(kvd);
    }

    /**
     * For the fake autochanger there is no autochanger trucks, so the position has not a real meaning.
     * The autochanger is supposed to be at HANDOFF.
     * Idem for fakeCarousel and fakeLoader.
     * @return 
     */
    @Override
    public boolean isAtHandoff() {
        return true;
    }

    /**
     * For the fake autochanger there is no autochanger trucks, so the position has not a real meaning.
     * The autochanger is supposed to be at STANDBY.
     * Idem for fakeCarousel and fakeLoader.
     * @return 
     */
    @Override
    public boolean isAtStandby() {
        return true;
    }

}
