
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;

/**
 * Represents a couple of DigitalSensor which are complementary :
 * when one has the value 1 the other must have value 0 and vice versa.
 * If both sensors have the same value, the couple of sensors is in error.
 * 
 * This sort of sensors are used in autochanger. 
 * The goal is to be able to detect errors in sensors (capteur défectueux, error de câblage)
 * 
 * @author virieux
 */
public class ComplementarySensors extends Module implements AlertRaiser {
    
    /**
     * first sensor in the couple of complementary sensors.
     */
    private final DigitalSensor sensor;
    
    /**
     * complementary sensor.
     */
    private final DigitalSensor sensorC;

    /**
     * Create a couple of complementary sensors.
     * @param sensor 
     * @param sensorC 
     */
    public ComplementarySensors(DigitalSensor sensor, DigitalSensor sensorC) {
        this.sensor = sensor;
        this.sensorC = sensorC;
    }

    /**
     * for simulation
     * @return 
     */
    public DigitalSensor getSensor() {
        return sensor;
    }

    /**
     * for simulation
     * @return 
     */
    public DigitalSensor getSensorC() {
        return sensorC;
    }
    
    
    
    /**
     * 
     * @return true if this couple of sensors are in error.
     */
    @Command(description="return true if this couple of sensors are in error.")
    public boolean isInError() {
        return sensor.isOn() == sensorC.isOn();
    }
    
    /**
     * 
     * @return true if sensor's value is 1 and sensorC's value is 0
     */
    @Command(description="return true if sensor's value is 1 and sensorC's value is 0.")
    public boolean isOn() {
        return sensor.isOn() && !sensorC.isOn();
    }
    
    /**
     * 
     * @return sensor's digital value
     */
    public boolean getDigitalValue() {
        return sensor.isOn();
    }
    
    /**
     * Updates sensors values from a 
     * @param hexaValues 
     */
    public synchronized void updateValues(String[] hexaValues) {
        sensor.updateValue(hexaValues);
        sensorC.updateValue(hexaValues);
    }
}
