
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;

/**
 * This class is the Main Module when carousel is in standalone mode.
 * 
 * CarouselModle implements HardwareController because we must initialize acPLutoGateway during
 * initialization Phase.
 * 
 * @author virieux
 */
public class CarouselMainModule extends MainModule implements FilterLocator {
    
    @ConfigurationParameter(isFinal = true)
    private String plutoGatewayName;
    
    private CarouselModule carousel;
    
    private FilterHolder autochanger;
     
    private final FilterManager filterManager;
    
    private PieceOfHardware fakePlutoGateway;
    

    
    /**
     * Build a CarouselMainModule.
     * @param bridge
     * @param filterMg 
     * @param plutoGatewayName 
     */
    public CarouselMainModule(
            BridgeToHardware bridge,
            FilterManager filterMg,
            String plutoGatewayName
            ) {
        super(bridge);
        this.filterManager = filterMg;
        this.plutoGatewayName = plutoGatewayName;
    }

    @Override
    public FilterManager getFilterManager() {
        return filterManager;
    }

    @Override
    public CarouselModule getCarousel() {
        return carousel;
    }
    
    @Override
    public void initModule() {
        ComponentLookup lookup = getComponentLookup();
        carousel = (CarouselModule) lookup.getComponentByName("carousel");
        
        fakePlutoGateway = (PieceOfHardware) lookup.getComponentByName(this.plutoGatewayName);
        
        if (lookup.getComponentByName("autochanger") instanceof FilterHolder) {
            autochanger = (FilterHolder) lookup.getComponentByName("autochanger");
        } else {
            final String MSG = name + " ==> autochanger doesn't implements FilterHolder -"
                    + " Please fix groovy description file or CarouselMainModule implementation.";
            FCSLOG.error(MSG);
            throw new IllegalArgumentException(MSG);
        }
    }
    
    
    /**
     * Update state in reading sensors.
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Update state in reading sensors.")
    @Override
    public void updateStateWithSensors()  {
        super.updateStateWithSensors();
        this.carousel.updateClampsStateWithSensors();
        this.autochanger.updateStateWithSensors();
    }
    
    /**
     * Check that Carousel hardware is ready to be operated and moved.
     * This means that :
     * - all CAN open devices are booted, identified and initialized,
     * - homing has been done on the controllers.
     * This updates the FCS state and FCS readyness state and publishes on the status bus.
     * 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, 
            description = "Update FCS state and FCS readyness state and publishes on the status bus.")
    @Override
    public void updateFCSState() {
        if (carousel.isInitialized()) {
            /* The initialization has been done, so now the hardware is ready */
            this.getSubsystem().updateAgentState(FcsEnumerations.FilterState.HARDWARE_READY, 
                FcsEnumerations.FilterReadinessState.READY);
        }
    }
 
    @Override
    public void postStart() {
        checkFiltersLocation();
        fakePlutoGateway.initializeAndCheckHardware();
    }


    @Override
    public void shutdownNow() {
        /*to save filter location in a file*/
        this.saveFilterLocation();
        super.shutdownNow();
    }


    
}
