package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import static org.lsst.ccs.subsystems.fcs.FCSCst.SENSOR14BITS_MAX;
import static org.lsst.ccs.subsystems.fcs.FCSCst.SENSOR14BITS_MIN;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.common.PDOStorage;
import org.lsst.ccs.subsystems.fcs.common.Sensor14bits;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.Thermometer;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.SensorValueOutOfRangeException;

/**
 * A class which extends Module to model a clamp that holds a filter
 * on the carousel. 
 * Each clamp on the carousel is coupled with a sensors which detect the
 * presence of a filter, or if the clamp is locked or unlocked. 
 * This class provides a method to lock, unlock or release the clamps, 
 * and to compute the clampState of the clamp. 
 * The clampState of the clamp is computed, updated and published on the status bus
 * each tick of the timer.
 *
 *
 * @author virieux
 */
public class CarouselClampModule extends MobileItemModule {



    /*to be able to read sensors from PDO*/
    private BridgeToHardware bridge;
    
    /* to be able to known if autochanger hold a filter or not at STANDBY*/
    private FilterHolder autochanger;
    
    /**
     * The motor controller which controls this clamp.
     * This is initialized in the initModule method of CarouselSocket.
     */
    protected EPOSController controller;

    /**
     * The filterPresenceSensor detects where if the filter is in the clamp.
     *
     * @See enum PresenceFilterStateOnClamp
     */
    private final Sensor14bits filterPresenceSensor;

    /**
     * The lock sensor detects if the clamp is locked or not.
     */
    private final Sensor14bits lockSensor;

    protected FilterClampState clampState;
    protected FilterPresenceStatus filterPresenceStatus;
    private LockStatus lockStatus;
    
    /**
     * The thermometer mesures the temperature inside the clamp.
     */
    private final Thermometer thermometer;
    protected double temperature;
    
    private final Condition stateUpdated = lock.newCondition();

    /* This is used when we update the clamp clampState with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;
    
    @ConfigurationParameter
    protected int currentToUnlock = 1500;
    
    @ConfigurationParameter
    protected int currentToMaintainUnlocked = 200;

    /**
     * If the lock sensor returns a value between 0 and lockSensorValueA: the
     * sensor is unlocked. 
     * If the lock sensor returns a value between lockSensorValueA and lockSensorValueB: we don't know.
     * If the sensor returns a value between lockSensorValueB and ValueC: the clamp is locked.
     * If the sensor returns a value greater than valueC the sensor is in ERROR.
     */
    @ConfigurationParameter(range="0..32768")
    private Integer lockSensorValueA = 5000; /*6000 for clampXplus*/
    
    /**
     * If sensor returns a value between lockSensorValueB and lockSensorValueC,
     * the clamp is locked.
     */
    @ConfigurationParameter(range="0..32768")
    private Integer lockSensorValueB = 28800;
    
    /**
     * If sensor returns a value above lockSensorValueC, the sensor is in error.
     */
    @ConfigurationParameter(range="0..32768")
    private Integer lockSensorValueC = 32000;
    
    @ConfigurationParameter(range="0..32768")
    private Integer lockSensorOffset = 0;


    /**
     * if the filter presence sensor returns a value between 0 and
     * filterPresenceValueA : the sensor is in error.
     * 
     */
    @ConfigurationParameter(range="0..32768")
    private Integer filterPresenceValueA = 1500; /*1533 for clampXplus*/

    /**
     * if filter presence sensor returns a value between filterPresenceValueA
     * and filterPresenceValueB : the filter is engaged and lockable.
     */
    @ConfigurationParameter(range="0..32768")
    private Integer filterPresenceValueB = 24400; /*10500 for clampXplus in January 2016 (hardware)*/

    /**
     * if filter presence sensor returns a value between filterPresenceValueB
     * and filterPresenceValueC : we don't know. And if the value is greater
     * than valueC, there's no filter.
     */
    @ConfigurationParameter(range="0..32768")
    private Integer filterPresenceValueC = 32000;

    @ConfigurationParameter(range="0..32768")
    private Integer filterPresenceOffset;


    
    @ConfigurationParameter(range="0..10000")
    protected long timeoutForUnlocking = 1500;
    
    @ConfigurationParameter(range="0..10000")
    protected long timeoutForReleasing = 1500;

    /**
     * Build a new CarouselClampModule with a tickMillis value:5000;
     * @param filterPresenceSensor
     * @param lockSensor
     * @param thermometer 
     */
    public CarouselClampModule(
            Sensor14bits filterPresenceSensor,
            Sensor14bits lockSensor,
            Thermometer thermometer
        ){
        super(5000);
        this.filterPresenceOffset = 0;
        this.filterPresenceSensor = filterPresenceSensor;
        this.lockSensor = lockSensor;
        this.thermometer = thermometer;
    }



    /**
     * Returns the amount of current needed to unlock this clamp.
     * This is defined in the ConfigurationSystem.
     * @return 
     */
    public int getCurrentToUnlock() {
        return currentToUnlock;
    }

    /**
     * In the initialization phase, this method is used to initialize the controller.
     * cf initModule in CarouselSocket
     * @param actuator 
     */
    protected void setController(EPOSController actuator) {
        this.controller = actuator;
    }
    
    /**
     * @return the lockSensorValueA
     */
    public int getLockSensorValueA() {
        return lockSensorValueA;
    }

    /**
     * @return the lockSensorValueB
     */
    public int getLockSensorValueB() {
        return lockSensorValueB;
    }


    public int getLockSensorValueC() {
        return lockSensorValueC;
    }

    /**
     * @return the filterPresenceValueA
     */
    public int getFilterPresenceValueA() {
        return filterPresenceValueA;
    }

    /**
     * @return the filterPresenceValueB
     */
    public int getFilterPresenceValueB() {
        return filterPresenceValueB;
    }

    /**
     * @return the filterPresenceValueC
     */
    public int getFilterPresenceValueC() {
        return filterPresenceValueC;
    }

    /**
     * For encapsulation : we want to access to filterPresenceSensor in
     * sub-packages.
     *
     * @return the filterPresenceSensor
     */
    public Sensor14bits getFilterPresenceSensor() {
        return filterPresenceSensor;
    }

    /**
     * @return the lockSensor
     */
    public Sensor14bits getLockSensor() {
        return lockSensor;
    }


    /**
     * @return the lockStatus
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, 
        description = "return the lockStatus")
    public LockStatus getLockStatus() {
        return lockStatus;
    }

    /**
     * 
     * @return filterPresenceStatus
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, 
        description = "return filterPresenceStatus")
    public FilterPresenceStatus getFilterPresenceStatus() {
        return filterPresenceStatus;
    }
    
    

    /**
     * ***********************************************************************************************
     */
    /**
     * ******************** END OF SETTERS AND GETTERS  **********************************************
     */
    /**
     * ***********************************************************************************************
     */
    @Override
    public void initModule() {
        super.initModule();
        ComponentLookup lookup = getComponentLookup();
        this.bridge = (BridgeToHardware) lookup.getComponentByName("tcpProxy");
        this.autochanger = (FilterHolder) lookup.getComponentByName("autochanger");
        /* controller is initialized in initModule of CarouselSocket*/
        this.clampState = FilterClampState.UNDEFINED;
        this.filterPresenceStatus = FilterPresenceStatus.NOT_LOCKABLE;
        this.lockStatus = LockStatus.UNKNOWN;
        this.temperature = 0;

    }   
    

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, 
            description = "Returns true if CANopen hardware is connected and ready.")
    @Override
    public boolean isCANDevicesReady() {
        return ((MainModule) getComponentLookup().getComponentByName("main")).isCANDevicesReady();
    }
    


    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {
        
        EposMode mode;
        try {
            controller.initializeAndCheckHardware();
            mode = controller.readMode();
        } catch (FcsHardwareException ex) {
            FCSLOG.error(ex);
            String msg = name + " ERROR while reading EPOS controller mode :" + ex.getMessage();
            FCSLOG.error(msg);
            throw new HardwareException(false, ex);
        }
        if (mode != EposMode.CURRENT) {
            throw new HardwareException(false, name + " is not in CURRENT mode.");
        }
        return TreeWalkerDiag.HANDLING_CHILDREN;
    }

    /**
     * Returns the clampState of the clamp. If the clampState is being updated
     * and waiting for a response from a sensor, this methods waits until the
     * clampState is updated. If the clampState is not being updated, it returns
     * immediatly the clampState.
     *
     * @return clampState
     *
     */
    public FilterClampState getClampState() {

        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.warning(name + ": getClampState interrupted while waiting for update.");
                }

            }
            return clampState;

        } finally {
            lock.unlock();
        }
    }

    /**
     * This methods returns true if the clamp is locked. In the simulator, it is
     * overriden.
     *
     * @return true if the clamp is locked
     */
    @Command(type = Command.CommandType.QUERY,level = Command.ENGINEERING1, 
            description = "Returns true if the clamp is locked")
    public boolean isLocked() {
        return this.lockStatus == LockStatus.LOCKED;
    }

    /**
     *
     * @return true if filter is engaged on the clamp
     */
    @Command(type = Command.CommandType.QUERY,level = Command.ENGINEERING1, 
            description = "Returns true if filter is engaged on the clamp : filter presence sensors sees it ")
    public synchronized boolean isFilterEngaged() {
        return this.filterPresenceStatus == FilterPresenceStatus.LOCKABLE;
    }

    /**
     * Update the field lockPresenceStatus in reading the filter presence
     * sensor.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @deprecated Used only if we read new sensor value with a readSDO which is not the case for carousel clamps.
     * They are read from PDO.
     */
    @Deprecated
    private void updateFilterPresenceStatus()  {
        //READ filter presence sensor
        this.filterPresenceSensor.updateValue();
        int newFilterPresenceSensorValue = filterPresenceSensor.getValue();
        int mechaValue = newFilterPresenceSensorValue - this.filterPresenceOffset;
        if (mechaValue < this.filterPresenceValueA) {
            this.filterPresenceStatus = FilterPresenceStatus.ERROR;
            FCSLOG.error(getName() + " ERROR new read value FOR FILTER POSITION SENSOR  = "
                    + newFilterPresenceSensorValue);
            throw new SensorValueOutOfRangeException("FILTER POSITION SENSOR",
                    this.filterPresenceSensor.getName(), SENSOR14BITS_MIN,
                    this.filterPresenceValueA, mechaValue);

        } else if (mechaValue < this.filterPresenceValueB) {
            this.filterPresenceStatus = FilterPresenceStatus.LOCKABLE;

        } else if (mechaValue < this.filterPresenceValueC) {
            this.filterPresenceStatus = FilterPresenceStatus.NOT_LOCKABLE;

        } else {
            this.filterPresenceStatus = FilterPresenceStatus.NOFILTER;
        }
    }

    private void updateFilterPresenceStatus(PDOStorage pdoStorage)  {
        int newFilterPresenceSensorValue = filterPresenceSensor.updateValue(pdoStorage);
        int mechaValue = newFilterPresenceSensorValue - this.filterPresenceOffset;
        if (mechaValue < this.filterPresenceValueA) {
            this.filterPresenceStatus = FilterPresenceStatus.ERROR;
            FCSLOG.error(getName() + " ERROR new read value FOR FILTER POSITION SENSOR  = "
                    + newFilterPresenceSensorValue);
            throw new SensorValueOutOfRangeException("FILTER POSITION SENSOR",
                    this.filterPresenceSensor.getName(), SENSOR14BITS_MIN,
                    SENSOR14BITS_MAX, mechaValue);

        } else if (mechaValue < this.filterPresenceValueB) {
            this.filterPresenceStatus = FilterPresenceStatus.LOCKABLE;

        } else if (mechaValue < this.filterPresenceValueC) {
            this.filterPresenceStatus = FilterPresenceStatus.NOT_LOCKABLE;

        } else {
            this.filterPresenceStatus = FilterPresenceStatus.NOFILTER;
        }
    }

    /**
     * Update the field lockStatus in reading the lock sensor.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @deprecated Used only if we read new sensor value with a readSDO which is not the case for carousel clamps.
     */
    @Deprecated
    private void updateLockStatus()  {
        //READ lock sensor
        this.lockSensor.updateValue();
        computeLockStatus(lockSensor.getValue());
    }

    /**
     * Update lockStatus from parameter pdoStorage
     *
     * @param pdoStorage
     * @throws FcsHardwareException
     */
    private void updateLockStatus(PDOStorage pdoStorage)  {
        computeLockStatus(lockSensor.updateValue(pdoStorage));
    }
    
    /**
     * Compute the LockStatus from the new value read on the lock sensor.
     * @param newValue
     * @throws FcsHardwareException 
     */
    private void computeLockStatus(int newValue)  {
        int mechaValue = newValue - this.lockSensorOffset;
        if (mechaValue < lockSensorValueA) {
            this.lockStatus = LockStatus.UNLOCKED;

        } else if (mechaValue <= lockSensorValueB) {
            this.lockStatus = LockStatus.UNKNOWN;

        } else if (mechaValue <= lockSensorValueC) {
            this.lockStatus = LockStatus.LOCKED;

        } else {
            this.lockStatus = LockStatus.ERROR;
            FCSLOG.error(getName() + " ERROR new read value FOR LOCK SENSOR  = " + newValue);
            throw new FcsHardwareException(name+ ": ERROR in reading LOCK SENSOR - value read="
                    +mechaValue+" should be > "+lockSensorValueC);
        }
    }

    /**
     * This method updates the clamp clampState regarding the value returned by
     * the filter presence sensor and the value returned by the method
     * isLocked(). When the update is completed, it sends a signal to threads
     * waiting to get the new value of clampState.
     * 
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @deprecated 
     */
    @Deprecated
    public void updateStateWithSensorsFromSDO()  {

        lock.lock();
        updatingState = true;

        try {

            this.updateFilterPresenceStatus();

            this.updateLockStatus();

            this.computeClampState();

        } finally {
            updatingState = false;
            stateUpdated.signalAll();
            lock.unlock();
        }

        this.publishData();

    }

    /**
     * This method updates the clamp clampState regarding the value returned by
     * the filter presence sensor and the value returned by the method
     * isLocked(). When the update is completed, it sends a signal to threads
     * waiting to get the new value of clampState.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, alias = "updateState",
            description = "Read clamp sensors and update clampState")
    public void updateStateWithSensors()  {
        updateStateWithSensors(this.bridge.readPDOs());
    }

    /**
     * Update clampState in retreaving the value in a PDOStorage.
     * @param pdoStorage
     * @throws FcsHardwareException 
     */
    void updateStateWithSensors(PDOStorage pdoStorage)  {
        lock.lock();

        try {
            updatingState = true;

            this.updateFilterPresenceStatus(pdoStorage);

            this.updateLockStatus(pdoStorage);

            this.computeClampState();

        } finally {
            updatingState = false;
            stateUpdated.signalAll();
            lock.unlock();

        }
        this.publishData();
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     *
     */
    @Override
    public void publishData() {
        StatusDataPublishedByCarouselClamp status = this.createStatusDataPublishedByClamp();
        this.getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData(name, status));
    }
    
    
    /**
     * Create an object to be published on the STATUS bus.
     * @return 
     */
    public StatusDataPublishedByCarouselClamp createStatusDataPublishedByClamp() {
        StatusDataPublishedByCarouselClamp status = new StatusDataPublishedByCarouselClamp();
        status.setName(name);
        status.setClampState(clampState.toString());
        status.setFilterPresenceStatus(filterPresenceStatus);
        status.setFilterPositionSensorValue(filterPresenceSensor.getValue());
        status.setLockSensorValue(lockSensor.getValue());
        status.setLockStatus(lockStatus);
        status.setTemperature(temperature);
        return status;
    }

    /**
     * Compute the global state of the clamp given the lock sensor and the
     * presence filter sensor state. This has to be overriden for the clamp X-.
     *
     */
    public void computeClampState() {

        if (this.filterPresenceStatus == FilterPresenceStatus.ERROR
                || this.lockStatus == LockStatus.ERROR) {
            clampState = FilterClampState.ERROR;
            
        } else if (this.filterPresenceStatus == FilterPresenceStatus.LOCKABLE) {
        // a filter is in the socket

            if (this.lockStatus == LockStatus.LOCKED) {
                clampState = FilterClampState.CLAMPEDONFILTER;

            } else if (this.lockStatus == LockStatus.UNLOCKED) {
                clampState =  FilterClampState.UNCLAMPEDONFILTER;
            } else {
                clampState =  FilterClampState.UNDEFINED;
            }
            
        } else if (this.filterPresenceStatus == FilterPresenceStatus.NOFILTER) {
        // no filter
        
            if (this.lockStatus == LockStatus.LOCKED) {
                clampState =  FilterClampState.READYTOCLAMP;

            } else if (this.lockStatus == LockStatus.UNLOCKED) {
                clampState =  FilterClampState.UNCLAMPEDEMPTY;

            } else {
                clampState =  FilterClampState.ERROR;
            }
            
        } else {
            // NOT_LOCKABLE (UNDEFINED)
            clampState =  FilterClampState.UNDEFINED;
            
        }
    }
    
    /**
     * Because clampXminus is different from clampXplus the way of computing clampState is different when
     * we release clamps.
     * When a clampXminus is released, (clampController is off), its lockSensor doesn't detect a change,
     * so lockStatus stays UNLOCKED, therefore clampState doesn't change and stays UNCLAMPEDEMPTY.
     * When a clampXplus is released, its lockSensor detects the change, and lockStatus updates to LOCKED.
     * Therefore clampState updates to READYTOCLAMP.
     * So when we release a clamp, we have to recompute clampState for clampXminus.
     * 
     */
    private void computeClampStateWhenReleasing() {
        if (name.contains("Xminus") && this.clampState == FilterClampState.UNCLAMPEDEMPTY) {
            clampState = FilterClampState.READYTOCLAMP;
        }
    }
    



    /**
     * The clamps on the carousel are locked automaticaly when the filter comes
     * at the standby position. To be able to lock automaticaly again, it has to
     * be released after each time it has been unlocked.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws RejectedCommandException
     */
    @Command(level = Command.ENGINEERING3, type = Command.CommandType.ACTION,
            description = "Release clamp in order to get ready to clamp a filter again")
    public void release()  {
        FCSLOG.info("Checking conditions for release clamp " + getName() + " on socket at standby position.");

        updateStateWithSensors();
        if (clampState != FilterClampState.UNCLAMPEDEMPTY) {
            throw new RejectedCommandException("Can't release a clamp if isn't unclamped and empty.");
        }

        FCSLOG.info("Releasing clamp " + getName() + " on socket at standby position.");

        this.executeAction(MobileItemAction.RELEASE, timeoutForReleasing);

    }

    /**
     * Unlock the clamp when a filter is locked by the clamp. This is used only
     * in engineering mode.
     *
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, 
            description = "Unlock the clamp")
    public void unlock()  {

        FCSLOG.info(getName() + ": "
                + "UNLOCK State1 = " + clampState.toString());
        updateStateWithSensors();

        if (clampState != FilterClampState.CLAMPEDONFILTER) {
            throw new RejectedCommandException("Can't unlock a clamp if it is already unlocked.");
        }

        if (autochanger == null) {
            throw new RejectedCommandException("NO AUTOCHANGER");
        }

        if (autochanger.isAtStandby() && !autochanger.isHoldingFilter()) {
            throw new RejectedCommandException("CANNOT UNLOCK A CLAMP if FILTER is not HELD by autochanger.");
        }

        this.executeAction(MobileItemAction.UNLOCK, timeoutForUnlocking);

    }
    


    @Override
    public void startAction(MobileItemAction action)  {
        if (action == MobileItemAction.UNLOCK) {
            this.controller.enable();
            this.controller.writeCurrent(this.currentToUnlock);
            
        } else if (action == MobileItemAction.RELEASE) {
            this.controller.off();
        } else {
            throw new IllegalArgumentException("Action on clamp must be UNLOCK "
                    + "or RELEASE");
        }

    }

    @Override
    public boolean isActionCompleted(MobileItemAction action) {
        if (action == MobileItemAction.UNLOCK) {
            return this.clampState == FilterClampState.UNCLAMPEDONFILTER;
            
        } else if (action == MobileItemAction.RELEASE) {
            computeClampStateWhenReleasing();
            return this.clampState == FilterClampState.READYTOCLAMP;
            
        } else {
            throw new IllegalArgumentException("Action on clamp must be UNLOCK "
                    + "or RELEASE");
        }
    }

    @Override
    public void postAction(MobileItemAction action) {
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        this.updateStateWithSensors();
    }

    /**
     * This methods read the thermometer, update the field temperature and
     * returns the value sent by the thermometer;
     *
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public double updateTemperature()  {
        double temp = this.thermometer.readTemperature();
        this.temperature = temp;
        return temp;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3, 
            description = "display configuration values for this clamp.")
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(this.getName());
        sb.append("/filterPresenceSensor=");
        sb.append(this.filterPresenceSensor.getName());
        sb.append("#valueA=");
        sb.append(this.filterPresenceValueA);
        sb.append("#valueB=");
        sb.append(this.filterPresenceValueB);
        sb.append("#valueC=");
        sb.append(this.filterPresenceValueC);
        sb.append("/lockSensor=");
        sb.append(this.lockSensor.getName());
        sb.append("#valueA=");
        sb.append(this.lockSensorValueA);
        sb.append("#valueB=");
        sb.append(this.lockSensorValueB);
        sb.append("#valueC=");
        sb.append(this.lockSensorValueC);
        return sb.toString();
    }

    //TODO check with Guillaume what is to be done to stop the actions.
    @Override
    public void abortAction(MobileItemAction action, long delay)  {
        FCSLOG.info("Current Command: "+getSubsystem().getCurrentAction()+" "+getSubsystem().getState());       
    }

    @Override
    public void quickStopAction(MobileItemAction action, long delay)  {
        throw new UnsupportedOperationException("Not supported yet."); 
    }

}
