
package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import org.lsst.ccs.bus.data.KeyValueData;

import org.lsst.ccs.command.annotations.Command;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NO_FILTER;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.AC_SENSOR_ERROR;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * This class represents a model for the 2 Latches of the autochanger. The
 * Latches hold a filter on the trucks. There is one latch on each side of the
 * filter. Because the ACTION open (or close) for 2 latches is different from
 * opening (or closing) one Latch, then the other one, it's more convenient to
 * have a class to represent 2 Latches at a time.
 *
 * @author virieux
 */
public class AutochangerTwoLatches extends MobileItemModule {

    private AutoChangerModule autochanger;
    private final AutochangerLatchModule latchXminus;
    private final AutochangerLatchModule latchXplus;

    private volatile LockStatus lockStatus = LockStatus.UNKNOWN;
    private volatile FilterPresenceStatus filterPresenceStatus = FilterPresenceStatus.UNKNOWN;
    private final long timeoutForClosing;
    private final long timeoutForOpening;

    private final Condition stateUpdated = lock.newCondition();

    /* This is used when we update the latches state with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;

    /**
     * Build an Object AutochangerTwoLatches from 2 AutochangerLatchModule.
     * @param latchXminus
     * @param latchXplus 
     */
    public AutochangerTwoLatches(
            AutochangerLatchModule latchXminus,
            AutochangerLatchModule latchXplus) 
    {
        this.latchXminus = latchXminus;
        this.latchXplus = latchXplus;
        this.timeoutForClosing = Math.max(latchXminus.getTimeoutForClosing(),
                latchXplus.getTimeoutForClosing());
        this.timeoutForOpening = Math.max(latchXminus.getTimeoutForOpening(),
                latchXplus.getTimeoutForOpening());
    }

    /**
     * Returns true if the 2 latches controllers are initialized.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if the 2 latches are initialized.")
    public boolean isInitialized() {
        return latchXminus.isInitialized() && latchXplus.isInitialized();
    }

    /**
     * return filterPresenceStatus
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, description= "return filterPresenceStatus")
    public FilterPresenceStatus getFilterPresenceStatus() {
        return filterPresenceStatus;
    }



    /**
     * Return the max time for closing in millisecond.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return the max time for closing in millisecond.")
    public long getTimeoutForClosing() {
        return timeoutForClosing;
    }

    /**
     * Return the max time for opening.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return the max time for opening in millisecond.")    
    public long getTimeoutForOpening() {
        return timeoutForOpening;
    }
    
    

    /**
     * Return if both latches are LOCKED.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if both latches are LOCKED.")
    public boolean isLocked() {
        return this.latchXminus.getLockStatus() == LockStatus.LOCKED
                && this.latchXplus.getLockStatus() == LockStatus.LOCKED;
    }
    
    /**
     * Return if both latches are UNLOCKED.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if both latches are UNLOCKED.")
    public boolean isUnlocked() {
        return this.latchXminus.getLockStatus() == LockStatus.UNLOCKED
                && this.latchXplus.getLockStatus() == LockStatus.UNLOCKED;
    }
    
    /**
     * Returns true if one of the latches is in ERROR state.
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
        description = "Returns true if one of the latches is in ERROR state.")
    public boolean isInError() {
        return this.latchXminus.getLockStatus() == LockStatus.ERROR
                || this.latchXplus.getLockStatus() == LockStatus.ERROR;
    }
    
    /**
     * 
     * @return true if both latches are in Travel
     */
    private boolean isInTravel() {
        return this.latchXminus.getLockStatus() == LockStatus.INTRAVEL
                && this.latchXplus.getLockStatus() == LockStatus.INTRAVEL;
    }
    
    /**
     * Returns true if there is no filter is latches.
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
        description = "Returns true if there is no filter is latches.")
    public boolean isEmpty() {
        return this.latchXminus.isEmpty() && this.latchXplus.isEmpty();
    }
    
    /**
     * Return true if autochanger is holding a filter.
     * A filter is ENGAGED and latches are LOCKED.
     * Does not read again sensors values.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return true if autochanger is holding a filter.")
    public boolean isHoldingFilter() {
        return isFilterEngaged() && this.isLocked();
    }
    
    /**
     * 
     * @return true if both latches detect filter presence.
     */
    private boolean isFilterEngaged() {
        return !this.latchXminus.isEmpty() && !this.latchXplus.isEmpty();
    }

    @Override
    public void initModule() {
        super.initModule();
        this.autochanger = (AutoChangerModule) getComponentLookup().getComponentByName("autochanger");

    }

    @Override
    public boolean isCANDevicesReady() {
        return autochanger.isCANDevicesReady() && isInitialized();
    }
    
    /**
     * Checks sensors.
     * @throws FcsHardwareException if some latch sensors are in error or
     * if latches state is not equal on both side of the filter.
     * @throws FcsHardwareException
     */
    public void updateStateAndCheckSensors() {
        autochanger.updateStateWithSensors();
        latchXminus.checkSensors(AC_SENSOR_ERROR);
        latchXplus.checkSensors(AC_SENSOR_ERROR);
        if (latchXminus.getLockStatus() != latchXplus.getLockStatus()) {
            String msg = getName() + " sensors are in ERROR: latchXminus state="+latchXminus.getLockStatus()
                +" latchXplus state="+ latchXplus.getLockStatus();
            this.raiseAlarm(AC_SENSOR_ERROR, msg);
            throw new FcsHardwareException(getName() + " LockStatus is different on both side. Can't close neither open latches.");
        }
        if (latchXminus.isEmpty() != latchXplus.isEmpty()) {
            String msg = getName()+ " Error in filterPresenceSensors of autochanger : the 2 sensors send different values."
                    + " latchXminus.isEmpty="+this.latchXminus.isEmpty()
                    + " latchXplus.isEmpty="+this.latchXplus.isEmpty();
            this.raiseAlarm(AC_SENSOR_ERROR, msg);
            throw new FcsHardwareException(msg + "Can't close neither open latches.");
        }
    }


    /**
     * Update latches lockStatus from an array of hexa values read from the
     * sensors.
     *
     * @param readHexaValues
     */
    void updateStateWithSensors(String[] readHexaValues) {
        lock.lock();

        try {
            updatingState = true;
            
            this.latchXminus.updateStateWithSensors(readHexaValues);
            this.latchXplus.updateStateWithSensors(readHexaValues);

            computeLockStatus();

            computeFilterPresenceStatus();

        } finally {

            updatingState = false;
            stateUpdated.signalAll();
            lock.unlock();
            this.publishData();
        }
    }
    
    /**
     * Compute a global lock status for the 2 latches.
     */
    private void computeLockStatus() {

        if (isInError()) { 
            this.lockStatus = LockStatus.ERROR;
            
        } else if (isLocked()) {
            this.lockStatus = LockStatus.LOCKED;
            
        } else if (isUnlocked()) {
            this.lockStatus = LockStatus.UNLOCKED;
            
        } else if (isInTravel()) {
            this.lockStatus = LockStatus.INTRAVEL;
            
        } else {
            /* if lockStatus is not the same for the 2 latches*/
            /* can happen when closing (or opening) when a latch is already closed and not the other*/
            this.lockStatus = LockStatus.UNKNOWN;
        }
    }
    
    private void computeFilterPresenceStatus() {
        /*compute filter presence*/
        if (this.latchXminus.isEmpty() != this.latchXplus.isEmpty()) {
            filterPresenceStatus =  FilterPresenceStatus.ERROR;


        } else if (this.isEmpty()) {
            filterPresenceStatus = FilterPresenceStatus.NOFILTER;

        } else if (isFilterEngaged()) {
            filterPresenceStatus = FilterPresenceStatus.ENGAGED;

        } else {
            filterPresenceStatus = FilterPresenceStatus.UNKNOWN;
        }
    }
    

    
    

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted()  {
        autochanger.updateStateWithSensors();
    }
    
    /**
     * Close latches
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Close latches.")
    public void close()  {
        FCSLOG.info(getName() + " about to close.");
        autochanger.updateStateWithSensors();

        if (lockStatus == LockStatus.LOCKED) {
            throw new RejectedCommandException(getName() + " are already LOCKED.");
        }
        autochanger.checkConditionsForClosingLatches();
        this.executeAction(FcsEnumerations.MobileItemAction.CLOSELATCHES,timeoutForClosing);
    }
    
    
    /**
     * Open the 2 latches.
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Open both latches.")
    public void open()  {

        if (latchXminus.isUnlocked() && latchXplus.isUnlocked()) {
            throw new RejectedCommandException(getName() + " are already UNLOCKED.");
        }
        autochanger.checkConditionsForOpeningLatches();
        this.executeAction(FcsEnumerations.MobileItemAction.OPENLATCHES,timeoutForOpening);
    }
    
    @Override
    public void startAction(FcsEnumerations.MobileItemAction action)  {
        
        switch (action) {
            case OPENLATCHES:
                latchXminus.getLatchController().enable();
                latchXplus.getLatchController().enable();
                latchXminus.getLatchController().writeCurrent(-latchXminus.getCurrentToClose());
                latchXplus.getLatchController().writeCurrent(-latchXplus.getCurrentToClose());
                break;

            case CLOSELATCHES:
                latchXminus.getLatchController().enable();
                latchXplus.getLatchController().enable();
                latchXminus.getLatchController().writeCurrent(latchXminus.getCurrentToClose());
                latchXplus.getLatchController().writeCurrent(latchXplus.getCurrentToClose());
                break;
                
            default:
                assert false: action;
        }
    }

    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay)  {
        FCSLOG.debug(getName() + " is ABORTING action " + action.toString()
                + " within delay " + delay);
        this.latchXminus.getLatchController().off();
        this.latchXplus.getLatchController().off();
    }

    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay)  {
        throw new UnsupportedOperationException("Not supported yet."); 
    }

    @Override
    public void postAction(FcsEnumerations.MobileItemAction action)  {
        FCSLOG.finest(getName() + " nothing to do in postAction");
    }
    
    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        switch (action) {
            case OPENLATCHES:
                return latchXminus.isUnlocked() && latchXplus.isUnlocked();

            case CLOSELATCHES:
                return latchXminus.isLocked() && latchXplus.isLocked();
                
            default:
                assert false: action;
        }
        return false;
    }

    /**
     * Creates an object to be published on the STATUS bus by AutochangerTwoLatches.
     * @return 
     */
    public StatusDataPublishedByAutochangerTwoLatches createStatusDataPublishedByTwoLatches() {
        StatusDataPublishedByAutochangerTwoLatches status = new StatusDataPublishedByAutochangerTwoLatches();
        status.setName(getName());
        status.setLockStatus(lockStatus);
        status.setFilterPresenceStatus(filterPresenceStatus);
        String filterName;
        if (autochanger.getFilterOnTrucksName() == null) {
            filterName = NO_FILTER;
        } else {
            filterName = autochanger.getFilterOnTrucksName();
        }
        status.setFilterName(filterName);
        return status;
    }

    @Override
    public void publishData() {
        this.getSubsystem().publishSubsystemDataOnStatusBus(
                new KeyValueData("autochangerLatches", createStatusDataPublishedByTwoLatches()));
    }

}
