
package org.lsst.ccs.subsystems.fcs;

import java.util.Observable;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.MOVE_TO_ABSOLUTE_POSITION;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerWithBrake;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;

/**
 * A model for an autochanger truck. 
 * The goal of this class is to factorise the code which reads AC truck sensors.
 * On the Autochanger there are 2 trucks : one on each side : 
 * truckXminus and truckXplus.
 * @author virieux
 */
public class AutochangerTruckModule extends MobileItemModule implements MovedByEPOSController {
    
    //Used because we have to wait for the update from the sensors to know if the action is completed.
    protected final Lock lock = new ReentrantLock();
    
    private final ComplementarySensors handoffPositionSensors;
    private final ComplementarySensors onlinePositionSensors;
    private final ComplementarySensors standbyPositionSensors;
    
    private final EPOSControllerWithBrake controller;
        
    private int position;

    private final Condition stateUpdated = lock.newCondition();

    /* This is used when we update the clamp clampState with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;
    
    private boolean controllerInFault;

    private int absoluteTargetPosition = 0;
    private int relativeTargetPosition = 0;
    
    /**
     * Builds a AutochangerTruckModule with 3 couple of complementary sensors 
     * to detect position of the truck at ONLINE, HANDOFF and STANDBY position.
     * @param controller
     * @param handoffPositionSensors
     * @param onlinePositionSensors
     * @param standbyPositionSensors 
     */
    public AutochangerTruckModule(
            EPOSControllerWithBrake controller,
            ComplementarySensors handoffPositionSensors, 
            ComplementarySensors onlinePositionSensors, 
            ComplementarySensors standbyPositionSensors) {
        this.controller = controller;
        this.handoffPositionSensors = handoffPositionSensors;
        this.onlinePositionSensors = onlinePositionSensors;
        this.standbyPositionSensors = standbyPositionSensors;        
    }
    
    
    @Override
    public void initModule() {
        /**
         * Because AutochangerOnlineClampModule implements MovedByEPOSController, it has to listen to its controller
         * to known when the controller is in Fault or when a Fault reset has been done.
         */
        if (controller == null) {
            FCSLOG.error(getName() + "==>>> controller == null - Please fix groovy description file.");
            throw new IllegalArgumentException(getName() + "==>>> null controller - fix groovy description file.");
        } else {
            //listens to my Controller to detect the controller's faultReset
            //or the emergency messages coming from the controller.
            if (controller instanceof Observable) {
                this.listens((Observable) controller);
            }
        }
    }

    /**
     * @return true if truck is at HANDOFF position.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return true if truck is at HANDOFF position.")    
    public boolean isAtHandoff() {
        return handoffPositionSensors.isOn();
    }
    
    /**
     * @return true if truck is at ONLINE position.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return true if truck is at ONLINE position.")
    public boolean isAtOnline() {
        return onlinePositionSensors.isOn();
    }

    /**
     * @return true if truck is at STANDBY position.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return true if truck is at STANDBY position.")    
    public boolean isAtStandby() {
        return standbyPositionSensors.isOn();
    }
    
    

    /**
     * Return false if the 2 redondant position sensors at Standby are equal.
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return false if the 2 redondant position sensors at Standby are equal. "
                    + "Doesn't read again sensors.")
    public boolean isStandbySensorsInError() {
        return standbyPositionSensors.isInError();
    }

    /**
     * Return false if the 2 redondant position sensors at Handoff are equal.
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return false if the 2 redondant position sensors at Handoff are equal."
                    + "Doesn't read again sensors.")
    public boolean isHandoffSensorsInError() {
        return handoffPositionSensors.isInError();
    }

    /**
     * Return false if the 2 redondant position sensors at Online are equal.
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return false if the 2 redondant position sensors at Online are equal."
                    + "Doesn't read again sensors.")
    public boolean isOnlineSensorsInError() {
        return onlinePositionSensors.isInError();
    }
    
    /**
     * Return true if position sensors are in error.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
        description = "Return true if position sensors are in error."
                + "Doesn't read again sensors.")
    public boolean isPositionSensorsInError() {
        return isOnlineSensorsInError() || isHandoffSensorsInError() || isStandbySensorsInError();
    }   

    /**
     * Return truck position enregistred in field position.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "return truck position.")    
    public int getPosition() {
        return position;
    }
    
    /**
     * @return controller name.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "return controller name.")    
    @Override
    public String getControllerName() {
        return this.controller.getName();
    }

    /**
     * Return true if controller is in fault.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return true if the controller is in fault.")    
    @Override
    public boolean isControllerInFault() {
        return controllerInFault;
    }

    @Override
    public void setControllerInFault(boolean controllerInFault) {
        this.controllerInFault = controllerInFault;
    }
    
    /**
     * Updates the field position of the truck in reading the CPU of the
     * controller.
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update truck position in reading controller.")
    public void updatePosition()  {
        this.position = controller.readPosition();
        this.publishData();
    }   

    
    
    /**
     * This methods updates the lockStatus from an array of hexaValues.
     *
     * @param readHexaValues
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    //TODO test with real hardware
    protected void updateStateWithSensors(String[] readHexaValues) {
        lock.lock();

        try {
            updatingState = true;
            this.handoffPositionSensors.updateValues(readHexaValues);
            this.onlinePositionSensors.updateValues(readHexaValues);
            this.standbyPositionSensors.updateValues(readHexaValues);

        } finally {
            updatingState = false;
            stateUpdated.signalAll();
            lock.unlock();
            this.publishData();
        }
    }
    
    /**
     * Updates field controllerInFault when the controller notifies its observers and sends new values.
     * This class observes its controller to publish data when its controller is in fault or after a faultReset. 
     * Needed to update the GUI.
     *
     * @param source
     * @param v
     */
    @Override
    public void processUpdate(Observable source, ValueUpdate v) {
        updateControllerInFault(((EPOSController) source).getName(),v);
    }
    
    /**
     * Creates an object to be published on the status bus.
     * @return 
     */
    public StatusDataPublishedByAutochangerTruck createStatusDataPublishedByAutoChangerTruck() {
        StatusDataPublishedByAutochangerTruck s = new StatusDataPublishedByAutochangerTruck();
        s.setControllerName(this.getControllerName());
        s.setHandoffInError(handoffPositionSensors.isInError());
        s.setOnlineInError(onlinePositionSensors.isInError());
        s.setStandbyInError(standbyPositionSensors.isInError());
        s.setHandoffSensorValue(handoffPositionSensors.isOn());
        s.setOnlineSensorValue(onlinePositionSensors.isOn());
        s.setStandbySensorValue(standbyPositionSensors.isOn());
        s.setPosition(position);
        s.setControllerInFault(controllerInFault);
        return s;
    }

    @Override
    public void publishData() {
        this.getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData(getName(), 
                createStatusDataPublishedByAutoChangerTruck()));
    }    
    
    public void moveToAbsoluteTargetPosition(int targetPosition)  {
        
        this.updatePosition();
        if (position == targetPosition) {
            FCSLOG.info(getName()
                    + " is already at target position=" + targetPosition);
        } else {

            this.absoluteTargetPosition = targetPosition;

            FCSLOG.info(getName() + " going to absolute position: " + absoluteTargetPosition);

            this.executeAction(MOVE_TO_ABSOLUTE_POSITION, 120000);
        }
    }

    @Override
    public boolean isCANDevicesReady() {
        return controller.isInitialized();
    }

    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
                switch (action) {
            case MOVE_TO_ABSOLUTE_POSITION:
                return this.position == this.absoluteTargetPosition;

            case MOVE_TO_RELATIVE_POSITION:
                return this.position == this.relativeTargetPosition;
                
            default :
                assert false: action;
        }
        return false;
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        try {
            controller.checkFault();
            this.position = this.controller.readPosition();
            FCSLOG.debug(getName() + " position=" + this.position);
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(getName() + "=> SDO ERROR IN READING CONTROLLER:",ex);
        } catch (SDORequestException ex) {
            FCSLOG.error(getName() + "=> SDO ERROR IN READING CONTROLLER:",ex);
        }
    }

    @Override
    //tested on CPPM testbench in september 2015
    public void startAction(FcsEnumerations.MobileItemAction action)  {
        
        switch (action) {
            case MOVE_TO_ABSOLUTE_POSITION:
                controller.enableAndWriteAbsolutePosition(this.absoluteTargetPosition);
                break;

            case MOVE_TO_RELATIVE_POSITION:
                controller.enableAndWriteRelativePosition(this.relativeTargetPosition);
                break;
                
            default:
                assert false : action;
        }
    }

    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public void postAction(FcsEnumerations.MobileItemAction action) {
        this.controller.activateBrakeAndDisable();
    }
    
}
