
package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.AC_SENSOR_ERROR;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.ControlledBySensors;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * The Three Online clamps which holds a filter when it is at ONLINE position
 * can ve viewed as a single object. It's the goal of this class to represent
 * this object. Opening or closing the 3 clamps at ONLINE is not the same action
 * as opening or closing online clamps one by one.
 *
 * @author virieux
 */
public class AutochangerThreeOnlineClamps extends MobileItemModule implements ControlledBySensors {
    
    private AutoChangerModule autochanger;
    private final AutochangerOnlineClampModule onlineClampXminus;
    private final AutochangerOnlineClampModule onlineClampXplus;
    private final AutochangerOnlineClampModule onlineClampYminus;
    private final AutochangerOnlineClampModule[] clampsList;

    private FcsEnumerations.LockStatus lockStatus = LockStatus.UNKNOWN;


    private final Condition stateUpdated = lock.newCondition();

    /* This is used when we update the clamps
     state with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;
    
    /* Tools for current ramp*/
    private ScheduledFuture<?> currentRampHandle;
    protected final Condition currentRampEnded = lock.newCondition();
    protected boolean hasToWaitForEndOfCurrentRamp;
    /* end of Tools for current ramp*/

    @ConfigurationParameter(description="timeout in milliseconds : if closing the clamps last more "
            + "than this amount of time, then the subsystem goes in ERROR.")
    private long timeoutForLockingClamps = 15000;
    
    @ConfigurationParameter(description="timeout in milliseconds : if unlocking the clamps last more "
            + "than this amount of time, then the subsystem goes in ERROR.")
    private long  timeoutForUnlockingClamps = 15000;

    /**
     * Create a AutochangerThreeOnlineClamps with 3 AutochangerOnlineClampModule.
     * @param onlineClampXminus
     * @param onlineClampXplus
     * @param onlineClampYminus
     */
    public AutochangerThreeOnlineClamps(
            AutochangerOnlineClampModule onlineClampXminus,
            AutochangerOnlineClampModule onlineClampXplus,
            AutochangerOnlineClampModule onlineClampYminus) {
        this.onlineClampXminus = onlineClampXminus;
        this.onlineClampXplus = onlineClampXplus;
        this.onlineClampYminus = onlineClampYminus;
        clampsList = new AutochangerOnlineClampModule[] {onlineClampXminus, onlineClampXplus, onlineClampYminus};
    }

    /**
     * Return lockStatus
     * If lockStatus is being updated, wait until update completion else return immediately lockStatus.
     * @return 
     */
    public FcsEnumerations.LockStatus getLockStatus() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.warning(getName() + ": interrupted in getLockStatus.",ex);
                }

            }
            return lockStatus;

        } finally {
            lock.unlock();
        }
    }
    
    /**
     * Returns true if LockStatus=LOCKED
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if the 3 clamps are CLOSED.")
    public boolean isLocked() {
        return onlineClampXminus.isLocked()
                //commented out in January 2017 because sensors of clampXplus are dammaged.
//                && onlineClampXplus.isLocked()
                && onlineClampYminus.isLocked();
    }
 
    /**
     * Returns true if LockStatus=CLOSED
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if the 3 clamps are CLOSED.")    
    public boolean isClosed() {
        return onlineClampXminus.isClosed()
                //commented out in January 2017 because sensors of clampXplus are dammaged.
//                && onlineClampXplus.isClosed()
                && onlineClampYminus.isClosed();
    }
    
    /**
     * Returns true if LockStatus=OPENED
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if the 3 clamps are OPENED.")
    public boolean isOpened() {
        return onlineClampXminus.isOpened()
                //commented out in January 2017 because sensors of clampXplus are dammaged.
//                && onlineClampXplus.isOpened()
                && onlineClampYminus.isOpened();
    }
    
    /**
     * @return true if the 3 clamps are in Travel between opened position and closed position.
     */
    private boolean isInTravel() {
        return onlineClampXminus.isInTravel()
                && onlineClampXplus.isInTravel()
                && onlineClampYminus.isInTravel();
    }
    
    /**
     * Returns true if LockStatus=ERROR
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if one of the clamp is in error.")
    @Override
    public boolean isInError() {
        return onlineClampXminus.isInError()
                || onlineClampXplus.isInError()
                || onlineClampYminus.isInError();
    }
    
    

    @Override
    public void initModule() {
        this.autochanger = (AutoChangerModule) getComponentLookup().getComponentByName("autochanger");
        this.lockStatus = FcsEnumerations.LockStatus.UNKNOWN;
    }
    
    /**
     * Return true if the 3 onlineClamps hardware is initialized. 
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
        description = "Return true if the 3 onlineClamps hardware is ready : controllers ready, "
                + "controllers parameters checked and controllers configured.")
    public boolean isInitialized() {
        return onlineClampXminus.isInitialized() && onlineClampXplus.isInitialized() 
                && onlineClampYminus.isInitialized();
    }
    
    /**
     * Check that the 3 controllers of the clamps are initialized and 
     * that parameters are correct.
     */
    private void enableAndCheckControllersBeforeAction() {
        for (AutochangerOnlineClampModule clamp: clampsList) {
            clamp.enableAndCheckControllerBeforeAction();
        }
    }
    
    
    /**
     * Closes and locks the online clamps.
     * This command first release brakes, sends currentToClose to the 3 clamps controllers 
     * and then increases progressively current until currentToClamp is reached for every 
     * clamps controllers.
     * 
     * 
     * state before this action : OPENED or CLOSED (if CLOSED, nothing has to be done)
     * state after this action : LOCKED
     * 
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
            description = "Closes and locks the online clamps.")
    public void closeAndLockClamps()  {
        updateStateAndCheckSensors();
        //TODO test conditions for closing clamps
        if (isOpened()) {
            /*See startAction */
            this.executeAction(MobileItemAction.CLOSE_AND_LOCK_ONLINECLAMPS, timeoutForLockingClamps);
            
        } else if (isLocked()) {
            FCSLOG.info(getName() + " is already LOCKED - nothing to do.");
            
        } else throw new RejectedCommandException(getName() + " has to be OPENED before a closeAndLockClamps");
    }

    /**
     * Unlock and open the online clamps.
     * initial state = LOCKED
     * final state = OPENED
     * This action is not 
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
            description = "Unlock and open the online clamps.")
    public void unlockAndOpenClamps()  {
        updateStateAndCheckSensors(); 
        //TODO test conditions for unlocking clamps
        if (this.isLocked()) {
            /*See startAction */
            this.executeAction(MobileItemAction.UNLOCK_AND_OPEN_ONLINECLAMPS, timeoutForUnlockingClamps);
            
        } else if (this.isClosed()) {
            FCSLOG.info(getName()+ " clamps already UNLOCKED and CLOSED - nothing to do");
            
        } else {
            throw new RejectedCommandException(getName() + " has to be LOCKED before unlockAndOpen action.");
        }
    }
    
    
     /**
     * Closes the 3 clamps with a small pressure. 
     * initial state = OPENED
     * final state = CLOSED
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
        description = "Close the 3 online clamps.")   
    public void closeClamps() {
        updateStateAndCheckSensors();
        //TODO test conditions for closing clamps
        if (isOpened()) {
            /*See startAction */
            this.executeAction(MobileItemAction.CLOSE_ONLINECLAMPS, timeoutForLockingClamps);
            
        } else if (isClosed()) {
            FCSLOG.info(getName()+ " clamps already CLOSED nothing to do");
            
        } else throw new RejectedCommandException(getName() + " has to be unlocked before.");
    }
    
    
    /**
     * Opens the 3 clamps with a small pressure.
     * initial state = CLOSED
     * final state = OPENED
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
        description = "Opens the 3 online clamps with a small pressure.")    
    public void openClamps() {
        updateStateAndCheckSensors();
        //TODO test conditions for opening clamps
        if (isClosed()) {
            this.executeAction(MobileItemAction.OPEN_ONLINECLAMPS, timeoutForUnlockingClamps);
            
        } else if (isOpened()) {
            FCSLOG.info(getName()+ " clamps already OPENED. Nothing to do");
            
        } else throw new RejectedCommandException(getName() + " has to be closed before.");
    }
    
    /**
     * Locks clamps : closed with a strong pressure (high current).
     * The clamps have to be CLOSED.
     * At the end of this action, the clamps are CLOSED but a strong pressure to hold safely the clamps.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
        description = "Locks clamps : closed with a strong pressure (high current).")    
    public void lockClamps() {
        updateStateAndCheckSensors();
        if (isClosed()) {
            this.executeAction(MobileItemAction.LOCK_ONLINECLAMPS, timeoutForLockingClamps);
            
        } else if (isLocked()) {
            FCSLOG.info(getName() + " is already LOCKED. Nothing to do.");
            
        } else {
            throw new RejectedCommandException(getName() + " have to be CLOSED before to be locked.");
        } 
    }
    
     /**
     * Unlocks clamps : slows down current sent to controller in order to decrease pressure on
     * the clamps.
     * The clamps have to be LOCKED.
     * At the end of this action, the clamps are CLOSED with a small pressure of the clamp hardware on the filter frame.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
        description = "unlocks clamps : decreases current in controller to decrease pressure.")     
    public void unlockClamps() {
        updateStateAndCheckSensors();
        if (isLocked()) {
            this.executeAction(MobileItemAction.UNLOCK_ONLINECLAMPS, timeoutForLockingClamps);
            
        } else if (isClosed()) {
            FCSLOG.info(getName() + " is already CLOSED. Nothing to do.");
            
        } else {
            throw new RejectedCommandException(getName() + " have to be CLOSED before to be locked.");
        }
    }   
    
    /**
     * Sends current to all controllers to close clamps and save sentCurrent value.
     * Controllers have been enabled and brakes released when this method is called.
     * used for actions CLOSE_AND_LOCK_ONLINECLAMPS and CLOSE_ONLINECLAMPS
     * see startAction
     */
    private void sendCurrentToCloseAllClamps() {
        for (AutochangerOnlineClampModule clamp: clampsList) {
            clamp.sendCurrentToControllerAndSaveValue(clamp.getCurrentToClose());
        } 
    }
    
    /**
     * Sends current to open to all controllers in order to open clamps and save sentCurrent value.
     * Controllers have been enabled and brakes released when this method is called.
     * used for action OPEN_ONLINE_CLAMPS
     */
    private void sendCurrentToOpenAllClamps() {
        for (AutochangerOnlineClampModule clamp: clampsList) {
            clamp.sendCurrentToControllerAndSaveValue(clamp.getCurrentToOpen());
        }  
    }

    /**
     * Return true if the 3 onlineClamps hardware is ready. 
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
        description = "Return true if the 3 onlineClamps hardware is ready.")
    @Override
    public boolean isCANDevicesReady() {
        return onlineClampXminus.isCANDevicesReady() && onlineClampXplus.isCANDevicesReady() 
                && onlineClampYminus.isCANDevicesReady();
    }

    /**
     * returns true if the action given as argument is completed. 
     * The completion of action has been confirmed by sensors.
     * @param action
     * @return 
     */
    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        if (action == MobileItemAction.CLOSE_AND_LOCK_ONLINECLAMPS || action == MobileItemAction.CLOSE_ONLINECLAMPS
                || action == MobileItemAction.UNLOCK_ONLINECLAMPS) {
            /* for action CLOSE_AND_LOCK_ONLINECLAMPS, lockStatus is set to LOCKED in postAction*/
            /* but the action started in method startAction is completed when clamps are CLOSED*/
            return isClosed(); 
            
        } else if (action == MobileItemAction.UNLOCK_AND_OPEN_ONLINECLAMPS || action == MobileItemAction.OPEN_ONLINECLAMPS) {
            return isOpened();
            
        } else if (action == MobileItemAction.LOCK_ONLINECLAMPS) {
            return isLocked(); 
            
        } else {
            throw new IllegalArgumentException(getName() + " invalid action:"+action);
        }
    }

    /**
     * reads sensors, computes new lockStatus.
     * used in MobileItemModule to check that action started in startAction is completed.
     */
    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted()  {
        autochanger.updateStateWithSensors();
    }

    @Override
    public void startAction(FcsEnumerations.MobileItemAction action)  {
        enableAndCheckControllersBeforeAction();
        autochanger.checkConditionsForActioningOnlineClamps();

        if (action == MobileItemAction.CLOSE_AND_LOCK_ONLINECLAMPS || action == MobileItemAction.CLOSE_ONLINECLAMPS) {
            /* for action CLOSE_AND_LOCK_ONLINECLAMPS, first we close and then we lock, the locking is done in postAction */
            /* clamps are OPENED : to close we must send currentToClose to the 3 clamps*/
            releaseBrakes();
            sendCurrentToCloseAllClamps();
            
        } else if (action == MobileItemAction.UNLOCK_AND_OPEN_ONLINECLAMPS) {
            /* clamps are LOCKED with a high pressure (currentToClamp previously sent to controllers)*/
            /* to avoid an elastic return we send currentToClamp BEFORE the brakes are released. */
            sendPreviouslySentCurrentAndReleaseBrakes();
            
            //go to 0 with a ramp on the 3 clamps.
            executeCurrentRampOnAllClamps(onlineClampYminus.getSentCurrent(),onlineClampXminus.getSentCurrent(),
                    onlineClampXplus.getSentCurrent(),0,0,0, 5);

            onlineClampXminus.sendCurrentToControllerAndSaveValue(onlineClampXminus.getCurrentToOpen());
            onlineClampXplus.sendCurrentToControllerAndSaveValue(onlineClampXplus.getCurrentToOpen());
            onlineClampYminus.sendCurrentToControllerAndSaveValue(onlineClampYminus.getCurrentToOpen());
            
        } else if (action == MobileItemAction.UNLOCK_ONLINECLAMPS) {
            /* clamps are LOCKED with a high pressure (currentToClamp previously sent to controllers)*/
            /* to avoid an elastic return we send currentToClamp BEFORE the brakes are released. */
            sendPreviouslySentCurrentAndReleaseBrakes();
            //go to currentToClose with a ramp on the 3 clamps in 5 steps.
            executeCurrentRampOnAllClamps(onlineClampYminus.getSentCurrent(),
                    onlineClampXminus.getSentCurrent(), onlineClampXplus.getSentCurrent(),
                    onlineClampYminus.getCurrentToClose(), onlineClampXminus.getCurrentToClose(),
                    onlineClampXplus.getCurrentToClose(), 5);   
         
        } else if (action == MobileItemAction.LOCK_ONLINECLAMPS) {
            /* clamps are CLOSED*/
            releaseBrakes();
            increaseCurrentToCurrentToClamp();
            
        } else if (action == MobileItemAction.OPEN_ONLINECLAMPS){
            /* clamps are OPENED*/
            releaseBrakes();
            sendCurrentToOpenAllClamps();
            
        } else {
            throw new IllegalArgumentException(getName() + " invalid action:"+action);
        } 
    }

    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay)  {
            onlineClampXminus.abortAction(action,delay);
            onlineClampXplus.abortAction(action,delay);
            onlineClampYminus.abortAction(action,delay);
    }

    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay)  {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public void postAction(FcsEnumerations.MobileItemAction action)  {
        if (action == MobileItemAction.CLOSE_AND_LOCK_ONLINECLAMPS) {
            increaseCurrentToCurrentToClamp();
            autochanger.updateStateWithSensors();
        } 
        /* activate brakes on the 3 clamps controllers and set current to 0*/
        activateBrakesAndSetCurrentToZero();
    }
    
    /**
     * activate brakes on the 3 clamps controllers and set current to 0
     */
    private void activateBrakesAndSetCurrentToZero() {
        for (AutochangerOnlineClampModule clamp: clampsList) {
            clamp.getController().activateBrake();
            /*
            /* in this case we use writeCurrent method of EPOSController instead of sendCurrentToControllerAndSaveValue
            /* because we don't want to keep on disk this last value of current sent to controller (0).
            */
            clamp.getController().writeCurrent(0);
        }
    }
    
    /**
     * releases brakes on the 3 clamps controllers
     */
    private void releaseBrakes() {
        for (AutochangerOnlineClampModule clamp: clampsList) {
            clamp.getController().doReleaseBrake();
        }
    }
    
    /**
     * increases slowly current sent to controllers to lock the clamps.
     * - first increases current in onlineClampYminus until 2/3 of currentToClamp
     * - increases current in both clamps X alternatively
     * - then increases current until currentToClamp for the 3 the clamps.
     */
    private void increaseCurrentToCurrentToClamp() {
        long period = 500;
        int n = 2;
        int d = 3;
        /* do a ramp of current until currentToClamp*n/d on onlineClampYminus */
        onlineClampYminus.executeCurrentRamp(onlineClampYminus.getCurrentToClose(), 
                onlineClampYminus.getCurrentToClamp()*n/d, period);
        /* increase current to -2A for X- and +2A for X+ alternativaly and waits until this task is completed*/
        executeCurrentRampOnClampsX(onlineClampXminus.getCurrentToClose(), onlineClampXplus.getCurrentToClose(),
                onlineClampXminus.getCurrentToClamp()*n/d, onlineClampXplus.getCurrentToClamp()*n/d);
        /* increase current on the 3 clamps until getCurrentToClamp() and waits until this task is completed.*/
        executeCurrentRampOnAllClamps(onlineClampYminus.getCurrentToClamp()*n/d,
            onlineClampXminus.getCurrentToClamp()*n/d,onlineClampXplus.getCurrentToClamp()*n/d,
            onlineClampYminus.getCurrentToClamp(), onlineClampXminus.getCurrentToClamp(), 
            onlineClampXplus.getCurrentToClamp(), 4);
    }
    
    /**
     * Sends the previous sent current to controller and Releases all brakes.
     */
    private void sendPreviouslySentCurrentAndReleaseBrakes() {
        /*sends last value of current to every controller.
        sentCurrent is near currentToClamp. It has been checked in checkCurrentBeforeUnlocking.
        */
        onlineClampXminus.sendCurrentToControllerAndSaveValue(onlineClampXminus.getSentCurrent());
        onlineClampXplus.sendCurrentToControllerAndSaveValue(onlineClampXplus.getSentCurrent());
        onlineClampYminus.sendCurrentToControllerAndSaveValue(onlineClampYminus.getSentCurrent());

        /* release all brakes in order to be able to move the clamps*/
        releaseBrakes();
    }
    
    /**
     * Creates an object to be published on the status bus.
     * @return 
     */
    public StatusDataPublishedByAutochangerThreeClamps createStatusDataPublishedByThreeClamps() {
        StatusDataPublishedByAutochangerThreeClamps status = new StatusDataPublishedByAutochangerThreeClamps();
        status.setName(getName());
        status.setLockStatus(lockStatus);
        return status;
    }

    @Override
    public void publishData() {
        this.getSubsystem().publishSubsystemDataOnStatusBus(
                new KeyValueData("autochangerClamps", createStatusDataPublishedByThreeClamps()));
    }

    private void updateStateAndCheckSensors() {
        autochanger.updateStateWithSensors();
        checkSensors(AC_SENSOR_ERROR);
    }

    /**
     * This methods updates lockStatus from the values return by the sensors.
     * This values are given in an array of hexa values as arguments of the
     * method.
     *
     * @param hexaValues
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Update state in reading sensors.")
    public void updateStateWithSensors(String[] hexaValues) {
        lock.lock();
        try {
            updatingState = true;
            for (AutochangerOnlineClampModule clamp: clampsList) {
                clamp.updateStateWithSensors(hexaValues);
            }
            computeLockStatus();

        } finally {

            updatingState = false;
            stateUpdated.signalAll();
            lock.unlock();
            this.publishData();
        }
    }
    
    /**
     * Compute lockStatus.
     */
    private void computeLockStatus() {
        if (isInError()) {
            this.lockStatus = LockStatus.ERROR;
            
        } else if (isLocked()) {
            this.lockStatus = LockStatus.LOCKED;
            
        } else if (isOpened()) {
            this.lockStatus = LockStatus.UNLOCKED;
            
        } else if (isInTravel()) {
            this.lockStatus = LockStatus.INTRAVEL;
            
        } else {
            /* if lockStatus is not the same for the 3 clamps we are in this case.
            It can happen during the closing (or the opening) of the 3 clamps when a clamp is already closed 
            but not the 2 others. 
            */
            this.lockStatus = LockStatus.UNKNOWN;
        }
    }
    
    /**********************************************************************************
     * Methods to write wurrent to controllers of clampX- and clampX+ with a ramp
    /**********************************************************************************
    */
    
    /**
     * Sends current to controllers of the clamps X- and X+ alternativaly 
     * from currentToClose to 2/3 currentToClamp and waits until the task is completed.
     */
    private void executeCurrentRampOnClampsX(int initialValueXminus, int initialValueXplus,
            int finalValueXminus, int finalValueXplus) {
        lock.lock();
        try {
            hasToWaitForEndOfCurrentRamp = true;
            writeCurrentRampOnClampsX(initialValueXminus, initialValueXplus, finalValueXminus, finalValueXplus);
            waitForEndOfCurrentRamp();
        } finally {
            hasToWaitForEndOfCurrentRamp = false;
            currentRampEnded.signalAll();
            lock.unlock();
        }
    }
    
    /**
     * Sends current alternativaly to the 3 clamps from 2/3 of currentToClamp to currentToClamp
     * and waits until the task is completed.
     * 
     */
    private void executeCurrentRampOnAllClamps(int initialValueYminus, int initialValueXminus, int initialValueXplus,
            int finalValueYminus, int finalValueXminus, int finalValueXplus, int nbStep) {
        lock.lock();
        try {
            hasToWaitForEndOfCurrentRamp = true;
            writeCurrentRampOnAllClamps(initialValueYminus, initialValueXminus, initialValueXplus,
                finalValueYminus, finalValueXminus, finalValueXplus, nbStep);
            waitForEndOfCurrentRamp();
        } finally {
            hasToWaitForEndOfCurrentRamp = false;
            currentRampEnded.signalAll();
            lock.unlock();
        }
    }
    
    /**
     * cancels current ramp
     * send signal to all threads waiting for currentRampEnded condition.
     */
    private void cancelCurrentRamp() {
        lock.lock();
        try {
            FCSLOG.finest(" => stop writing current");
            currentRampEnded.signalAll();

        } finally {
            lock.unlock();
        }
        this.currentRampHandle.cancel(true);
        FCSLOG.finest(" => current ramp ended");
    }
    
    
    /**
     * This method waits until the newCurrentValue ramp is completed. 
     */
    private void waitForEndOfCurrentRamp() {
        while (hasToWaitForEndOfCurrentRamp) {
            try {
                FCSLOG.finest(getName()+" waiting for end of current Ramp" );
                currentRampEnded.await();
            } catch (InterruptedException ex) {
                FCSLOG.finest(getName()+" InterruptedException received=" + ex.toString());
                break;
            }
        }
        FCSLOG.finest(getName()+" STOP WAITING FOR END OF CURRENT RAMP");
    }
    
    /**
     * sends current alternativaly to clamps X- and X+ with a ramp of current from
     * currentToClaose et 2/3*currentToClamp.
     * Uses a scheduler.
     */
    private void writeCurrentRampOnClampsX(int initialValueXminus, int initialValueXplus,
            int finalValueXminus, int finalValueXplus) {
        final int nbStep = 4;
        final int period = 250;
        final int stepHeightXminus = (finalValueXminus - initialValueXminus) / nbStep;
        final int stepHeightXplus = (finalValueXplus - initialValueXplus) / nbStep;
        FCSLOG.finest("############################################################");
        FCSLOG.finest("initialValue for clampXminus="+initialValueXminus);
        FCSLOG.finest("finalValue for clampXminus="+finalValueXminus);
        FCSLOG.finest("initialValue for clampXplus="+initialValueXplus);
        FCSLOG.finest("finalValue for clampXplus="+finalValueXplus);        
        FCSLOG.finest("nbStep="+nbStep);
        FCSLOG.finest("period="+period);
        FCSLOG.finest("stepHeightXminus="+stepHeightXminus);
        FCSLOG.finest("stepHeightXplus="+stepHeightXplus);
        FCSLOG.finest("############################################################");

        final Runnable currentRamp = new Runnable() {
            private int newCurrentValueXminus = initialValueXminus + stepHeightXminus;
            private int newCurrentValueXplus = initialValueXplus + stepHeightXplus;

            @Override
            public void run() {
                if (finalValueReached(stepHeightXminus,newCurrentValueXminus,finalValueXminus)
                        && finalValueReached(stepHeightXplus,newCurrentValueXplus,finalValueXplus)) {
                    hasToWaitForEndOfCurrentRamp = false;
                    cancelCurrentRamp();
                } else {
                    onlineClampXminus.sendCurrentToControllerAndSaveValue(newCurrentValueXminus);
                    onlineClampXplus.sendCurrentToControllerAndSaveValue(newCurrentValueXplus);
                    newCurrentValueXminus = newCurrentValueXminus + stepHeightXminus;
                    newCurrentValueXplus = newCurrentValueXplus + stepHeightXplus;
                }
            }
        };       
        this.currentRampHandle = scheduler.scheduleAtFixedRate(currentRamp, 0, period, TimeUnit.MILLISECONDS);
    }
    
    /**
     * Sends current alternativaly to the 3 clamps controllers 
     * with a ramp of current from 2/3 currentToClamp to currentToClamp.
     * Uses a sheduler.
     */
    private void writeCurrentRampOnAllClamps(int initialValueYminus, int initialValueXminus, int initialValueXplus,
            int finalValueYminus, int finalValueXminus, int finalValueXplus, int nbStep) {
        final int period = 250;
        final int stepHeightYminus = (finalValueYminus - initialValueYminus) / nbStep;
        final int stepHeightXminus = (finalValueXminus - initialValueXminus) / nbStep;
        final int stepHeightXplus = (finalValueXplus - initialValueXplus) / nbStep;
        FCSLOG.finest("############################################################");
        FCSLOG.finest("initialValue for clampYminus="+initialValueYminus);
        FCSLOG.finest("finalValue for clampYminus="+finalValueYminus);
        FCSLOG.finest("initialValue for clampXminus="+initialValueXminus);
        FCSLOG.finest("finalValue for clampXminus="+finalValueXminus);
        FCSLOG.finest("initialValue for clampXplus="+initialValueXplus);
        FCSLOG.finest("finalValue for clampXplus="+finalValueXplus);        
        FCSLOG.finest("nbStep="+nbStep);
        FCSLOG.finest("period="+period);
        FCSLOG.finest("stepHeightYminus="+stepHeightYminus);
        FCSLOG.finest("stepHeightXminus="+stepHeightXminus);
        FCSLOG.finest("stepHeightXplus="+stepHeightXplus);
        FCSLOG.finest("############################################################");

        final Runnable currentRamp = new Runnable() {
            private int newCurrentValueYminus = initialValueYminus + stepHeightYminus;
            private int newCurrentValueXminus = initialValueXminus + stepHeightXminus;
            private int newCurrentValueXplus = initialValueXplus + stepHeightXplus;

            @Override
            public void run() {
                if (finalValueReached(stepHeightXminus,newCurrentValueXminus,finalValueXminus)
                        && finalValueReached(stepHeightXplus,newCurrentValueXplus,finalValueXplus)
                        && finalValueReached(stepHeightYminus,newCurrentValueYminus,finalValueYminus)) {
                    hasToWaitForEndOfCurrentRamp = false;
                    cancelCurrentRamp();
                } else {
                    onlineClampYminus.sendCurrentToControllerAndSaveValue(newCurrentValueYminus);
                    onlineClampXminus.sendCurrentToControllerAndSaveValue(newCurrentValueXminus);
                    onlineClampXplus.sendCurrentToControllerAndSaveValue(newCurrentValueXplus);
                    newCurrentValueYminus = newCurrentValueYminus + stepHeightYminus;
                    newCurrentValueXminus = newCurrentValueXminus + stepHeightXminus;
                    newCurrentValueXplus = newCurrentValueXplus + stepHeightXplus;
                }
            }
        };       
        this.currentRampHandle = scheduler.scheduleAtFixedRate(currentRamp, 0, period, TimeUnit.MILLISECONDS); 
    }
    
    
    /**
     * Returns true if the final value is reached.
     * @param stepHeight
     * @param newCurrentValue
     * @param finalValue
     * @return 
     */
    private static boolean finalValueReached(int stepHeight, int newCurrentValue, int finalValue) {
        boolean goingUp = stepHeight > 0 && newCurrentValue > finalValue;
        boolean goingDown = stepHeight < 0 && newCurrentValue < finalValue;
        return goingUp || goingDown;
    }
    
    


}
