package org.lsst.ccs.subsystems.fcs;

import java.util.Observable;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.AC_SENSOR_ERROR;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.ControlledBySensors;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerWithBrake;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import static org.lsst.ccs.subsystems.fcs.utils.FcsUtils.TICKMILLIS;

/**
 * This is a model for online clamp which holds a filter when it is at ONLINE
 * position. The online clamps are part of the autochanger subsystem. There is 3
 * online clamps.
 *
 * @author virieux
 */
public class AutochangerOnlineClampModule extends MobileItemModule implements MovedByEPOSController, ControlledBySensors {

    private static final int CURRENT_RAMP_PERIOD = 500;
    public static final String ONLINE_CLAMP_CONFIG_CATEGORY = "onlineClampCurrent";

    private static final int TIMEOUT_FOR_CLOSING = 10000;
    private static final int TIMEOUT_FOR_OPENING = 10000;
    private AutoChangerModule autochanger;
    private final EPOSControllerWithBrake controller;

    /* couple of close sensors*/
    private final ComplementarySensors closeSensors;

    /* couple of open sensors*/
    private final ComplementarySensors openSensors;

    /**
     * currentToClose is different for each clamp so the value is initialized
     * from configuration system. tests have been done on hardware in January
     * 2017 with a value of 300mA for onlineClampXplus and value of -300mA for
     * onlineClampXminus and onlineClampYminus
     */
    @ConfigurationParameter(description = "current to close ONLINE clamp in mA", category = ONLINE_CLAMP_CONFIG_CATEGORY)
    private int currentToClose = 0;

    /**
     * currentToClamp is different for each clamp so the value is initialized
     * from configuration system.
     */
    @ConfigurationParameter(description = "current to clamp ONLINE clamp in mA", category = ONLINE_CLAMP_CONFIG_CATEGORY)
    private int currentToClamp = 0;

    @ConfigurationParameter(description = "current to open ONLINE clamp in mA", category = ONLINE_CLAMP_CONFIG_CATEGORY)
    private int currentToOpen = -currentToClose;

    /**
     * sentCurrent is not really a ConfigurationParameter but we want to save
     * this value of current between two instances of running FCS and each time
     * we sent current to the controller.
     *
     */
    @ConfigurationParameter(description = "current sent to controller in mA", category = ONLINE_CLAMP_CONFIG_CATEGORY)
    private int sentCurrent = 0;

    /**
     * increment of current in a current ramp, in mA
     */
    @ConfigurationParameter(description = "increment of current in a current ramp in mA", category = ONLINE_CLAMP_CONFIG_CATEGORY)
    private int stepHeightAbsValue = 250;

    /**
     * For the onlineClamp the lockStatus is OPENED, CLOSED, or LOCKED. Sensors
     * say if clamp is CLOSED or OPENED. To know if a clamp is LOCKED, we must
     * know the value of the current that was sent to the controller to close
     * the clamp. (field sentCurrent).
     */
    private FcsEnumerations.LockStatus lockStatus = LockStatus.UNKNOWN;

    private boolean controllerInFault;

    private volatile boolean initialized;
    private boolean controllerConfigured = false;

    private final Condition stateUpdated = lock.newCondition();

    /* This is used when we update the latch state with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;

    /* Tools for current ramp*/
    private ScheduledFuture<?> currentRampHandle;
    protected final Condition currentRampEnded = lock.newCondition();
    protected boolean hasToWaitForEndOfCurrentRamp;
    /* end of Tools for current ramp*/

    /**
     * Builds an AutochangerOnlineClampModule with a controller, 4 sensors and 2
     * maps of parameters for the controller.
     *
     * @param controller
     * @param closeSensors
     * @param openSensors
     * @param currentToClose
     * @param currentToClamp
     */
    public AutochangerOnlineClampModule(
            EPOSControllerWithBrake controller,
            ComplementarySensors closeSensors,
            ComplementarySensors openSensors,
            int currentToClose,
            int currentToClamp) {
        super(TICKMILLIS);
        this.controller = controller;
        this.closeSensors = closeSensors;
        this.openSensors = openSensors;
        this.currentToClose = currentToClose;
        this.currentToOpen = -currentToClose;
        this.currentToClamp = currentToClamp;
    }

    /**
     *
     * @param sentCurrent
     */
    @ConfigurationParameterChanger
    public void setSentCurrent(int sentCurrent) {
        this.sentCurrent = sentCurrent;
    }

    /**
     *
     * @param stepHeightAbsValue
     */
    @ConfigurationParameterChanger
    public void setStepHeightAbsValue(int stepHeightAbsValue) {
        if (stepHeightAbsValue > 0) {
            this.stepHeightAbsValue = stepHeightAbsValue;
        } else {
            throw new IllegalArgumentException(stepHeightAbsValue + ":illegal value. stepHeightAbsValue must be > 0");
        }
    }

    /**
     * Returns an adjusted value of the increment in current ramp.
     *
     * @param initialValue
     * @param finalValue
     * @return
     */
    public int getAdjustedStepHeight(int initialValue, int finalValue) {
        FCSLOG.finest("############################");
        FCSLOG.finest("initialValue=" + initialValue);
        FCSLOG.finest("finalValue=" + finalValue);
        FCSLOG.finest("|stepHeight|=" + stepHeightAbsValue);
        /*nbSteps is always > 0*/
        int nbSteps = Math.abs(finalValue - initialValue) / stepHeightAbsValue;
        /*adjustedStepHeight can be > 0 or < 0*/
        int adjustedStepHeight = (finalValue - initialValue) / nbSteps;
        FCSLOG.finest("adjustedStepHeight=" + adjustedStepHeight);
        return adjustedStepHeight;
    }

    /**
     * returns last value of current sent to controller
     *
     * @return
     */
    public int getSentCurrent() {
        return sentCurrent;
    }

    /**
     * Return ONLINE clamp controller
     *
     * @return
     */
    public EPOSControllerWithBrake getController() {
        return controller;
    }

    /**
     * For simulator
     *
     * @return
     */
    public int getCurrentToClose() {
        return currentToClose;
    }

    /**
     * For simulator
     *
     * @return
     */
    public int getCurrentToOpen() {
        return currentToOpen;
    }

    /**
     * For simulator
     *
     * @return
     */
    public int getCurrentToClamp() {
        return currentToClamp;
    }

    /**
     * returns closeSensors for tests
     *
     * @return
     */
    public ComplementarySensors getCloseSensors() {
        return closeSensors;
    }

    /**
     * returns openSensors for tests
     *
     * @return
     */
    public ComplementarySensors getOpenSensors() {
        return openSensors;
    }

    /**
     * Returns true if closeSensor and closeSensorC return the same value.
     * Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if lockSensor and lockSensorC retun the same value. "
            + "Doesn't read again sensors.")
    public boolean isCloseSensorsInError() {
        return closeSensors.isInError();
    }

    /**
     * Returns true if openSensor and openSensorC retun the same value. Doesn't
     * read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if unlockSensor and unlockSensorC retun the same value. "
            + "Doesn't read again sensors.")
    public boolean isOpenSensorsInError() {
        return openSensors.isInError();
    }

    /**
     * Returns true if LockStatus=ERROR, this means closeSensor or openSensor is
     * in ERROR or openSensor and closeSensor return non consistant values.
     * Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=ERROR, this means that "
            + "closeSensor or openSensor is in ERROR or that"
            + "openSensor and closeSensor return non consistant values. Doesn't read again sensors.")
    @Override
    public boolean isInError() {
        return lockStatus == FcsEnumerations.LockStatus.ERROR;
    }

    /**
     * Returns true if LockStatus=CLOSED. Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=CLOSED. Doesn't read again sensors.")
    public boolean isClosed() {
        return lockStatus == FcsEnumerations.LockStatus.CLOSED;
    }

    /**
     * Returns true if LockStatus=LOCKED. Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=LOCKED. Doesn't read again sensors.")
    public boolean isLocked() {
        return lockStatus == FcsEnumerations.LockStatus.LOCKED;
    }

    /**
     * Returns true if LockStatus=OPENED. Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=OPENED. Doesn't read again sensors.")
    public boolean isOpened() {
        return lockStatus == FcsEnumerations.LockStatus.OPENED;
    }

    /**
     * Returns true if LockStatus=INTRAVEL. Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=INTRAVEL. Doesn't read again sensors.")
    public boolean isInTravel() {
        return lockStatus == FcsEnumerations.LockStatus.INTRAVEL;
    }

    /**
     * Returns true if controller is in fault. Doesn't read controller CPU.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if controller is in fault. Doesn't read controller CPU.")
    @Override
    public boolean isControllerInFault() {
        return controllerInFault;
    }

    /**
     * set controllerInFault
     *
     * @param controllerInFault
     */
    @Override
    public void setControllerInFault(boolean controllerInFault) {
        this.controllerInFault = controllerInFault;
    }

    /**
     * return controller name
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns controller name.")
    @Override
    public String getControllerName() {
        return controller.getName();
    }

    /**
     * Returns true if clamp is initialized : hardware is ready (at least
     * booted) and clamp controller is initialized (parameters in the controller
     * CPU have been checked and controller has been configured).
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if clamp is initialized : controller is booted, parameters in the controller "
            + "CPU have been checked and controller is configured.")
    public boolean isInitialized() {
        return initialized;
    }

    /**
     * Initialize fields autochanger, lockStatus and listens to its controller.
     */
    @Override
    public void initModule() {
        super.initModule();
        this.autochanger = (AutoChangerModule) getComponentLookup().getComponentByName("autochanger");
        this.lockStatus = FcsEnumerations.LockStatus.UNKNOWN;

        /**
         * Because AutochangerOnlineClampModule implements
         * MovedByEPOSController, it has to listen to its controller to known
         * when the controller is in Fault or when a Fault reset has been done.
         */
        if (controller == null) {
            FCSLOG.error(getName() + "==>>> onlineController == null - Please fix groovy description file.");
            throw new IllegalArgumentException(getName() + "==>>> null onlineClampController - fix groovy description file.");
        } else {
            //listens to my Controller to detect the controller's faultReset
            //or the emergency messages coming from the controller.
            if (controller instanceof Observable) {
                this.listens((Observable) controller);
            }
        }
    }

    /**
     * Return true when ONLINE clamp hardware is ready. That is when controller
     * is initialized and configured.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if controller is initialized and configured.")
    @Override
    //tested on CPPM testbench in september 2015
    public boolean isCANDevicesReady() {
        return ((MainModule) getComponentLookup().getComponentByName("main")).isCANDevicesReady()
                && controller.isInitialized() && controllerConfigured;
    }

    /**
     * Check if it's safe to open the clamp.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the onlineClamp can be locked.")
    public void checkConditionsForOpening() {
        checkConditionsForClosing();

        if (!autochanger.isHoldingFilter()) {
            throw new RejectedCommandException(getName() + " can't be OPENED if autochanger is not HOLDING filter.");
        }
    }

    /**
     * Check if it's safe to close the clamp.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the onlineClamp can be locked.")
    public void checkConditionsForClosing() {
        if (!autochanger.getAutochangerTrucks().isAtOnline()) {
            throw new RejectedCommandException(getName() + " can't be CLOSED if autochanger trucks are not at ONLINE position.");
        } else if (autochanger.isEmpty()) {
            throw new RejectedCommandException(getName() + " can't be CLOSED if there is no filter in autochanger trucks.");
        }
    }

    /**
     * This method is called during INITIALIZATION phase. It configures the
     * controller for the digital inputs and outputs. It changes it to CURRENT
     * mode if it's not in this mode and it writes in the CPU of the controller
     * the values of configuration for CURRENT mode parameters.
     *
     * @return
     * @throws HardwareException
     */
    @Override
    //tested on CPPM testbench in september 2015
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Initialize controller and check parameters.")
    public TreeWalkerDiag checkHardware() throws HardwareException {
        super.checkHardware();

        FCSLOG.debug(getName() + " checking hardware.");

        try {
            controller.initializeAndCheckHardware();
            configureController();
            controller.changeMode(EposMode.CURRENT);
            controller.writeParameters(EposMode.CURRENT);
        } catch (FcsHardwareException ex) {
            throw new HardwareException(false, ex);

        } catch (FailedCommandException ex) {
            throw new HardwareException(true, ex);
        }
        this.initialized = true;
        return TreeWalkerDiag.HANDLING_CHILDREN;
    }

    /**
     * Enables controller, changes mode into CURRENT, Reads parameters for mode
     * CURRENT on controller CPU. If a parameter has a different value than in
     * configuration, throws an exception.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read in the CPU of the controller parameters for mode CURRENT."
            + "If a parameter has a different value than in configuration, "
            + "throws an exception.")
    public void enableAndCheckControllerBeforeAction() {
        controller.enable();
        controller.changeMode(EposMode.CURRENT);
        try {
            controller.checkParameters(EposMode.CURRENT);
            controller.checkFault();
        } catch (HardwareException ex) {
            String msg = getName() + " error in parameters ";
            FCSLOG.error(msg + ex);
            throw new FcsHardwareException(msg, ex);
        }
        if (!controller.isParametersOK()) {
            String msg = getName() + " Some parameter values are not"
                    + " the same in CPU and configuration system.";
            FCSLOG.error(msg);
            throw new FcsHardwareException(msg);
        }
    }

    /**
     * Configure ONLINE clamp controller.
     *
     * @throws HardwareException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Configure controller.")
    //tested on CPPM testbench in september 2015
    public void configureController() throws HardwareException {
        try {
            controllerConfigured = false;
            controller.activateBrake();
//            controller.shutdownEPOS();
            this.configureDigitalInputOfOnlineClamps();
            this.configureDigitalOutputOfOnlineClamps();
            controllerConfigured = true;
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(getName(), ex);
        }
    }

    /**
     * reads sensors, computes and state and raises an ALERT if sensors are in error.
     *
     */
    public void updateStateAndCheckSensors() {
        autochanger.updateStateWithSensors();
        checkSensors(AC_SENSOR_ERROR);
    }

    /**
     * This methods updates lockStatus from the values return by the sensors.
     * This values are given in an array of hexa values as arguments of the
     * method.
     *
     * @param hexaValues
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Update state in reading sensors.")
    public void updateStateWithSensors(String[] hexaValues) {
        lock.lock();
        try {
            updatingState = true;

            this.closeSensors.updateValues(hexaValues);
            this.openSensors.updateValues(hexaValues);

            boolean closed = this.closeSensors.isOn();
            boolean opened = this.openSensors.isOn();
            boolean someSensorsInError = closeSensors.isInError() || openSensors.isInError();
            boolean inError = someSensorsInError || (closed && opened);

            if (inError) {
                lockStatus = FcsEnumerations.LockStatus.ERROR;
            } else if (closed) {
                lockStatus = FcsEnumerations.LockStatus.CLOSED;
            } else if (opened) {
                lockStatus = FcsEnumerations.LockStatus.OPENED;
            } else {
                lockStatus = FcsEnumerations.LockStatus.INTRAVEL;
            }
            /* if the clamp is closed and a currentToClamp has been previously sent to controller
             then clamp is LOCKED*/
            if (closed && this.sentCurrent == this.currentToClamp) {
                lockStatus = FcsEnumerations.LockStatus.LOCKED;
            }

        } finally {

            updatingState = false;
            stateUpdated.signalAll();
            lock.unlock();
            this.publishData();
        }

    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        autochanger.updateStateWithSensors();
    }

    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        if (action == MobileItemAction.CLOSE_ONLINECLAMP || action == MobileItemAction.UNLOCK_ONLINECLAMP) {
            return isClosed();
            //TODO check if this condition is the good one for unlock.
            //because this condition is true when action is started, check that is could not make the 
            //unlock action complete prematuraly?

        } else if (action == MobileItemAction.OPEN_ONLINECLAMP) {
            return isOpened();

        } else if (action == MobileItemAction.LOCK_ONLINECLAMP) {
            return isLocked();

        } else {
            throw new IllegalArgumentException(getName() + " invalid action for ONLINE clamp:" + action);
        }
    }

    /**
     * Closes the ONLINE clamp.
     */
    /*Tested in december 2016*/
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Closes the ONLINE clamp.", timeout = TIMEOUT_FOR_CLOSING)
    public void close() {
        //TODO : il faut un filtre !!!
        //il faut être en ONLINE et qu'il y ait un filtre.
        updateStateAndCheckSensors();
        if (this.isOpened()) {
            //uncommented for tests in december 2016
//            checkConditionsForClosing();
            this.executeAction(MobileItemAction.CLOSE_ONLINECLAMP, TIMEOUT_FOR_CLOSING);

        } else if (this.isClosed()) {
            FCSLOG.info(getName() + " is already CLOSED. Nothing to do.");

        } else {
            throw new RejectedCommandException(getName() + " has to be OPENED before a close action.");
        }
    }

    /**
     * Opens the ONLINE clamp.
     *
     */
    /*Tested in december 2016*/
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Opens the ONLINE clamp.", timeout = TIMEOUT_FOR_OPENING)
    public void open() {
        updateStateAndCheckSensors();
        if (this.isClosed() || this.isInTravel()) {
            //uncommented for tests in december 2016
            //        checkConditionsForOpening();
            this.executeAction(MobileItemAction.OPEN_ONLINECLAMP, TIMEOUT_FOR_OPENING);

        } else if (this.isOpened()) {
            FCSLOG.info(getName() + " is already OPENED. Nothing to do.");

        } else {
            throw new RejectedCommandException(getName() + " has to be CLOSED before an open action.");
        }
    }

    /**
     * Locks the ONLINE clamp : sends currentToClamp to the controller with a
     * ramp of current from currentToClose to currentToRamp. The clamp has to be
     * CLOSED before this action. At the end of this action, the clamp is
     * LOCKED.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Locks the ONLINE clamp : sends currentToClamp to the controller with a ramp of current"
            + "from currentToClose to currentToRamp.", timeout = TIMEOUT_FOR_CLOSING)
    public void lock() {
        updateStateAndCheckSensors();
        //TODO check conditions for locking
        if (this.isClosed() || this.isInTravel()) {
            this.executeAction(MobileItemAction.LOCK_ONLINECLAMP, TIMEOUT_FOR_CLOSING);

        } else if (this.isLocked()) {
            FCSLOG.info(getName() + " is already CLOSED. Nothing to do.");

        } else {
            throw new RejectedCommandException(getName() + " has to be CLOSED before lock action.");
        }
    }

    /**
     * Unlocks the ONLINE clamp : sends currentToClamp to the controller with a
     * ramp of current from currentToClamp to currentToClose. The clamp has to
     * be LOCKED before this action. At the end of this action, the clamp is
     * CLOSED. The goal if this action is to slow down the pressure exerted by
     * the clamp on the filter. If we decrease too quickly the pressure the
     * clamp could be ejected out by elastic return.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Unlocks the ONLINE clamp : sends currentToClamp to the controller "
            + " with a ramp of current from currentToClamp to currentToClose.",
            timeout = TIMEOUT_FOR_OPENING)

    public void unlock() {
        updateStateAndCheckSensors();
        //TODO check conditions for unlocking
        if (this.isLocked()) {
            this.executeAction(MobileItemAction.UNLOCK_ONLINECLAMP, TIMEOUT_FOR_CLOSING);
            FCSLOG.info(getName() + " is already CLOSED. Nothing to do.");

        } else if (this.isClosed()) {
            FCSLOG.info(getName() + " is already CLOSED. Nothing to do.");

        } else {
            throw new RejectedCommandException(getName() + " has to be LOCKED before unlock action.");
        }
    }
    
    


    /**
     * Start action of open or close or lock or unlock.
     *
     * @param action
     * @throws FcsHardwareException
     */
    @Override
    public void startAction(FcsEnumerations.MobileItemAction action) {
        this.enableAndCheckControllerBeforeAction();
        autochanger.checkConditionsForActioningOnlineClamps();
        if (action == MobileItemAction.CLOSE_ONLINECLAMP) {
            sendCurrentToControllerAndSaveValue(this.currentToClose);
            controller.doReleaseBrake();

        } else if (action == MobileItemAction.OPEN_ONLINECLAMP) {
            sendCurrentToControllerAndSaveValue(this.currentToOpen);
            controller.doReleaseBrake();

        } else if (action == MobileItemAction.LOCK_ONLINECLAMP) {
            controller.doReleaseBrake();
            /* write a ramp from  currentToClose to currentToClamp with a period CURRENT_RAMP_PERIOD*/
            this.executeCurrentRamp(this.currentToClose, this.currentToClamp, CURRENT_RAMP_PERIOD);

        } else if (action == MobileItemAction.UNLOCK_ONLINECLAMP) {
            /* the clamp is LOCKED so we send currentToClamp before releasing brake to avoid an elastic 
             return of the clamp */
            sendCurrentToControllerAndSaveValue(this.currentToClamp);
            controller.doReleaseBrake();
            /* write a ramp from  currentToClamp to currentToClose with a period CURRENT_RAMP_PERIOD*/
            this.executeCurrentRamp(this.currentToClamp, this.currentToClose, CURRENT_RAMP_PERIOD);

        } else {
            throw new IllegalArgumentException(getName() + " invalid action for ONLINE clamp:"
                    + action);
        }
    }

    /**
     * Writes current to controller and saves the current value in the
     * ConfigurationParameter sentCurrent.
     *
     * @param current to send to controller
     */
    public void sendCurrentToControllerAndSaveValue(int current) {
        FCSLOG.finest(getName() + " current to write to controller " + getControllerName() + ":" + current);
        controller.writeCurrent(current);
        getComponentConfigurationEnvironment().change("sentCurrent", current);
        autochanger.saveCurrentSentToOnlineClampsControllers();
        this.publishData();
    }

    /**
     * What to do to abort an action.
     *
     * @param action
     * @param delay
     * @throws FcsHardwareException
     */
    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay) {
        controller.quickStop();
        controller.activateBrake();
    }

    /**
     * What to be done after the action is completed.
     *
     * @param action
     * @throws FcsHardwareException
     */
    @Override
    public void postAction(FcsEnumerations.MobileItemAction action) {
        controller.activateBrake();
        controller.writeCurrent(0);
    }

    /**
     * This method is used to configure the controllers of the autochanger
     * online clamps. ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ Digital
     * Inputs activated for OnlineClamps controllers are:
     * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ QuickStop Input=01
     * Mask=Enabled Polarity=HighActive HE_Sensor General_A Input=02
     * Mask=Enabled Polarity=HighActive HE_Sensor General_B Input=03
     * Mask=Enabled Polarity=HighActive Check Brake Status
     * ******************************************** 1 - Configure the Digital
     * Input Functionnality index 0x2070 sindex input_ID Functionnality tab
     * Value = 15 General Purpose A Value = 14 General Purpose B Value = 13
     * General Purpose C Value = 12 General Purpose D Value = 11 General Purpose
     * E Value = 10 General Purpose F Value = 9 General Purpose G Value = 8
     * General Purpose H Value = 7 General Purpose I Value = 6 General Purpose J
     * Value = 5 QuickStop Value = 4 Drive enable Value = 3 Position marker
     * Value = 2 Home switch Value = 1 Positive Limit switch Value = 0 Negative
     * Limit switch
     * *************************************************************** 2 -
     * Configure the Digital Input Polarity index 0x2071 subindex 0x03 Value = 0
     * High Active Value = 1 Low Active
     * **************************************************************** 3 -
     * Configure the Digital Input Mask index 0x2071 subindex 0x02 Value = 0
     * Functionnality state will not be displayed Value = 1 Functionnality state
     * will be displayed Digital Input Functionnality tab value Bit_15 General
     * Purpose A 1 Bit_14 General Purpose B 1 Bit_13 General Purpose C 0 Bit_12
     * General Purpose D 0 Bit_11 General Purpose E 0 Bit_10 General Purpose F 0
     * Bit_9 General Purpose G 0 Bit_8 General Purpose H 0 Bit_7 General Purpose
     * I 0 Bit_6 General Purpose J 0 Bit_5 QuickStop 1 Bit_4 Drive enable 0
     * Bit_3 Position marker 0 Bit_2 Home switch 0 Bit_1 Positive Limit switch 0
     * Bit_0 Negative Limit switch 0
     * ********************************************************************* 4 -
     * Configure the Digital Input Execution Mask index 0x2071 subindex 0x04
     * Digital Input Excecution Mask Value Bit_15 to Bit_6 reserved 0 Bit_5
     * QuickStop 1 Bit_4 Drive enable 1 Bit_3 Position marker 0 Bit_2 Home
     * switch 0 Bit_1 Positive Limit switch 0 Bit_0 Negative Limit switch 0
     * **********************************************************************
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "To configure the online clamps controllers.")
    public void configureDigitalInputOfOnlineClamps() {
        //1-Configure Digital Input Fonctionnality
        //QuickStop
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.ConfigurationOfDigitalInput1, "0005");

        //2-Configure the Digital Input Polarity
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalInputFonctionnalityPolarity, "0");

        //3-Configure the Digital Input Mask C20:1100 0010 0000 
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalInputFonctionnalityMask, "C20");

        //4 - Configure the Digital Input Execution Mask 30:0011 0000 
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalInputFonctionnalityExecutionMask, "30");
    }

    /**
     * This method is used to configure the autochanger onlineClamps EPOS
     * controllers. Digital Outputs activated for OnlineClamps controller are:
     * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ Ready/Fault
     * Output=01 Mask=Enabled Polarity=HighActive General_A Output=04
     * Mask=Enabled Polarity=HighActive
     * ******************************************* 1 - Configure the Digital
     * Output Functionnality index 0x2079 subindex input_ID Functionnality tab :
     * Value = 15 General Purpose Out_A Value = 14 General Purpose Out_B Value =
     * 13 General Purpose Out_C Value = 12 General Purpose Out_D Value = 11
     * General Purpose Out_E Value = 10..8 not used Value = 7..3 reserved Value
     * = 2 Holding Brake Value = 1 Position compare Value = 0 Ready/Fault
     * ********************************************************************************
     * 2 - Configure the Digital Output Functionnality Mask index 0x2078
     * subindex 0x02 Value = 0 functionnality not activatd Value = 1
     * functionnality activated
     * ********************************************************************************
     * Digital Output Functionnality Mask Value Bit_15 General Purpose Out_A 1
     * Bit_14 General Purpose Out_B 0 Bit_13 General Purpose Out_C 0
     * Bit_12General Purpose Out_D 0 Bit_11 General Purpose Out_E 0 Bit_10
     * General Purpose Out_F 0 Bit_9 General Purpose Out_G 0 Bit_8 General
     * Purpose Out_H 0 Bit_7 reserved 0 Bit_6 reserved 0 Bit_5 reserved 0 Bit_4
     * reserved 0 Bit_3 reserved 0 Bit_2 Holding Brake 0 (read only) Bit_1
     * Position Compare (read only) Bit_0 Ready/Fault (read only)
     * ********************************************************************************
     * 3 - Configure the Digital Output Functionnality Polarity index 0x2078
     * subindex 0x03 Value = 0 associated output not change => HighActive Value
     * = 1 associated output inverted => LowActive
     * ********************************************************************************
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "To configure the online clamps controllers.")
    public void configureDigitalOutputOfOnlineClamps() {
        //1 - Configure the Digital Output Functionnality
        //Ready/Fault Active on Device Ready / Inactive on Fault
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.ConfigurationOfDigitalOutput1, "0");
        //General_A
        //gives an error : CAN OPEN DEVICE ERROR CODE=6090031 index=2079 subindex=04 error name=Value too high Error
        //writeParameterInHexa(Parameter.ConfigurationOfDigitalOutput4, "8000"); //pas bon

        //2 - Configure the Digital Output Functionnality Mask 8001=1000 0000 0000 0001 
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalOutputFonctionnalityMask, "8001");

        //3 - Configure the Digital Output Functionnality Polarity 8001=1000 0000 0000 0001 
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalOutputFonctionnalityPolarity, "8001");
    }

    /**
     * Updates field controllerInFault when the controller notifies its
     * observers and sends new values. This class observes its controller to
     * publish data when its controller is in fault or after a faultReset.
     * Needed to update the GUI.
     *
     * @param source
     * @param v
     */
    @Override
    public void processUpdate(Observable source, ValueUpdate v) {
        updateControllerInFault(((EPOSController) source).getName(), v);
    }

    /**
     * Creates and returns the object to be published on the STATUS bus.
     *
     * @return
     */
    public StatusDataPublishedByAutochangerOnlineClamp createStatusDataPublishedByOnlineClamp() {
        StatusDataPublishedByAutochangerOnlineClamp status = new StatusDataPublishedByAutochangerOnlineClamp();
        status.setName(getName());
        status.setLockSensorValue(closeSensors.isOn());
        status.setUnlockSensorValue(openSensors.isOn());
        status.setLockStatus(lockStatus);
        status.setLockSensorInError(closeSensors.isInError());
        status.setUnlockSensorInError(openSensors.isInError());
        status.setInError(lockStatus == FcsEnumerations.LockStatus.ERROR);
        status.setControllerBooted(controller.isBooted());
        status.setControllerInFault(controllerInFault);
        status.setSentCurrent(sentCurrent);
        return status;
    }

    @Override
    public void publishData() {
        this.getSubsystem().publishSubsystemDataOnStatusBus(
                new KeyValueData(getName(), createStatusDataPublishedByOnlineClamp()));
    }

    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay) {
        controller.quickStop();
    }

    /**
     * **********************************************************
     * Methods to write current to the controller with a ramp
     * /************************************************************
     */
    //TODO test with a synchronized 
    private void cancelCurrentRamp() {
        lock.lock();
        try {
            FCSLOG.debug(" => stop writing current");
            currentRampEnded.signalAll();

        } finally {
            lock.unlock();
        }
        this.currentRampHandle.cancel(true);
        FCSLOG.debug(" => current ramp ended");
    }

    /**
     * This method waits until the newCurrentValue ramp is completed.
     */
    private void waitForEndOfCurrentRamp() {
        while (hasToWaitForEndOfCurrentRamp) {
            try {
                FCSLOG.debug(getName() + " waiting for end of current Ramp");
                currentRampEnded.await();
            } catch (InterruptedException ex) {
                FCSLOG.debug(getName() + " InterruptedException received=" + ex.toString());
                break;
            }
        }
        FCSLOG.debug(getName() + " STOP WAITING FOR END OF CURRENT RAMP");
    }

    /**
     * Sends current value to controller with a current ramp from initialValue
     * to finalValue. Uses a scheduler to schedule the task.
     *
     * @param initialValue first value to send to controller
     * @param finalValue last value to send to controller
     * @param nbStep number of steps that we wait between initialValue to
     * finalValue.
     * @param period period of time between 2 instructions writeCurrent.
     */
    private void writeCurrentRamp(final int initialValue, final int finalValue, final long period) {
        FCSLOG.debug("############################");
        FCSLOG.debug("initialValue=" + initialValue);
        FCSLOG.debug("finalValue=" + finalValue);
        FCSLOG.debug("|stepHeight|=" + stepHeightAbsValue);
        FCSLOG.debug("period=" + period);
        FCSLOG.debug("############################");

        final Runnable currentRamp = new Runnable() {
            private final int adjustedStepHeight = getAdjustedStepHeight(initialValue, finalValue);
            private int newCurrentValue = initialValue + adjustedStepHeight;

            public boolean finalValueReached() {
                boolean reachedUp = adjustedStepHeight > 0 && newCurrentValue > finalValue;
                boolean reachedDown = adjustedStepHeight < 0 && newCurrentValue < finalValue;
                return reachedUp || reachedDown;
            }

            @Override
            public void run() {
                if (finalValueReached()) {
                    hasToWaitForEndOfCurrentRamp = false;
                    cancelCurrentRamp();
                } else {
                    sendCurrentToControllerAndSaveValue(newCurrentValue);
                    newCurrentValue = newCurrentValue + adjustedStepHeight;
                }
            }
        };
        this.currentRampHandle = scheduler.scheduleAtFixedRate(currentRamp, 0, period, TimeUnit.MILLISECONDS);
    }

    /**
     * Sends current to controller with a ramp : from initialValue to finalValue
     * in a given number of steps (nbStep) and waits until the final value is
     * reached. It uses method writeCurrentRamp which uses a scheduler.
     *
     * @param initialValue first value to be sent to controller
     * @param finalValue last value to send to controller
     * @param period of time to wait between 2 writes to the controller.
     */
    public void executeCurrentRamp(int initialValue, int finalValue, long period) {
        lock.lock();
        try {
            hasToWaitForEndOfCurrentRamp = true;
            writeCurrentRamp(initialValue, finalValue, period);
            waitForEndOfCurrentRamp();
        } finally {
            hasToWaitForEndOfCurrentRamp = false;
            currentRampEnded.signalAll();
            lock.unlock();
        }
    }

    /**
     * **************************************************************
     * end of Methods to write wurrent to the controller with a ramp
     * /****************************************************************
     */
}
