
package org.lsst.ccs.subsystems.fcs;

import java.util.Observable;
import java.util.concurrent.locks.Condition;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.KeyValueData;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.framework.TreeWalkerDiag;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.AC_SENSOR_ERROR;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.common.ControlledBySensors;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;

/**
 * This class is a model for the latch which lock a filter on the autochanger
 * trucks. 
 * The ACTION methods : open and close of this class is used in ENGINEERING mode only 
 * when an operator may need to open or close only one Latch. 
 * In NORMAL mode, when we have to open or close both Latches in the same
 * time, the class AutochangerTwoLatches is used. 
 * On the autochanger there are 2 latches : one on each side of the filter : at X+ and X-.
 *
 * @author virieux
 */
public class AutochangerLatchModule extends MobileItemModule implements MovedByEPOSController, ControlledBySensors {

    private AutoChangerModule autochanger;
    private final EPOSController latchController;
    private final ComplementarySensors filterEngagedSensors;
    private final ComplementarySensors lockSensors;
    private final ComplementarySensors unlockSensors;


    @ConfigurationParameter(description="in mA current to be sent to the controller to open the latch.")
    private int currentToOpen = 300;
    
    @ConfigurationParameter(description = "current to close Autochanger latch in mA")
    private int currentToClose = -currentToOpen;
 
    
    @ConfigurationParameter(description="timeout for opening a latch in milliseconds.")
    private long timeoutForOpening = 3000;
    
    @ConfigurationParameter(description="timeout for closing a latch in milliseconds.")
    private long timeoutForClosing = 3000;
    
    private int readCurrent;

    private LockStatus lockStatus = LockStatus.UNKNOWN;

    private volatile boolean initialized;

    private final Condition stateUpdated = lock.newCondition();

    /* This is used when we update the latch state with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;

    /**
     * FOR THE GUI
     * true if the controller is in fault
     * false if the user does a faultReset
     */
    private boolean controllerInFault;
    

    /**
     * Buils a AutochangerLatchModule with a controller and 3 sensors.
     * @param latchController 
     * @param lockSensors 
     * @param unlockSensors 
     * @param filterEngagedSensors 
     */
    public AutochangerLatchModule(
            EPOSController latchController,
            ComplementarySensors lockSensors,
            ComplementarySensors unlockSensors,
            ComplementarySensors filterEngagedSensors) {
        this.latchController = latchController;
        this.lockSensors = lockSensors;
        this.unlockSensors = unlockSensors;
        this.filterEngagedSensors = filterEngagedSensors;
    }


    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns this latch controller name.")
    @Override
    public String getControllerName() {
        return latchController.getName();
    }

    /**
     * Return latchController
     * @return 
     */
    public EPOSController getLatchController() {
        return latchController;
    }

    /**
     * Returns true if latchController is in Fault.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if latchController is in Fault.")
    @Override
    public boolean isControllerInFault() {
        return controllerInFault;
    }

    /**
     * Set controllerInFault
     * @param controllerInFault 
     */
    @Override
    public void setControllerInFault(boolean controllerInFault) {
        this.controllerInFault = controllerInFault;
    }

    /**
     * Returns true if latch is initialized. 
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if latch is initialized.")
    public boolean isInitialized() {
        return initialized;
    }

    /**
     * For simulation, return currentToClose
     * @return 
     */
    public int getCurrentToClose() {
        return currentToClose;
    }

    /**
     * Return timeoutForOpening
     * @return 
     */
    public long getTimeoutForOpening() {
        return timeoutForOpening;
    }

    /**
     * Return timeoutForClosing
     * @return 
     */
    public long getTimeoutForClosing() {
        return timeoutForClosing;
    }

    /**
     * Return lockStatus.
     * If another thread is updating lockStatus it waits untils update is completed and
     * then returns lockStatus.
     * Otherwise it returns lockStatus immediately.
     * 
     * @return 
     */
    public FcsEnumerations.LockStatus getLockStatus() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.warning(getName() + ": interrupted in getLockStatus.",ex);
                }
            }
            return lockStatus;

        } finally {
            lock.unlock();
        }
    }

    /**
     * Returns true if LockStatus=LOCKED
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=LOCKED. Doesn't read again sensors.",
            alias="isClosed")
    public boolean isLocked() {
        return this.lockSensors.isOn();
    }

    /**
     * Returns true if LockStatus=UNLOCKED
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=UNLOCKED. Doesn't read again sensors.",
            alias="isOpen")
    public boolean isUnlocked() {
        return this.unlockSensors.isOn();
    }

    /**
     * Returns true if LockStatus=ERROR, this means that unlockSensor and lockSensor return 
     * non consistant values.
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=ERROR, this means that "
            + "unlockSensor and lockSensor return non consistant values. Doesn't read again sensors.")
    @Override
    public boolean isInError() {
        return this.lockStatus == LockStatus.ERROR;
    }

    /**
     * Returns true if autochanger is empty.
     * 
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if autochanger is empty. Doesn't read again sensors.")
    public boolean isEmpty() {
        return !this.filterEngagedSensors.isOn();
    }

    /**
     * Initialize field autochanger and latchController
     */
    @Override
    public void initModule() {
        super.initModule();
        ComponentLookup lookup = getSubsystem().getComponentLookup();
        this.autochanger = (AutoChangerModule) lookup.getComponentByName("autochanger");

        if (latchController == null) {
            FCSLOG.error(getName() + "==>>> latchController == null - Please fix groovy description file.");
            throw new IllegalArgumentException(getName() + "==>>> null latchController - fix groovy description file.");
        } else {
            //listens to latchController to detect the controller's faultReset
            //or the emergency messages coming from the controller.
            if (latchController instanceof Observable) {
                this.listens((Observable) latchController);
            }
        }
    }


    /**
     * Executed during INITIALIZATION phase.
     * Initialize and check latchController.
     * @return
     * @throws HardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Initialize latch controller.")
    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {
        try {
            latchController.initializeAndCheckHardware();
            latchController.changeMode(EPOSEnumerations.EposMode.CURRENT);
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(getName(),ex);
        } catch (FcsHardwareException ex) {
            throw new HardwareException(false, ex);
        }
        //TODO check if there is a homing to be done.
        this.initialized = true;
        return TreeWalkerDiag.HANDLING_CHILDREN;
    }
    

    /**
     * This methods updates lockStatus from the values return by the sensors.
     * This values are given in an array of hexa values as arguments of the
     * method.
     *
     * @param hexaValues
     * 
     */
    protected void updateStateWithSensors(String[] hexaValues) {
        lock.lock();
        try {
            updatingState = true;
            this.filterEngagedSensors.updateValues(hexaValues);
            this.lockSensors.updateValues(hexaValues);
            this.unlockSensors.updateValues(hexaValues);

            boolean locked = this.lockSensors.isOn();            
            boolean unlocked = this.unlockSensors.isOn();
            
            boolean someSensorsInError = this.filterEngagedSensors.isInError()
                    || this.lockSensors.isInError() || this.unlockSensors.isInError();
            
            boolean inError = (locked && unlocked) || someSensorsInError;

            if (inError) {
                lockStatus = FcsEnumerations.LockStatus.ERROR;
            } else if (locked) {
                lockStatus = FcsEnumerations.LockStatus.LOCKED;
            } else if (unlocked) {
                lockStatus = FcsEnumerations.LockStatus.UNLOCKED;
            } else {
                lockStatus = FcsEnumerations.LockStatus.INTRAVEL;
            }

        } finally {

            updatingState = false;
            stateUpdated.signalAll();
            lock.unlock();
            this.publishData();
        }
    }
    

    /**
     * Updates the field readCurrent of the latch in reading the CPU of the
     * controller. This method is used in the simulator mainly.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update latch current in reading controller.")
    public void updateCurrent()  {
        try {
            this.readCurrent = this.latchController.readCurrent();
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(getName() + "=> ERROR IN READING CONTROLLER:",ex);
        }
        this.publishData();
    }

    /**
     * Returns true if autochanger CANopen hardware is connected and ready.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if autochanger CANopen hardware is connected and ready.")
    @Override
    public boolean isCANDevicesReady() {
        return autochanger.isCANDevicesReady();
    }

    /**
     * Close this latch.
     * @throws FcsHardwareException 
     * @throws RejectedCommandException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Close latch.")
    public void close()  {
        updateStateAndCheckSensors();
        if (isLocked()) {
            throw new RejectedCommandException(getName() + " is already LOCKED.");
        }
        autochanger.checkConditionsForClosingLatches();
        this.executeAction(FcsEnumerations.MobileItemAction.CLOSE, timeoutForClosing);
    }

    /**
     * Open this latch.
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Open latch.")
    public void open()  {
        updateStateAndCheckSensors();
        if (isUnlocked()) {
            throw new RejectedCommandException(getName() + " is already UNLOCKED.");
        }
        autochanger.checkConditionsForOpeningLatches();
        this.executeAction(FcsEnumerations.MobileItemAction.OPEN,timeoutForOpening);
    }

    /**
     * Return true if action is completed with success.
     * @param action
     * @return 
     */
    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        switch (action) {
            case OPEN:
                return lockStatus == LockStatus.UNLOCKED;

            case CLOSE:
                return lockStatus == LockStatus.LOCKED;
             
            default:
                assert false : action;
        }
        return false;
    }
    
    /**
     * Reads sensors, updates state and checks if sensors are in error.
     * If sensors are in error, raises an ALERT.
     */
    public void updateStateAndCheckSensors() {
        autochanger.updateStateWithSensors();
        checkSensors(AC_SENSOR_ERROR);
    }

    /**
     * Read sensors and update State. Used during an action.
     */
    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        autochanger.updateStateWithSensors();
    }

    /**
     * Start Action OPEN or CLOSE
     * @param action
     * @throws FcsHardwareException 
     */
    @Override
    public void startAction(FcsEnumerations.MobileItemAction action)  {
        
        switch (action) {
            case OPEN:
                latchController.enableAndWriteCurrent(this.currentToOpen);
                break;

            case CLOSE:
                latchController.enableAndWriteCurrent(this.currentToClose);
                break;
                
            default:
                assert false : action;
        }
    }

    /**
     * Stop action OPEN or CLOSE
     * @param action
     * @param delay
     * @throws FcsHardwareException 
     */
    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay)  {
        FCSLOG.debug(getName() + " is ABORTING action " + action.toString()
                + " within delay " + delay);
        this.latchController.off();
    }

    /**
     * @throws FcsHardwareException
     * @param action
     * @param delay 
     */
    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay) {
        //TODO : is there something different to do ? TBD
        abortAction(action, delay);
    }

    /**
     * Switch off controller after a CLOSE or OPEN action.
     * @param action
     * @throws FcsHardwareException
     */
    @Override
    public void postAction(FcsEnumerations.MobileItemAction action)  {
        //because we don't want to let the controller on power
        latchController.writeCurrent(0);
        this.publishData();
        FCSLOG.info(getName() + ":" + action.toString() + " completed - doing postAction.");
    }
    
    /**
     * Updates field controllerInFault when the controller notifies its observers and sends new values.
     * This class observes its controller to publish data when its controller is in fault or after a faultReset. 
     * Needed to update the GUI.
     *
     * @param source
     * @param v
     */
    @Override
    public void processUpdate(Observable source, ValueUpdate v) {
        updateControllerInFault(((EPOSController) source).getName(),v);
    }


    
    /**
     * Creates an return an object to be published on the STATUS bus.
     * @return 
     */
    public StatusDataPublishedByAutochangerLatch createStatusDataPublishedByLatch() {
        StatusDataPublishedByAutochangerLatch status = new StatusDataPublishedByAutochangerLatch();
        status.setName(getName());
        status.setLockSensorValue(lockSensors.isOn());
        status.setUnlockSensorValue(unlockSensors.isOn());
        status.setFilterPresenceSensorValue(filterEngagedSensors.isOn());
        status.setLockStatus(lockStatus);
        status.setControllerInFault(controllerInFault);
        status.setLockSensorsInError(lockSensors.isInError());
        status.setUnlockSensorsInError(unlockSensors.isInError());
        status.setFilterEngagedSensorsInError(filterEngagedSensors.isInError());
        status.setInError(lockStatus == LockStatus.ERROR);
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Publish Data on the Status Bus.")
    @Override
    public void publishData() {
        StatusDataPublishedByAutochangerLatch status = createStatusDataPublishedByLatch();
        this.getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData(getName(), status));
    }
}
