package org.lsst.ccs.subsystems.fcs;

import java.util.Observable;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.MASTER_ENCODER;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.PROFILE_POSITION;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.AC_TRUCKS_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.*;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerForLinearRail;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;

/**
 * This is a model for the autochanger trucks. The autochanger trucks move a
 * filter from STANDBY position to ONLINE position or HANDOFF position. The
 * ONLINE position is just in front of the focal front. The STANDBY position is
 * near the carousel. The HANDOFF position is a position when the loader can
 * hold a filter. The trucks are moved by 2 controllers, a master and a slave.
 * Each controller has an absolute encoder.
 * *****************************************
 * 
 * MASTER CONTROLLER FOR truckXminus
 * SLAVE CONTROLLER FOR truckXplus
 * 
 * *****************************************
 *
 * @author virieux
 */
public class AutoChangerTwoTrucksModule extends MobileItemModule {

    AutochangerTruckModule truckXminus;
    AutochangerTruckModule truckXplus;

    private final int minActualPositionValue = 0;
    private final int maxActualPositionValue = 998000;

    @ConfigurationParameter(description = "STANDBY position in micron")
    private int standbyPosition = 992000;

    @ConfigurationParameter(description = "HANDOFF position in micron")
    private int handoffPosition = 70000;

    @ConfigurationParameter(description = "ONLINE position in micron")
    private int onlinePosition = 4700;

    @ConfigurationParameter(description = "timeout for trucks motion from ONLINE to STANDBY in milliseconds")
    private long timeoutForTrucksMotion = 20000;

    private int current = 0;
    private int speed = 0;
    private int position;
    private int deltaPosition;
    
    /** position is reached if |targetPosition - position| < positionRange */
    private int positionRange = 10;

    private int absoluteTargetPosition = 0;
    private int relativeTargetPosition = 0;

    private EPOSControllerForLinearRail linearRailMasterController;
    private EPOSControllerForLinearRail linearRailSlaveController;
    private AutoChangerModule autochanger;



    /**
     * true if Handoff position sensors for TruckXminus and TruckXplus return
     * value 1*
     */
    private boolean atHandoff = false;

    /**
     * true if Online position sensors for TruckXminus and TruckXplus return
     * value 1 *
     */
    private boolean atOnline = false;

    /**
     * true if Standby position sensors for TruckXminus and TruckXplus return
     * value 1 *
     */
    private boolean atStandby = false;

    /**
     * Build a new AutoChangerTrucksModule with 6 sensors. Initialize fields
     * with initial default values.
     *
     * @param truckXminus
     * @param truckXplus
     */
    public AutoChangerTwoTrucksModule(
            AutochangerTruckModule truckXminus,
            AutochangerTruckModule truckXplus
    ) {
        this.truckXminus = truckXminus;
        this.truckXplus = truckXplus;
    }

    /**
     * Return actual trucks position. Doesn't read again the position on the
     * controller.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return actual trucks position. Doesn't read again the position on the controller.")
    public int getPosition() {
        return position;
    }

    /**
     * Return HANDOFF position. Doesn't read again the position on the
     * controller.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return HANDOFF position. Doesn't read again the position on the controller.")
    public int getHandoffPosition() {
        return handoffPosition;
    }

    /**
     * Return STANDBY position. Doesn't read again the position on the
     * controller.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return STANDBY position. Doesn't read again the position on the controller.")
    public int getStandbyPosition() {
        return standbyPosition;
    }

    /**
     * Return ONLINE position. Doesn't read again the position on the
     * controller.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return ONLINE position. Doesn't read again the position on the controller.")
    public int getOnlinePosition() {
        return onlinePosition;
    }


    /**
     * Return the timeout for trucks motion. If trucks motion duration is
     * greater than timeoutForTrucksMotion, an ALERT is raised.
     *
     * @return timeoutForTrucksMotion
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return timeout for trucks motion.")
    public long getTimeoutForTrucksMotion() {
        return timeoutForTrucksMotion;
    }

    /**
     * Returns the boolean field atHandoff. If the atHandoff boolean is being
     * updated and waits for a response from a sensor, this methods waits until
     * update is completed. If the field atHandoff is not being updated, it
     * returns its immediatly.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the carrier is at handoff position. "
            + "This command doesn't read again the sensors.")
    public boolean isAtHandoff() {
        return atHandoff;
    }

    /**
     * Returns the boolean field atOnline. If the atOnline boolean is being
     * updated and waits for a response from a sensor, this methods waits until
     * update is completed. If the field atOnline is not being updated, it
     * returns its immediatly.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the carrier is at ONLINE position. "
            + "This command doesn't read again the sensors.")
    public boolean isAtOnline() {
        return atOnline;
    }

    /**
     * Returns the boolean field atStandby. If the atHandoff boolean is being
     * updated and waits for a response from a sensor, this methods waits until
     * update is completed. If the field atStandby is not being updated, it
     * returns its immediatly..
     *
     * @return atStandby
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the carrier is at STANDBY position. "
            + "This command doesn't read again the sensors.")
    public boolean isAtStandby() {
        return atStandby;
    }

    /**
     * Return true if the homing of the trucks has been done.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the homing of the trucks has been done.")
    public boolean isHomingDone() {
        return linearRailSlaveController.isHomingDone()
                && linearRailMasterController.isHomingDone();
    }

    /**
     * Return true if TruckXminus position sensors or TruckXminus position
     * sensors are in error.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if TruckXminus position sensors or TruckXminus position sensors are in error.")
    public boolean isPositionSensorsInError() {
        return truckXminus.isPositionSensorsInError()
                || truckXplus.isPositionSensorsInError();
    }

    @Override
    public void initModule() {
        super.initModule();
        ComponentLookup lookup = getSubsystem().getComponentLookup();
        autochanger = (AutoChangerModule) lookup.getComponentByName("autochanger");
        linearRailMasterController = (EPOSControllerForLinearRail) lookup.getComponentByName("linearRailMasterController");
        linearRailSlaveController = (EPOSControllerForLinearRail) lookup.getComponentByName("linearRailSlaveController");
        //listens to its controllers to detect the controller's faultReset
        //or the emergency messages coming from the controller.
        if (linearRailMasterController instanceof Observable) {
            this.listens((Observable) linearRailMasterController);
        }
        if (linearRailSlaveController instanceof Observable) {
            this.listens((Observable) linearRailSlaveController);
        }
        
    }
    
    /**
     * check that truckXminus is controlled by master controller and
     * truckXplus is controlled by slave controller.
     */
    private void checkControllersName() {
        if (truckXminus.getControllerName().contains("Master")) {
            FCSLOG.info(getName() + " is controlled by " + truckXminus.getControllerName());
        } else {
            throw new FcsHardwareException("truckXminus has to be controlled by Master controller.");
        }
 
        if (truckXplus.getControllerName().contains("Slave")) {
            FCSLOG.info(getName() + " is controlled by " + truckXplus.getControllerName());
        } else {
            throw new FcsHardwareException("truckXplus has to be controlled by Slave controller.");
        }
    }

    /**
     * This method is called by method checkHardware in AutoChangerModule during
     * INITIALIZATION phase.
     *
     * @return
     * @throws HardwareException
     */
    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {
        super.checkHardware();
        checkControllersName();
        FCSLOG.debug(getName() + " checking hardware.");
        try {
            linearRailSlaveController.initializeAndCheckHardware();
            linearRailMasterController.initializeAndCheckHardware();
//            configureControllers();
            homing();
            //updatePosition
//            linearRailSlaveController.changeMode(MASTER_ENCODER);
//            linearRailSlaveController.activateBrakeAndDisable();
//            linearRailMasterController.changeMode(PROFILE_POSITION);
//            linearRailSlaveController.activateBrakeAndDisable();

        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(getName(), ex);

        } catch (SDORequestException ex) {
            throw new HardwareException(false, ex);

        } catch (FailedCommandException | FcsHardwareException ex) {
            FCSLOG.error(getName(), ex);
            throw new HardwareException(false, ex);
        }
//        updatePosition();
//        this.checkDeltaPosition();
        return TreeWalkerDiag.HANDLING_CHILDREN;
    }

    /**
     * Configure Autochanger trucks master and slave controllers.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Configure Autochanger trucks master and slave controllers.",
            alias = "initControllers")
    //tested on CPPM testbench in september 2015
    public void configureControllers() {
        FCSLOG.info(getName() + " Begin configuration of the controllers");
        linearRailMasterController.configureDigitalInputOfLinearRails();
        linearRailSlaveController.configureDigitalInputOfLinearRails();
        linearRailMasterController.configureDigitalOutputOfLinearRails();
        linearRailSlaveController.configureDigitalOutputOfLinearRails();
        FCSLOG.info(getName() + " End configuration of the controllers");
    }

    /**
     * Return true if both controllers are initialized and homing of trucks are
     * done.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if both controllers are initialized and homing of trucks are done.")
    public boolean isInitialized() {
        return linearRailMasterController.isInitialized()
                && linearRailSlaveController.isInitialized()
                && this.isHomingDone();
    }

    /**
     * This method sends Exceptions if the conditions for trucks motion are not
     * filled. Autochanger trucks motion is allowed if : - controllers are
     * initialized - homing is done - trucks are empty OR (loaded with a filter
     * but latches are closed AND carousel or loader doesn't hold filter)
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the motion of trucks are allowed.")
    public void checkConditionsForTrucksMotion() {

        if (!linearRailSlaveController.isInitialized() || !linearRailMasterController.isInitialized()) {
            String msg = getName() + " can't move autochanger trucks because controllers are not both initialized:\n";
            FCSLOG.error(msg);
            throw new RejectedCommandException(msg);
        }
        if (!isHomingDone()) {
            throw new RejectedCommandException(getName()
                    + " can't be moved because the homing has not been done yet. ");
        }

        updatePosition();
        checkDeltaPosition();
        autochanger.checkFilterSafetyBeforeMotion();
    }

    /**
     * Move Autochanger trucks to the Handoff position. If Autochanger trucks
     * are already at HANDOFF position, it does nothing.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move Autochanger trucks to the Handoff position.",
            alias = "goToHandoff")
    //tested on CPPM testbench in september 2015
    public void goToHandOff() {
        moveToAbsoluteTargetPosition(this.handoffPosition);
    }

    /**
     * Move Autochanger trucks to the Online position. If Autochanger trucks are
     * already at ONLINE position, it does nothing.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move Autochanger trucks to the Online position.")
    public void goToOnline() {
        this.getSubsystem().updateAgentState(FcsEnumerations.FilterState.MOVING_TRUCKS_TO_ONLINE,
                FcsEnumerations.FilterReadinessState.NOT_READY);
        //tested on CPPM testbench in september 2015
        moveToAbsoluteTargetPosition(this.onlinePosition);
    }

    /**
     * Move Autochanger trucks to the Standby position. If Autochanger trucks
     * are already at STANDBY position, it does nothing.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move Autochanger trucks to the Standby position.")
    public void goToStandby() {
        this.getSubsystem().updateAgentState(FcsEnumerations.FilterState.MOVING_TRUCKS_TO_STANDBY,
                FcsEnumerations.FilterReadinessState.NOT_READY);
        //tested on CPPM testbench in september 2015
        moveToAbsoluteTargetPosition(this.standbyPosition);
    }

    /**
     * Move the trucks to an absolute position given as argument.
     *
     * @param targetPosition
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move Autochanger trucks to the absolute position given as argument.",
            alias = "mobeABSPos")
    //tested on CPPM testbench in september 2015 and in April 2017
    public void moveToAbsoluteTargetPosition(int targetPosition) {

        if (targetPosition < minActualPositionValue || targetPosition > maxActualPositionValue) {
            throw new IllegalArgumentException(getName() + "argument has to be between "
                    + minActualPositionValue + " and " + maxActualPositionValue);
        }

        this.updatePosition();
        if (position == targetPosition) {
            FCSLOG.info(getName()
                    + " is already at target position=" + targetPosition);
        } else {

            checkConditionsForTrucksMotion();

            this.absoluteTargetPosition = targetPosition;

            FCSLOG.info(getName() + " going to absolute position: " + absoluteTargetPosition);

            this.executeAction(MOVE_TO_ABSOLUTE_POSITION, timeoutForTrucksMotion);
        }
    }
    
        /**
     * Move the trucks to an absolute position given as argument.
     *
     * @param targetPosition
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Move Autochanger trucks to the absolute position given as argument."
                    + " This command is dangerous because no control are done."
                    + " It can be executed when onlineClamps are in an UNKNOWN state.")
    //tested on CPPM testbench in september 2015 and in April 2017
    public void moveToAbsoluteTargetPositionAtYourOwnRisk(int targetPosition) {

        if (targetPosition < minActualPositionValue || targetPosition > maxActualPositionValue) {
            throw new IllegalArgumentException(getName() + "argument has to be between "
                    + minActualPositionValue + " and " + maxActualPositionValue);
        }

        this.updatePosition();
        if (position == targetPosition) {
            FCSLOG.info(getName()
                    + " is already at target position=" + targetPosition);
        } else {
            
            //check if LOCKED on ne bouge pas !

            this.absoluteTargetPosition = targetPosition;

            FCSLOG.info(getName() + " going to absolute position: " + absoluteTargetPosition);

            this.executeAction(MOVE_TO_ABSOLUTE_POSITION, timeoutForTrucksMotion);
        }
    }
    


    /**
     * Move the trucks to a relative position given as argument.
     *
     * @param relativePosition
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move Autochanger trucks to the relative position given as argument.",
            alias = "moveRELPos")
    //tested on CPPM testbench in september 2015
    //ONLY for small values for test purpose to realign the trucks
    public void moveToRelativeTargetPosition(int relativePosition) {

        checkConditionsForTrucksMotion();

        if (relativePosition < -2000 || relativePosition > 2000) {
            throw new IllegalArgumentException("relativePosition=" + relativePosition 
                    + " Relative position value has to be between -2000 and 2000");
        }

        linearRailMasterController.changeMode(EposMode.PROFILE_POSITION);
        updatePosition();

        this.relativeTargetPosition = position + relativePosition;

        if (relativeTargetPosition < minActualPositionValue || relativeTargetPosition > maxActualPositionValue) {
            throw new IllegalArgumentException(name + " actual position + argument has to be between "
                    + minActualPositionValue + " and " + maxActualPositionValue
                    + " relativeTargetPosition=" + relativeTargetPosition);
        }

        FCSLOG.info(getName() + " going to relative position: " + relativeTargetPosition);
        this.executeAction(MOVE_TO_RELATIVE_POSITION, timeoutForTrucksMotion);
    }

    @Override
    //tested on CPPM testbench in september 2015
    public void startAction(FcsEnumerations.MobileItemAction action) {

        switch (action) {
            case MOVE_TO_ABSOLUTE_POSITION:
                checkControllersMode();
                this.enableControllersAndReleaseBrake();
                linearRailMasterController.writeTargetPosition(this.absoluteTargetPosition);
                linearRailMasterController.writeControlWord("3F");
                break;
            //TODO this case is useless definitly
            case MOVE_TO_RELATIVE_POSITION:
                checkControllersMode();
                this.enableControllersAndReleaseBrake();
                linearRailMasterController.writeTargetPosition(this.relativeTargetPosition);
                linearRailMasterController.writeControlWord("7F");
                break;

            default:
                assert false : action;
        }
    }

    /**
     * This method is used in the startAction method. It is was has to be done
     * before moving the trucks.
     */
    private void enableControllersAndReleaseBrake() {
        linearRailMasterController.enable();
        linearRailSlaveController.enable();
        linearRailMasterController.doReleaseBrake();
        if (linearRailMasterController.isEnabled()) {
            linearRailSlaveController.doReleaseBrake();
        } else {
            throw new FcsHardwareException(getName() + " can't release Slave Controller brake "
                    + " because Master Controller is disabled.");
        }
    }

    private void checkControllersMode() {
        if (!linearRailMasterController.isInMode(EposMode.PROFILE_POSITION)) {
            throw new FcsHardwareException(getName() + " : linearRailMasterController is not"
                    + " in PROFILE_POSITION mode, can't move.");
        }

        if (!linearRailSlaveController.isInMode(EposMode.MASTER_ENCODER)) {
            throw new FcsHardwareException(getName() + " : linearRailSlaveController is not"
                    + " in MASTER_ENCODER mode, can't move. Slave controller mode =" + linearRailSlaveController.getMode() +
                    " readMode=" + linearRailSlaveController.readMode());
        }
    }

    /**
     * move slave truk to position. When moving along the rails with slave
     * controller in MASTER_ENCODER mode, at the end of motion there is a small
     * misalignement.
     *
     * This consists in setting slaveController mode to PROFILE_POSITION, go to
     * absolute position pos, and then set slaveController mode back to
     * MASTER_ENCODER mode.
     *
     *
     * @param masterPosition
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Align slave controller position to master controller position.")
    public void alignSlave(int masterPosition) {
        //TODO test that masterPosition entered is 
        if (Math.abs(linearRailMasterController.readPosition() - masterPosition) > positionRange) {
            throw new IllegalArgumentException(getName() + " masterPosition and position of master controller "
                    + " are too different.");
        }
        //TODO clamps must be openend
        try {
            linearRailSlaveController.changeMode(PROFILE_POSITION);
            
            if (!linearRailSlaveController.isInMode(PROFILE_POSITION)) {
                throw new FcsHardwareException(getName() + " can't align slave truck because"
                        + " couldn't change its mode to PROFILE_POSITION.");
            }
            linearRailSlaveController.enableAndWriteAbsolutePosition(masterPosition);
            int sleepTime = 700;
            linearRailSlaveController.checkTargetReached();
            Thread.sleep(sleepTime);
            linearRailSlaveController.activateBrake();
            linearRailSlaveController.changeMode(MASTER_ENCODER);
            linearRailSlaveController.disable();
        } catch (InterruptedException ex) {
            FCSLOG.error(ex);
        }
    }

    @Override
    //TODO test on CPPM test bench
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay) {
        FCSLOG.info("ABORTING action: " + action.toString());

        this.linearRailSlaveController.off();

        this.linearRailMasterController.off();
    }

    @Override
    //tested on CPPM testbench in september 2015
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay) {
        this.linearRailMasterController.quickStop();
        this.linearRailSlaveController.quickStop();
    }

    @Override
    //tested on CPPM testbench in september 2015
    public void postAction(FcsEnumerations.MobileItemAction action) {
        //because we don't want to let the controllers on power
        this.linearRailSlaveController.activateBrakeAndDisable();
        this.linearRailMasterController.activateBrakeAndDisable();
        FCSLOG.info(getName() + ":" + action.toString() + " completed - doing postAction: checking position sensors.");

        if (action == MOVE_TO_ABSOLUTE_POSITION) {
            checkPositionSensors();
        }
        updatePosition();
//        alignSlave();
    }

    /**
     * This method is called in the postAction method to check that after a
     * motion to well known position as STANDBY, HANDOFF and ONLINE, position
     * sensors confirm the trucks position. Otherwise it throws a
     * FailedCommandException.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void checkPositionSensors() {
        /*Read position sensors and check that values return are consistant with trucks position
         otherwise throw an Exception.*/
        autochanger.updateStateWithSensors();
        if (this.absoluteTargetPosition == this.standbyPosition && !this.atStandby) {
            throw new FailedCommandException(name
                    + ": check with sensors: STANDBY sensors don't confirm trucks position.");

        } else if (this.absoluteTargetPosition == this.handoffPosition && !this.atHandoff) {
            FCSLOG.info("this.absoluteTargetPosition=" + this.absoluteTargetPosition);
            FCSLOG.info("this.handoffPosition=" + this.handoffPosition);
            FCSLOG.info("this.atHandoff=" + this.atHandoff);
            throw new FailedCommandException(name
                    + ": check with sensors: HANDOFF sensors don't confirm trucks position.");

        } else if (this.absoluteTargetPosition == this.onlinePosition && !this.atOnline) {
            throw new FailedCommandException(name
                    + ": check with sensors: ONLINE sensors don't confirm trucks position.");
        }
    }

    /**
     * Do homing of both controllers.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Do homing for both  controllers.")
    public void homing() {
        FCSLOG.debug(getName() + " ==> END homing of linearRails controllers");
        linearRailMasterController.homing();
        linearRailSlaveController.homing();
        updatePosition();
        FCSLOG.debug(getName() + " ==> END homing of linearRails controllers");
    }

    /**
     * Updates the field position of the carrier in reading the CPU of the
     * controller.
     * *****************************************
     * MASTER CONTROLLER FOR truckXminus
     * SLAVE CONTROLLER FOR truckXplus
     * *****************************************

     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    //tested on CPPM testbench in april 2017
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update trucks position in reading controller.")
    public void updatePosition() {
        this.truckXminus.updatePosition();
        this.truckXplus.updatePosition();
        this.position = linearRailMasterController.readPosition();
        this.deltaPosition = this.position - truckXplus.getPosition();
        this.publishData();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read position on controllers, compute difference and throw exception"
            + "if difference is > 1000.")
    public void checkDeltaPosition() {
        this.position = linearRailMasterController.readPosition();
        int slavePosition = linearRailSlaveController.readPosition();
        this.deltaPosition = this.position - slavePosition;
        if (Math.abs(deltaPosition) > 1000) {
            this.raiseAlarm(AC_TRUCKS_ERROR, "Too much difference between trucks position. "
                    + "Alignment of Slave Truck to be done.");
            throw new FcsHardwareException();
        }
        this.publishData();
    }

    /**
     * For the GUI and end users for test purpose. Update positon in reading the
     * master controller and returns the value read.
     *
     * @return trucks position
     * @throws RejectedCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    //tested on CPPM testbench in september 2015
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update trucks position in reading controller and return position.")
    public int readPosition() {
        updatePosition();
        return position;
    }

    /**
     * @return true if both controllers are initialized and homing of trucks are
     * done.
     */
    @Override
    //tested on CPPM testbench in september 2015
    public boolean isCANDevicesReady() {
        //commented out for CPPM testbench
        //return autochanger.isCANDevicesReady();
        return this.isInitialized();
    }

    @Override
    //tested on CPPM testbench in May 2017
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        switch (action) {
            case MOVE_TO_ABSOLUTE_POSITION:
                return Math.abs(this.position - this.absoluteTargetPosition) < positionRange;

            case MOVE_TO_RELATIVE_POSITION:
                return Math.abs(this.position - this.relativeTargetPosition) < positionRange;

            default:
                assert false : action;
        }
        return false;
    }

    @Override
    //tested on CPPM testbench in september 2015
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        try {
            linearRailMasterController.checkFault();
            linearRailSlaveController.checkFault();
            updatePosition();
            FCSLOG.debug(getName() + " position=" + this.position);
            autochanger.updateStateWithSensors();
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(getName() + "=> SDO ERROR IN READING CONTROLLER:", ex);
        } catch (SDORequestException ex) {
            FCSLOG.error(getName() + "=> SDO ERROR IN READING CONTROLLER:", ex);
        }
    }

    void updateStateWithSensors(String[] readHexaValues) {
        truckXminus.updateStateWithSensors(readHexaValues);
        truckXplus.updateStateWithSensors(readHexaValues);
        this.atHandoff = truckXminus.isAtHandoff() && truckXplus.isAtHandoff();
        this.atOnline = truckXminus.isAtOnline() && truckXplus.isAtOnline();
        this.atStandby = truckXminus.isAtStandby() && truckXplus.isAtStandby();
    }

    /**
     * release brake on slave controller. This command is forbidden if slave is
     * in MASTER_ENCODER mode and master is disabled because in this case,
     * releasing slave could lead to a filter fall.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Release slave controller brake.")
    public void releaseBrakeSlave() {
        if (linearRailSlaveController.isInMode(MASTER_ENCODER)
                && !linearRailMasterController.isEnabled()) {
            throw new RejectedCommandException(getName() + " couldn't release slave controller brake"
                    + "because master controller is disabled and slave controller is in MASTER_ENCODER mode");
        } else {
            linearRailSlaveController.doReleaseBrake();
        }
    }

    /**
     * release brake on master controller. This command is forbidden if slave is
     * in MASTER_ENCODER mode and master is disabled because in this case,
     * releasing slave could lead to a filter fall.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Release master controller brake.")
    public void releaseBrakeMaster() {
        linearRailMasterController.doReleaseBrake();
    }

    /**
     * Creates an object to be published on the status bus.
     *
     * @return
     */
    public StatusDataPublishedByAutoChangerTrucks createStatusDataPublishedByAutoChangerTrucks() {
        StatusDataPublishedByAutoChangerTrucks s = new StatusDataPublishedByAutoChangerTrucks();
        s.setPosition(position);
        s.setDeltaPosition(deltaPosition);
        s.setCurrent(current);
        s.setSpeed(speed);
        s.setHomingDone(isHomingDone());
        s.setAtHandoff(atHandoff);
        s.setAtOnline(atOnline);
        s.setAtStandby(atStandby);
        s.setInError(isPositionSensorsInError());
        return s;
    }

    @Override
    public void publishData() {
        this.getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData(getName(),
                createStatusDataPublishedByAutoChangerTrucks()));
    }

}
