
package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.OperationalState;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.AC_SENSOR_ERROR;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.PlutoGatewayInterface;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * This class is used with the final hardware in which we have a clamp online to
 * hold the filter at online position. It's not used for Single Filter Test but
 * is used for testbenchCPPM.
 *
 * @author virieux
 */
public class AutoChangerModule extends Module implements HardwareController, FilterHolder, AlertRaiser {

    @ConfigurationParameter
    private final String plutoGatewayName;
        
    private BridgeToHardware bridge;
    private PlutoGatewayInterface plutoGateway;
    private FilterManager filterManager;
    private FilterIdentificator filterIdentificator;

    private final ComplementarySensors loaderConnectedSensors;

    private final AutoChangerTwoTrucksModule autochangerTrucks;
    private final AutochangerTwoLatches latches;
    private final AutochangerThreeOnlineClamps onlineClamps;
    
    private int filterOnTrucksID = 0;
    
    private FilterHolder carousel;

    private FilterHolder loader;
    
    //si on lit la même valeur sur les 2 c'est OK
    //plage de valeurs acceptables pour l'horizontalité
    //max acceptable pour la différence entre les 2 inclinometres
    private final InclinometerModule inclinometerXminus;
    private final InclinometerModule inclinometerXplus;


    private final Lock lock = new ReentrantLock();
    private final Condition stateUpdated = lock.newCondition();
    private volatile boolean updatingState = false;

    /**
     * Builds  new AutoChangerModule with following arguments :
     * @param plutoGatewayName
     * @param loaderConnectedSensors
     * @param trucks
     * @param latches
     * @param onlineClamps 
     * @param inclinometerXminus 
     * @param inclinometerXplus 
     */
    public AutoChangerModule(
            String plutoGatewayName,
            ComplementarySensors loaderConnectedSensors,
            AutoChangerTwoTrucksModule trucks,
            AutochangerTwoLatches latches,
            AutochangerThreeOnlineClamps onlineClamps,
            InclinometerModule inclinometerXminus,
            InclinometerModule inclinometerXplus) {
        this.plutoGatewayName = plutoGatewayName;
        this.autochangerTrucks = trucks;
        this.latches = latches;
        this.onlineClamps = onlineClamps;
        this.loaderConnectedSensors = loaderConnectedSensors;
        this.inclinometerXminus = inclinometerXminus;
        this.inclinometerXplus = inclinometerXplus;
    }
    
    /**
     * Return true if an error has been detected on sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
        description = "Return true if an error has been detected on sensors.")
    public boolean isSensorsInError() {
        return latches.isInError() || onlineClamps.isInError() || autochangerTrucks.isPositionSensorsInError();
    }




    /**
     * returns autochangerTrucks to be able to send commands to trucks in fcs subsystem.
     * @return autochangerTrucks
     */
    public AutoChangerTwoTrucksModule getAutochangerTrucks() {
        return autochangerTrucks;
    }

    /**
     * returns onlineClamps to be able to send commands to onlineClamps in fcs subsystem.
     * @return onlineClamps
     */
    public AutochangerThreeOnlineClamps getOnlineClamps() {
        return onlineClamps;
    }

    /**
     * If autochanger holds a filter, returns filterID, else returns 0.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "If autochanger holds a filter, return filterID, else return 0.")    
    public int getFilterOnTrucksID() {
        return filterOnTrucksID;
    }

    /**
     * Used for simulator only.
     * @param filterOnTrucksID 
     */
    public void setFilterOnTrucksID(int filterOnTrucksID) {
        this.filterOnTrucksID = filterOnTrucksID;
    }
    
    /**
     * If autochanger holds a filter, return filter name else return null.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "If autochanger holds a filter, return filter name else return null.")
    public String getFilterOnTrucksName() {
        if (this.isHoldingFilter()) {
            return filterManager.getFilterNameByID(filterOnTrucksID);
        } else {
            return null;
        }
    }
    
    /**
     * If autochanger holds a filter at ONLINE, return filter name else return null.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "If autochanger holds a filter at ONLINE, return filter name else return null.")    
    public String getFilterOnlineName() {
        if (this.autochangerTrucks.isAtOnline() && !this.isEmpty()) {
            return filterManager.getFilterNameByID(filterOnTrucksID);
        } else {
            return null;
        }
    }

    /**
     * Returns the boolean field loaderConnected. If the loaderConnected boolean
     * is being updated and waits for a response from a sensor, this methods
     * waits until loaderConnected is updated. If the field loaderConnected is
     * not being updated, it returns immediatly the field loaderConnected.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the loader is connected to the camera. "
            + "This command doesn't read again the sensors.")
    public boolean isLoaderConnected() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(getName() + ": has been interrupted while waiting for end of update.");
                }

            }
            return loaderConnectedSensors.isOn() && !loaderConnectedSensors.isInError();

        } finally {
            lock.unlock();
        }
    }

    /**
     * TODO when autochanger is in standalone mode this method has to answer the value of the
     * fake carousel sensor.
     * @return 
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the carousel is holding the filter at STANDBY position.")
    public boolean isCarouselHoldingFilterAtStandby()   {
            return this.carousel.isAtStandby() && this.carousel.isHoldingFilter();
    }
    
    /**
     * Return true if loader is connected and holding a filter.
     * This command doesn't read again the sensors.
     * @return
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if loader is connected and holding a filter. "
            + "This command doesn't read again the sensors.")
    public boolean isLoaderHoldingFilterAtHandoff()  {
        return isLoaderConnected() && loader.isHoldingFilter();
    }
    
   
    @Override
    public void initModule() {
        FCSLOG.info(getName() + ": init MODULE");
        ComponentLookup lookup = getSubsystem().getComponentLookup();
        this.plutoGateway = (PlutoGatewayInterface) lookup.getComponentByName(plutoGatewayName);
        this.bridge = (BridgeToHardware) lookup.getComponentByName("tcpProxy");
        this.carousel = (FilterHolder) lookup.getComponentByName("carousel");
        this.loader = (FilterHolder) lookup.getComponentByName("loader");
        this.filterManager = (FilterManager) lookup.getComponentByName("filterManager");
        this.filterIdentificator = (FilterIdentificator) lookup.getComponentByName("filterIdentificator");
    }

    /**
     * @return true if all autochanger hardware is initialized and bridge is hardwareReady.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if all autochanger CANopen devices are booted, identified and initialized.")
    public boolean isCANDevicesReady() {
        return bridge.isCANDevicesReady();
    }
    
    /**
     * @return if all autochanger CANopen devices are booted, identified and nitialized and homing of the 
     * controllers is done.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if all autochanger CANopen devices are booted, identified and "
                    + "initialized and homing of the controllers is done.")
    public boolean isHardwareReady() {
        return bridge.isCANDevicesReady() && isInitialized();
    }    

    /**
     * Return true if plutoGateway, trucks, latches and onlineClamps are initialized.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if all autochanger hardware is initialized.")
    public boolean isInitialized() {
        return plutoGateway.isInitialized() && autochangerTrucks.isInitialized() 
                && latches.isInitialized() && onlineClamps.isInitialized();
    }

    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {
        FCSLOG.debug(getName() + " checking hardware.");
        try {
            this.plutoGateway.initializeAndCheckHardware();
            updateStateWithSensors();
        } catch (FcsHardwareException ex) {
            throw new HardwareException(true, ex);
        }
        /* checkHardware has to be executed on my children which are HardwareController 
        (latches, onlineClamps, trucks)*/
        return TreeWalkerDiag.GO;
    }

    @Override
    public void checkStarted() throws HardwareException {
        FCSLOG.info(getName() + " BEGIN checkStarted");

        //check that all hardware is booted and identified.
        bridge.checkHardware();

        try {
            this.plutoGateway.initializeAndCheckHardware();
            updateStateWithSensors();
            autochangerTrucks.checkHardware();

        } catch (FcsHardwareException ex) {
            throw new HardwareException(true, ex);
        }

    }

    /**
     * This method is executed during a shutdown of the subystem to prevent a
     * shutdown when some hardware is moving. Nothing to do here because all the
     * moving hardware of the autochanger is represented by the class
     * MobilItemModule where the method checkStopped is implemented.
     *
     * @throws HardwareException
     */
    @Override
    public void checkStopped() throws HardwareException {
    }

    /**
     * Check if latches can be opened. 
     * If latches can't be openened because the conditions for opening are not filled, 
     * this command throws a RejectedCommandException.
     * If latches are in error, it throws a FcsHardwareException.
     * 
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the latches can be opened.")
    public void checkConditionsForOpeningLatches()  {
        FCSLOG.info(getName() + " checking pre-conditions for opening latches");

        updateStateWithSensors();

        checkLatchesInitialized();
            
        if (latches.isInError()) {
            throw new FcsHardwareException(getName()  + ": latches are in ERROR state - can't unlockClamps latches.");
            
//        } else if(!this.getSubsystem().isInState(OperationalState.NORMAL)) {
//            /* for test bench we want to be able to open latches even if there is no filter*/
//            raiseWarning(getName()+":SECURITY","Security conditions bypassed"," in checkConditionsForOpeningLatches");
//            
        } else if (isEmpty()) {
            throw new RejectedCommandException(getName() + ": no filter in autochanger - can't open latches.");
            
        } else if (!autochangerTrucks.isAtStandby() && !autochangerTrucks.isAtHandoff()) {
            throw new RejectedCommandException(getName() + ": autochanger is loaded with a filter but is not "
                    + "at handoff position neither at standby - can't open latches.");
            
        } else if (autochangerTrucks.isAtStandby()) {
            checkConditionsForOpeningLatchesAtStandby();
            
        } else if (autochangerTrucks.isAtHandoff()) { 
            checkConditionsForOpeningLatchesAtHandoff();
        }
    }
    
    private void checkLatchesInitialized() {
        if (!latches.isInitialized()) {
            throw new RejectedCommandException(getName()  + ": latches are not intialized. "
                    + "Please initialize hardware first.");
        }
    }
    
    private void checkConditionsForOpeningLatchesAtStandby() {
        /* autochanger holds a filter at STANDBY position*/
        if (isCarouselHoldingFilterAtStandby()) {
            FCSLOG.info(getName() + " carousel is holding filter at STANDBY => latches can be open safely.");
            
        } else {
            throw new RejectedCommandException(getName() + ": autochanger is loaded with a filter and is  "
                    + "at STANDBY position but carousel doesn't hold the filter "
                    + "- can't open latches.");
        }
    }
    
    private void checkConditionsForOpeningLatchesAtHandoff() {
        /* autochanger holds a filter at HANDOFF position*/
        if (isLoaderHoldingFilterAtHandoff()) {
            FCSLOG.info(getName() + " loader is holding filter at HANDOFF => latches can be open safely.");
            
        } else {
            throw new RejectedCommandException(getName() + ": autochanger is loaded with a filter and is  "
                    + "at HANDOFF position but loader doesn't hold the filter "
                    + "- can't open latches.");
        }
    }

    /**
     * Check if Autochanger latches can be closed.
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if Autochanger latches can be closed.")
    public void checkConditionsForClosingLatches()  {
        
        FCSLOG.info(getName()+" checking conditions for closing latches.");

        updateStateWithSensors();
        String message;
        checkLatchesInitialized();

        if (latches.isInError()) {
            throw new RejectedCommandException(getName() + ": latches are in ERROR state - can't close latches.");
                        
//        } else if(!this.getSubsystem().isInState(OperationalState.NORMAL)) {
//            /* for test bench we want to be able to close latches even if there is no filter*/
//            raiseWarning(getName()+":SECURITY","Security conditions bypassed"," in checkConditionsForClosingLatches");
        
        } else if (isEmpty()) {
            message = getName() + ": no filter in autochanger - can't close latches.";
            throw new RejectedCommandException(message);
            
        } else if (!autochangerTrucks.isAtStandby() && !autochangerTrucks.isAtHandoff()) {
            throw new RejectedCommandException(getName() + ": autochanger is not "
                    + "at handoff position neither at standby - can't close latches.");
        } 
    }
    
    /**
     * log if actions on online clamps are allowed,
     * throws an exception otherwise.
     */
    @Command(type = Command.CommandType.QUERY, description = "log if actions on online clamps are allowed,"
            + "throws an exception otherwise.")
    public void checkConditionsForActioningOnlineClamps() {
        if (!isEmpty() && isAtOnline()) {
            FCSLOG.info(getName() + " autochangerTrucks are at ONLINE, a filter is there : "
                    + "actions are allowed.");
        } else if (!isAtOnline()) {
            throw new RejectedCommandException(getName() + " actions not allowed on ONLINE clamps "
                    + "when trucks are not at ONLINE.");
        } else {
            throw new RejectedCommandException(getName() + " actions not allowed on ONLINE clamps "
                    + "when no filter on trucks.");
        }
        
    }
    
    /**
     * This methods checks that filter can be moved by trucks.
     * Returns immediately if autochanger trucks are empty.
     * Throws RejectedCommandException if a filter is in trucks AND held both by autochanger
     * and (loader or carousel).
     * 
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws RejectedCommandException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, 
            description = "Check if its safe for filters to move autochanger trucks.")
    public void checkFilterSafetyBeforeMotion()  {
        
        String msg = getName() + " can't move autochanger trucks because ";
        this.updateStateWithSensors();
        
        if (this.isSensorsInError()) {
            handleSensorsError();
            
        } else if (isEmpty()) {
            FCSLOG.info(getName() + " trucks are empty - can move");
            
        } else if (isAtStandby()) {
            checkFilterSafetyAtStandby(msg);
            
        } else if (autochangerTrucks.isAtHandoff()) {
            FCSLOG.info(getName() + " COUCOU : trucks are AT HANDOFF ");
            checkFilterSafetyAtHandoff(msg);
            
        } else if (this.autochangerTrucks.isAtOnline()) {
            FCSLOG.info(getName() + " COUCOU : trucks are at ONLINE ");
            checkFilterSafetyAtOnline(msg);
            
        } else if (!isHoldingFilter()) {
            throw new RejectedCommandException(msg + 
                "latches are not LOCKED");
        }
    }
    
    /**
     * Check if trucks can be moved when loaded with a filter at STANDBY.
     * @param message
     * @throws FcsHardwareException 
     */
    private void checkFilterSafetyAtStandby(String message)  {
        if (isHoldingFilter() && isCarouselHoldingFilterAtStandby())  {
            throw new RejectedCommandException(message + 
                    "both carousel and autochanger are holding filter at STANDBY");
            
        } else if (!isHoldingFilter() && !isCarouselHoldingFilterAtStandby()) {
            throw new RejectedCommandException(message + 
                    "neither carousel nor autochanger are holding filter at STANDBY");
        } else {
            FCSLOG.info(getName() + " filter safe at STANDBY - can move");
        }
    }
    
    /**
     * Check if trucks can be moved when loaded with a filter at HANDOFF.
     * @param message
     * @throws FcsHardwareException 
     */
    private void checkFilterSafetyAtHandoff(String message)  {
        if (isHoldingFilter() && this.isCarouselHoldingFilterAtStandby()) {
            throw new RejectedCommandException(message
                + "autochanger is holding filter at HANDOFF but another filter is at STANDBY "
                    + "- can't move trucks");
        }
        if (isHoldingFilter() && isLoaderHoldingFilterAtHandoff()) {
            throw new RejectedCommandException(message
                + "both loader and autochanger are holding filter at HANDOFF");
        } else if (!isHoldingFilter() && !isLoaderHoldingFilterAtHandoff()) {
            throw new RejectedCommandException(message + 
                    "neither loader nor autochanger are holding filter at HANDOFF");
        } else {
            FCSLOG.info(getName() + " filter safe at HANDOFF - can move");
        }
    }
    
    /**
     * Check if trucks can be moved when loaded with a filter at ONLINE.
     * @param message
     * @throws FcsHardwareException 
     */
    private void checkFilterSafetyAtOnline(String message)  {
        if (isHoldingFilter() && this.isCarouselHoldingFilterAtStandby()) {
            throw new RejectedCommandException(message
                + "autochanger is holding filter at ONLINE but another filter is at STANDBY "
                    + "- can't move trucks");
        }
        
        if (isHoldingFilter() && !onlineClamps.isOpened()) {
            throw new RejectedCommandException(message
                + "onlineClamps have to be opened on filter at ONLINE - can't move trucks.");
        }
            
        if (!isHoldingFilter() && !onlineClamps.isLocked()) {
            throw new RejectedCommandException(message + 
                    "neither latches nor onlineClamps are holding filter at ONLINE");
        } else {
            FCSLOG.info(getName() + " filter safe at ONLINE - can move");
        }
    }
    
    /**
     * Raise ALERT and throw FcsHardwareException with a detailed message when error is detected in sensors.
     * @throws FcsHardwareException 
     */
    private void handleSensorsError()  {
        String msg = getName() + " error detected in sensors:";
        if (this.autochangerTrucks.isPositionSensorsInError()) {
            msg += " trucks position sensors";
        }
        if (this.latches.isInError()) {
            msg += " latches sensors";
        }
        if (this.onlineClamps.isInError()) {
            msg += " onlineClamps sensors";
        }
        this.raiseAlarm(AC_SENSOR_ERROR, msg);
        throw new FcsHardwareException(msg);
    }
    
    
    /**
     * Return true if a filter is in trucks and latches are LOCKED.
     * @return 
     */
    @Override
    public boolean isHoldingFilter() {
        return !this.latches.isEmpty() && this.latches.isLocked();
    }
    
    
    

    /**
     * This method reads all the sensors and updates the autochangerTrucks, latches and online clamps state. 
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, 
            description = "Update clamp state in reading sensors.")
    @Override
    public void updateStateWithSensors()  {
        
        plutoGateway.checkInitialized();

        lock.lock();
        try {
            updatingState = true;
            updateStateWithSensors(this.plutoGateway.readNewHexaValues());

        } finally {

            updatingState = false;
            stateUpdated.signalAll();
            lock.unlock();
            this.publishData();
            this.bridge.publishData();
            this.autochangerTrucks.publishData();
            this.latches.publishData();
            this.onlineClamps.publishData();
        }
    }
    
    public void updateStateFromPDO() {
        String[] readHexaValues = this.plutoGateway.readHexaValuesFromPDO();
        updateStateWithSensors(readHexaValues);
    }
    
    private void updateStateWithSensors(String[] readHexaValues) {
        //update of loaderConnectedSensors
        this.loaderConnectedSensors.updateValues(readHexaValues);
        this.autochangerTrucks.updateStateWithSensors(readHexaValues);
        this.latches.updateStateWithSensors(readHexaValues);
        this.onlineClamps.updateStateWithSensors(readHexaValues);
        this.filterIdentificator.updateSensorsValues(readHexaValues);
        //commented in May 2017
//        this.inclinometerXminus.updateInclinaison(readHexaValues);
//        this.inclinometerXplus.updateInclinaison(readHexaValues);
    }

    /**
     * Return true if autochanger trucks are at HANDOFF.
     * This command doesn't read again the sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if autochanger trucks are at HANDOFF. "
            + "This command doesn't read again the sensors.")
    @Override
    public boolean isAtHandoff() {
        return autochangerTrucks.isAtHandoff();
    }
    
    
    /**
     * Return true if autochanger trucks are at STANDBY.
     * This command doesn't read again the sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if autochanger trucks are at STANDBY. "
            + "This command doesn't read again the sensors.")
    @Override
    public boolean isAtStandby() {
        return autochangerTrucks.isAtStandby();
    }
    
    /**
     * return true if autochanger trucks are at ONLINE.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, 
            description = "Return true if autochanger trucks are at ONLINE. "
            + "This command doesn't read again the sensors.")    
    public boolean isAtOnline() {
        return autochangerTrucks.isAtOnline();
    }

    /**
     * Return true if there is no filter in the autochanger.
     * This command doesn't read again the sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if there is no filter in the autochanger. "
            + "This command doesn't read again the sensors.")
    public boolean isEmpty() {
        return latches.isEmpty();
    }
    
    
    
    
    

    void grabFilterAtHandoff(Filter aFilter) {
        throw new UnsupportedOperationException("Not supported yet."); 
    }
    
    void ungrabFilterAtHandoff(Filter aFilter) {
        throw new UnsupportedOperationException("Not supported yet."); 
    }

    void moveFilterToHandoff(Filter aFilter) {
        throw new UnsupportedOperationException("Not supported yet."); 
    }
    

    /**
     * Close latches on filter at STANDBY.
     * Identify filter at STANDBY in reading identifier ID.
     * Update filter location.
     * @throws RejectedCommandException if autochanger trucks are not at STANDBY.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, alias = "grabFilter",
        description = "Go to STANDBY to grab filter and stay at STANDBY.")
    public void grabFilterAtStandby() {
        if (!this.isAtStandby()) {
            throw new RejectedCommandException(getName() + ": can't grab a filter when trucks are not at STANDBY.");
        } 
        FCSLOG.info(getName() + ": grabbing a filter at standby position.");
        latches.close();
        filterOnTrucksID = identifyFilterAtStandby();
        filterManager.setFilterLocation(filterOnTrucksID,FcsEnumerations.FilterLocation.AUTOCHANGER);
        publishData();
        FCSLOG.info(getName() + ": filter " + filterOnTrucksID + " is grabbed on autochanger");
    }
    
    /**
     * Open the latches at standby position.
     * position.
     *
     * @throws RejectedCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, 
            description = "Ungrab filter at STANDBY")
    public void ungrabFilterAtStandby()  {
        if (!this.isCarouselHoldingFilterAtStandby()) {
            throw new RejectedCommandException(getName()+": can't ungrab a filter at STANDBY when carousel is not holding filter.");
        }
        FCSLOG.info(getName() + ": ungrabing filter at standby position.");
        latches.open();
        //eject filter from autochanger
        synchronized (this) {
            filterManager.setFilterLocation(filterOnTrucksID,FcsEnumerations.FilterLocation.CAROUSEL);
            filterOnTrucksID = 0;
        }
        publishData();
    }
    
    
    /**
     * Read the filterID sensors at STANDBY to check filterID.
     * @return 
     */
    private int identifyFilterAtStandby() {
        return filterIdentificator.readFilterIdentifier(); 
    }

    
    /**
     * Creates an Object to be published on the STATUS bus.
     * @return 
     */
    public StatusDataPublishedByAutoChanger createStatusDataPublishedByAutoChanger() {
        StatusDataPublishedByAutoChanger status = new StatusDataPublishedByAutoChanger();
        status.setName(getName());
        status.setLoaderConnectedSensorValue(this.loaderConnectedSensors.isOn());
        status.setLoaderConnectedSensorsInError(this.loaderConnectedSensors.isInError());
        status.setInclinaisonXminus(inclinometerXminus.getInclinaison());
        status.setInclinaisonXplus(inclinometerXplus.getInclinaison());
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    public void publishData() {
        this.getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData("autochangerGeneral", 
                createStatusDataPublishedByAutoChanger()));
    }

    /********************************************************************************************
     * end of Methods to write current to controllers of clampX- and clampXplus with a ramp
    /********************************************************************************************
     */
    /**
     * Save on disk the values of current sent to online clamps controllers.
     * This command has to be executed before shutting down the subsystem to store current values sent to controllers.
     * The goal is to be able to restart next time and know which value of current has been sent to controller.
     * To know this value of current is important because the pressure exerted on the filter by the online clamps
     * depends on the value of this current.
     * Higher is the current value, higher is the pressure.
     * And the process of opening or closing of a online clamp depends on this previous sent current value.
     */
    public void saveCurrentSentToOnlineClampsControllers() {
        try {
            getSubsystem().getSubsystemConfigurationEnvironment().saveChangesForCategories(
                    AutochangerOnlineClampModule.ONLINE_CLAMP_CONFIG_CATEGORY);
        } catch (Exception ex) {
            FCSLOG.error("Unable to save ONLINE clamps currents.", ex);
        }
    }
    
    @Override
    public void shutdownNow() {
        saveCurrentSentToOnlineClampsControllers();
        super.shutdownNow();
    }

}
