
package org.lsst.ccs.subsystems.fcs.singlefiltertest;

import org.lsst.ccs.UsesSubsystem;
import org.lsst.ccs.description.ComponentLookup;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.DigitalSensor;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;

/**
 *
 * A simulated actuator for the latch used in single-filter-test.
 * This is used only for single-filter-test simulation.
 * 
 * @author virieux
 */
public class SimuLatchActuatorModule extends LatchActuatorModule implements UsesSubsystem {

    private final SimuCompactIOModule simuDIO;

    /**
     * Build a new SimuLatchActuatorModule
     * @param currentLimitInVolt
     * @param speedVoltageInVolt
     * @param speedVoltageMinValue
     * @param speedVoltageMaxValue
     * @param simuDIO
     */
    public SimuLatchActuatorModule(
            SimuCompactIOModule simuDIO,
            double currentLimitInVolt,
            double speedVoltageInVolt,
            int speedVoltageMinValue,
            int speedVoltageMaxValue) {
        super(currentLimitInVolt, speedVoltageInVolt, speedVoltageMinValue, speedVoltageMaxValue);
        this.simuDIO = simuDIO;
    }

    /**
     * For the simulated Latch Actuator the command open has to update the value
     * of the lock and unlock sensors on the same side of the filter (Xminus or
     * Xplus).
     *
     * @return
     * @throws FailedCommandException
     */
    @Override
    public String open() {

        ComponentLookup lookup = getComponentLookup();
        int lockSensorInputNumero;
        int unlockSensorInputNumero;
        if (name.endsWith("Xminus")) {
            lockSensorInputNumero = ((DigitalSensor) lookup.getComponentByName("lockSensorLatchXminus")).getInputNumero();
            unlockSensorInputNumero = ((DigitalSensor) lookup.getComponentByName("unlockSensorLatchXminus")).getInputNumero();
        } else if (name.endsWith("Xplus")) {
            lockSensorInputNumero = ((DigitalSensor) lookup.getComponentByName("lockSensorLatchXplus")).getInputNumero();
            unlockSensorInputNumero = ((DigitalSensor) lookup.getComponentByName("unlockSensorLatchXplus")).getInputNumero();
        } else {
            throw new FailedCommandException(name + " Can't retrieve the latch for which I work.");
        }

        //For the simulated Actuator, the command open updates the value of the lock and unlock sensors.
        simuDIO.replaceDigitalValues(lockSensorInputNumero, 0, unlockSensorInputNumero, 1);
        FCSLOG.debug(name + " is OPEN");
        return name + " is OPEN";
    }

    /**
     * For the simulated latch actuator, the command close has just to update
     * the values of the lock and unlock simulated sensors.
     *
     * @return
     * @throws FailedCommandException
     */
    @Override
    public String close()  {

        ComponentLookup lookup = getComponentLookup();
        int lockSensorInputNumero;
        int unlockSensorInputNumero;
        if (name.endsWith("Xminus")) {
            lockSensorInputNumero = ((DigitalSensor) lookup.getComponentByName("lockSensorLatchXminus")).getInputNumero();
            unlockSensorInputNumero = ((DigitalSensor) lookup.getComponentByName("unlockSensorLatchXminus")).getInputNumero();
        } else if (name.endsWith("Xplus")) {
            lockSensorInputNumero = ((DigitalSensor) lookup.getComponentByName("lockSensorLatchXplus")).getInputNumero();
            unlockSensorInputNumero = ((DigitalSensor) lookup.getComponentByName("unlockSensorLatchXplus")).getInputNumero();
        } else {
            throw new FailedCommandException(name + " Can't retrieve the latch for which I work.");
        }

        //For the simulated Actuator, the command close updates the value of the lock and unlock sensors.
        simuDIO.replaceDigitalValues(lockSensorInputNumero, 1, unlockSensorInputNumero, 0);
        return name + " is CLOSED";
    }

    @Override
    public String powerOff() {
        return name + " is OFF";
    }

}
