
package org.lsst.ccs.subsystems.fcs.singlefiltertest;

import java.time.Duration;
import java.util.concurrent.locks.Condition;
import java.util.logging.Level;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystems.fcs.CarouselModule;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.AutoChangerTrucksLocation;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterLocation;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceStatus;
import org.lsst.ccs.subsystems.fcs.Filter;
import org.lsst.ccs.subsystems.fcs.MainModule;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByBasicAutoChanger;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.common.ModuleState;
import org.lsst.ccs.subsystems.fcs.errors.ClampsOrLatchesDisagreeException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * An abstract changer module without a fliprail, without online clamp, and
 * without a motor. This Module is in charge of the 2 latches which hold a
 * filter in the autochanger. This basic autochanger is used in single filter
 * test. It extends MobileItemModule for the actions which can be applied to the
 * 2 latches : OPENLATCHES and CLOSELATCHES.
 *
 * @author virieux
 */
public abstract class BasicAutoChangerModule extends MobileItemModule implements FilterHolder, TruckMotorListener {
    
    @LookupField(strategy=Strategy.TREE)
    private MainModule mainModule;
    
    @LookupField(strategy=Strategy.TREE)
    private AgentPeriodicTaskService  periodicTaskService;
    /**
     * This is the name of the DIO on which are plugged the filter sensors. Used
     * in the groovy file and in initModule.
     *
     */
    @ConfigurationParameter(description="The name of the DIO where the rails sensors are plugged.")
    private String railsSensorsDIOName;
    /**
     * This is the name of the DIO on which are plugged the rails sensors. Used
     * in the groovy file and in initModule.
     *
     */
    @ConfigurationParameter(description="The name of the DIO where the filter sensors are plugged.")
    private String filterSensorsDIOName;

    private boolean trucksEmpty;
    private Filter filterOnTrucks;
    
    @LookupField(strategy=Strategy.BYNAME)
    protected CarouselModule carousel;

    /**
     * Position of the trucks of the autochanger. It is monitored by the 4 rail
     * sensors. If the sensors detects a value different for the 2 sides, the
     * trucksLocation is ERROR.
     */
    private AutoChangerTrucksLocation trucksLocation = AutoChangerTrucksLocation.UNKNOWN;

    /**
     * A latch which hold a filter on the trucks at position X-
     */
    private final SftFilterLatchModule latchXminus;

    /**
     * A latch which hold a filter on the trucks at position X+
     */
    private final SftFilterLatchModule latchXplus;

    /**
     * The truck on side X-
     */
    protected TruckModule truckXminus;

    /**
     * The truck on side X+
     */
    protected TruckModule truckXplus;

    private CompactIOModule railsSensorsDIO;
    private CompactIOModule filterSensorsDIO;


    public final static String publishedByAutoChangerOutputName = "publishedByAutoChanger";
    protected volatile ModuleState state = ModuleState.HALTED;
    private volatile FilterPresenceStatus presenceStatus;
    protected volatile LockStatus latchesState = LockStatus.UNKNOWN;

    /**
     * ****************************************************************************************
     */
    /**
     * ******************** Tools for the synchronization *************************************
     */
    /**
     * ****************************************************************************************
     */
    protected volatile boolean updatingLatches = false;
    protected volatile boolean closingLatches = false;
    protected volatile boolean openingLatches = false;

    private volatile boolean updatingTrucksLocation = false;
    //When we update the 2 latches, we want to take a lock on the autochanger until
    //the end of the update, so any other thread has to wait for this update.
    private final Condition stateUpdated = lock.newCondition();

    //When we update the 2 trucks position reading the sensors, 
    //we want to take a lock on the autochanger until
    //the end of the update, so any other thread has to wait for this update.
    private final Condition trucksLocationUpdated = lock.newCondition();

    protected long timeoutForOpeningLatches;
    protected long timeoutForClosingLatches;



    /**
     * Build a BasicAutoChangerModule with 2 latches, 2 trucks and 2 DIO devices.
     * @param latchXminus
     * @param latchXplus
     * @param truckXminus
     * @param truckXplus
     * @param railsSensorsDIOName
     * @param filterSensorsDIOName 
     */
    public BasicAutoChangerModule(
            SftFilterLatchModule latchXminus,
            SftFilterLatchModule latchXplus,
            TruckModule truckXminus,
            TruckModule truckXplus,
            String railsSensorsDIOName,
            String filterSensorsDIOName) {
        this.latchXminus = latchXminus;
        this.latchXplus = latchXplus;
        this.truckXminus = truckXminus;
        this.truckXplus = truckXplus;
        this.railsSensorsDIOName = railsSensorsDIOName;
        this.filterSensorsDIOName = filterSensorsDIOName;
    }

    /**
     * ****************************************************************************************
     */
    /**
     * ******************** End of Tools for the synchronization*******************************
     */
    /**
     * ****************************************************************************************
     */
    /**
     * ***********************************************************************************************
     */
    /**
     * ******************** SETTERS AND GETTERS  *****************************************************
     */
    /**
     * ***********************************************************************************************
     * @return 
     */
    public CompactIOModule getRailsSensorsDIO() {
        return railsSensorsDIO;
    }

    public CompactIOModule getFilterSensorsDIO() {
        return filterSensorsDIO;
    }


    public Filter getFilterOnTrucks() {
        return filterOnTrucks;
    }

    /**
     * @return the state
     */
    public ModuleState getState() {
        return state;
    }

    /**
     * @param state the state to set
     */
    public void setState(ModuleState state) {
        this.state = state;
    }

    public SftFilterLatchModule getLatchXminus() {
        return latchXminus;
    }

    public SftFilterLatchModule getLatchXplus() {
        return latchXplus;
    }

    public TruckModule getTruckXminus() {
        return truckXminus;
    }

    public TruckModule getTruckXplus() {
        return truckXplus;
    }

    public boolean isTrucksEmpty() {
        return trucksEmpty;
    }

    public void setTrucksEmpty(boolean isEmpty) {
        this.trucksEmpty = isEmpty;
    }

    public void setFilterOnTrucks(Filter filterOnTrucks) {
        this.filterOnTrucks = filterOnTrucks;
    }

    public String getFilterOnTrucksName() {
        if (this.filterOnTrucks == null) {
            return "none";
        } else {
            return this.filterOnTrucks.getName();
        }
    }

    public String getFilterSensorsDIOName() {
        return filterSensorsDIOName;
    }

    // TO-DO : do we want this ? changing the dio name will not change the dio
    public void setFilterSensorsDIOName(String filterSensorsDIOName) {
        this.filterSensorsDIOName = filterSensorsDIOName;
    }

    public String getRailsSensorsDIOName() {
        return railsSensorsDIOName;
    }

    public void setRailsSensorsDIOName(String railsSensorsDIOName) {
        this.railsSensorsDIOName = railsSensorsDIOName;
    }





    public abstract boolean isMovingToStandback();

    public abstract boolean isMovingToStandby();

    /**
     * **************************************************************************************************
     */
    /**
     * ******************** END OF SETTERS AND GETTERS  *************************************************
     */
    /**
     * **************************************************************************************************
     */
    /**
     * **************************************************************************************************
     */
    /**
     * ******************** Overriden methods from Modules or generic methodes **************************
     */
    /**
     * **************************************************************************************************
     */
    @Override
    public void init() {
        FCSLOG.info("[AutoChangerModule] Initializing the Auto Changer module ");
        ComponentLookup lookup = getComponentLookup();
        this.filterSensorsDIO = (CompactIOModule) lookup.getComponentByName(this.filterSensorsDIOName);
        this.railsSensorsDIO = (CompactIOModule) lookup.getComponentByName(this.railsSensorsDIOName);
        
        this.timeoutForClosingLatches = java.lang.Math.max(this.latchXminus.getTimeoutForClosing(), 
                this.latchXplus.getTimeoutForClosing());
        
        this.timeoutForOpeningLatches = java.lang.Math.max(this.latchXminus.getTimeoutForOpening(), 
                this.latchXplus.getTimeoutForOpening());
        
        SftTruckMotor sftTrucksMotor = (SftTruckMotor) lookup.getComponentByName("sftTrucksMotor");
        
        /*This is used for the single-filter-test simulation. It's a quick fix to get rid of the closure in groovy 
        * description file. It should be redesigned in another way (see simulation of the loader carrier).
        */
        if (sftTrucksMotor == null) {
            FCSLOG.error(name + "==>>> sftTrucksMotor == null - Please fix groovy description file.");
            throw new IllegalArgumentException(name + "==>>> null sftTrucksMotor - fix groovy description file.");
        } else {
            sftTrucksMotor.setTruckMotorListener(this);
        }
        
        
        //ATTENTION : doing updateLatchesStateWithSensors() in the tick method is a bad idea because:
// 1 - updateLatchesStateWithSensors takes a lock and so it could be obliged to wait 
// for the lock to be released and we don't want a tick method to wait.
// 2 - during opening latches, the latches Xminus and the latches Xplus disagree during
// a short delay and we don't want to raise an alarm in this case.
// If we want to read sensors and update state we have to write another method which
// does something only if the hardware is not moving and begins with a lock.tryLock().
// But is it useful to updateStateWithSensors in the tick method if the hardware is not moving ?
        periodicTaskService.scheduleAgentPeriodicTask(new AgentPeriodicTask("ac-tick",this::publishData).withIsFixedRate(true).withLogLevel(Level.WARNING).withPeriod(Duration.ofSeconds(1)));
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if hardware is connected and ready.")
    @Override
    public boolean myDevicesReady() {
        return mainModule.allDevicesBooted();
    }

    /*
     * what to do when the Modules we observe change their values.
     */
    @Override
    public void onTruckEvent() {
        FCSLOG.debug(name + "===> PROCESS_UPDATE from trucksMotor");
        this.publishDataAndNotifyObservers();
    }

    public StatusDataPublishedByBasicAutoChanger getStatusData() {
        return SftUtils.createStatusDataPublishedByBasicAutoChanger(this);
    }

    public void publishDataAndNotifyObservers() {
        StatusDataPublishedByBasicAutoChanger status = getStatusData();
        this.publishData();
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     *
     */
    @Override
    public void publishData() {
        StatusDataPublishedByBasicAutoChanger status = this.getStatusData();
        this.getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData("autochanger", status));
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(name);
        sb.append("\n");
        sb.append("Trucks position = ");
        sb.append("Filter on trucks:");
        if (this.filterOnTrucks == null) {
            sb.append(" NONE").append("\n");
        } else {
            sb.append(this.filterOnTrucks.getName()).append("\n");
        }
        return sb.toString();
    }

    /**
     * **************************************************************************************************
     */
    /**
     * ******************** END of Overriden methods from Modules  **************************************
     */
    /**
     * **************************************************************************************************
     */
    /**
     * **************************************************************************************************
     */
    /**
     * ******************** Methods for the motion of the trucks ****************************************
     */
    /**
     * **************************************************************************************************
     */
    /**
     *
     * @return true if the trucks are moving
     */
    @Override
    public abstract boolean isMoving();

    public abstract String goToStandby() ;

    public abstract String goToStandback() ;

    public abstract String moveFilterToStandby(Filter aFilter) ;

    public abstract String moveFilterToStandback(Filter aFilter) ;

    @Override
    public abstract void updateStateWithSensors() ;

    /**
     * This method checks if the preconditions are ok before a motion of the
     * trucks can be started. If the preconditions are not ok, it throws a
     * RejectedCommandException. A motion of the trucks can be started : if the
     * latches are not in error and if the truks are empty or (the trucks are
     * loaded and the latches are Locked).
     *
     * @throws RejectedCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, 
            description = "checks if the preconditions are ok before a motion of the trucks can be started")
    public void checkPreConditionsForTrucksMotion()  {
        
        FCSLOG.info("Checking pre-conditions for trucks motion");
        updateLatchesStateWithSensors();
        FCSLOG.debug(name + " LATCHES STATUS= " + this.latchesState.toString());
        if (latchesState == LockStatus.ERROR) {
            throw new RejectedCommandException(name
                    + "Trucks motion forbidden when latches status is ERROR.");
        }
        if (latchesState == LockStatus.UNKNOWN) {
            throw new RejectedCommandException(name
                    + "Trucks motion forbidden when latches status is UNKNOWN.");
        }
        if (!trucksEmpty && latchesState == LockStatus.UNLOCKED) {
            throw new RejectedCommandException(name
                    + "Trucks motion forbidden when trucks are loaded and latches are UNLOCKED.");
        }

        if (this.isHoldingFilterAtStandby() && carousel.isHoldingFilterAtStandby()) {
            throw new RejectedCommandException(name
                    + "Trucks motion forbidden when both carousel and autochanger are holding filter.");
        }

        //TODO Trucks motion forbidden when both carousel and autochanger are holding filter.
        //TODO Trucks motion forbidden when both loader and autochanger are holding filter.
        FCSLOG.info("===== Preconditions for trucks motion are checked.");
    }

    /**
     * Go to STANDBY to grab filter and stay at STANDBY.
     * @param filter
     * @return
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, 
            description = "Go to STANDBY to grab filter and stay at STANDBY")
    public String grabFilterAtStandby(Filter filter)  {
        if (this.filterOnTrucks != null) {
            throw new RejectedCommandException(name + ": can't grab a filter " + "when a filter is already loaded in trucks ");
        }
        FCSLOG.info(name + ": grabbing " + filter.getName() + " at standby position.");
        goToStandby();
        closeLatchesAtStandby();
        //now the  filter is on the autochanger
        synchronized (this) {
            this.filterOnTrucks = filter;
            setTrucksEmpty(false);
            this.filterOnTrucks.setFilterLocation(FilterLocation.AUTOCHANGER);
        }
        publishData();
        String ack = getName() + ": " + filter.getName() + " is grabbed on autochanger";
        FCSLOG.info(ack);
        return ack;
    }

    /**
     * Open the latches at standby position and go back empty to STANBACK
     * position.
     *
     * @return
     * @throws RejectedCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, 
            description = "Ungrab filter at STANDBY and go back to HANDOFF")
    public String ungrabFilterAtStandby()  {
        //TODO to check : carousel clamp must be locked
        //
        FCSLOG.info(name + ": ungrabbing filter at standby position.");
        openLatchesAtStandby();
        //eject filter from autochanger
        synchronized (this) {
            this.filterOnTrucks = null;
            setTrucksEmpty(true);
            this.publishDataAndNotifyObservers();
        }
        //when a filter is at standby position the trucks goes empty to SWAPOUT position
        FCSLOG.info(name + " trucks going empty to STANDBACK position");
        //goToPosition(this.getTrucksPositionSwapout());
        goToStandback();
        publishData();
        return getName() + ": trucks are empty at STANDBACK position ";
    }

    /**
     * **************************************************************************************************
     */
    /**
     * ******************** END of Methods for the motion of the trucks *********************************
     */
    /**
     * **************************************************************************************************
     */
    /**
     * **************************************************************************************************
     */
    /**
     * ******************** Methods related to the latches **********************************************
     */
    /**
     * **************************************************************************************************
     */
    /**
     * This method returns the latchesState if it is not been updated otherwise
     * it waits until the end of the update.
     *
     * @return the trucksLocation
     */
    public LockStatus getLatchesState() {
        lock.lock();
        try {
            while (this.updatingLatches) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(ex);
                }
            }
            return this.latchesState;
        } finally {
            lock.unlock();
        }
    }
    
    @Override
    public boolean isHoldingFilter() {
        return isHoldingFilterAtStandby();
    }

    /**
     *
     * @return true if the trucks are at standby position and hold a filter.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, 
            description = "Return true if the trucks are at STANDBY position and hold a filter")
    public boolean isHoldingFilterAtStandby() {
        if (isMoving()) {
            return false;
        }
        if (!isAtStandby()) {
            return false;
        }
        if (isTrucksEmpty()) {
            return false;
        }
        if (this.getLatchesState().equals(LockStatus.LOCKED)) {
            return true;
        }
        return false;
    }

    /**
     *
     * @return true if the trucks are at standback position and hold the filter.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, 
            description = "Return true if the trucks are at STANDBACK position and hold a filter")
    public boolean isHoldingFilterAtStandback() {
        if (isMoving()) {
            return false;
        }
        if (!isAtStandback()) {
            return false;
        }
        if (isTrucksEmpty()) {
            return false;
        }
        if (this.getLatchesState().equals(LockStatus.LOCKED)) {
            return true;
        }
        return false;

    }

    @Override
    public boolean isActionCompleted(MobileItemAction action) {
        if (action.equals(MobileItemAction.OPENLATCHES)) {
            return this.getLatchesState().equals(LockStatus.UNLOCKED);
        } else if (action.equals(MobileItemAction.CLOSELATCHES)) {
            return this.getLatchesState().equals(LockStatus.LOCKED);
        } else {
            throw new IllegalArgumentException("Action on latches must be OPENLATCHES or CLOSELATCHES");
        }
    }

    @Override
    public void postAction(MobileItemAction action)  {
        updateLatchesStateWithSensors();
        this.publishData();
    }

    @Override
    public void startAction(MobileItemAction action)  {
        if (action.equals(MobileItemAction.OPENLATCHES)) {
            this.latchXminus.latchActuator.open();
            this.latchXplus.latchActuator.open();
        } else if (action.equals(MobileItemAction.CLOSELATCHES)) {
            this.latchXminus.latchActuator.close();
            this.latchXplus.latchActuator.close();
        } else {
            throw new IllegalArgumentException(name + "Action on latches must be OPENLATCHES or CLOSELATCHES");
        }

    }

    @Override
    public void abortAction(MobileItemAction action, long delay)  {
        //TODO complete this
        FCSLOG.info(name + " stopAction : nothing to be done.");
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted()  {
        try {
            this.updateLatchesStateWithSensors();
        } catch (ClampsOrLatchesDisagreeException ex) {
            FCSLOG.warning(ex);
            FCSLOG.warning(name + ": a little delay between the update of the 2 latches : have to wait ...");
        }

    }

    /**
     * Open the 2 latches when the autochanger trucks are at standby position.
     *
     * @return a message for the end user
     * @throws RejectedCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, 
            description = "Open the 2 latches when the autochanger trucks are at standby position", 
            alias = "openLatches")
    public String openLatchesAtStandby() throws 
            FcsHardwareException {

        this.updateStateWithSensors();

        if (!this.isAtStandby()) {
            throw new RejectedCommandException(name + "CANNOT OPEN LATCHES at STANDBY : AUTOCHANGER is NOT AT STANDBY");
        }

        if (!carousel.isHoldingFilterAtStandby()) {
            throw new RejectedCommandException(name + "CANNOT OPEN LATCHES at STANDBY : CAROUSEL is NOT HOLDING THE FILTER");
        }

        this.executeAction(MobileItemAction.OPENLATCHES, this.timeoutForOpeningLatches);

        this.latchXminus.latchActuator.powerOff();
        this.latchXplus.latchActuator.powerOff();

        publishDataAndNotifyObservers();
        return "Autochanger latches are unlocked";
    }

    /**
     * Close the 2 latches when the autochanger trucks are at standby position.
     *
     * @return a message for the end user
     * @throws RejectedCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, 
            description = "Close the 2 latches when the autochanger trucks are at standby position", 
            alias = "closeLatches")
    public String closeLatchesAtStandby()  {

        this.updateStateWithSensors();

        if (!this.isAtStandby()) {
            throw new RejectedCommandException(name + "CANNOT CLOSE LATCHES at STANDBY : AUTOCHANGER is NOT AT STANDBY");
        }

        this.executeAction(MobileItemAction.CLOSELATCHES, this.timeoutForClosingLatches);

        this.latchXminus.latchActuator.powerOff();
        this.latchXplus.latchActuator.powerOff();

        publishDataAndNotifyObservers();
        return "Autochanger latches are locked";
    }

    /**
     * This methods updates the latches status in reading the sensors.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void updateLatchesStateWithSensors()  {
        lock.lock();
        this.updatingLatches = true;

        try {

            this.filterSensorsDIO.updateValue();

            int filterSensorsHexaValue = this.filterSensorsDIO.getHexaValue();
            this.latchXminus.updateState(filterSensorsHexaValue);
            this.latchXplus.updateState(filterSensorsHexaValue);

            if (latchXminus.getLockStatus().equals(latchXplus.getLockStatus())) {
                this.latchesState = latchXminus.getLockStatus();
            } else {
                throw new ClampsOrLatchesDisagreeException(name + "Error in latches lock status : the latches don't agree");
            }

            if (latchXminus.getPresenceStatus().equals(latchXplus.getPresenceStatus())) {
                this.presenceStatus = latchXminus.getPresenceStatus();
            } else {
                throw new ClampsOrLatchesDisagreeException(name + "Error in latches filter presence status : the latches don't agree.");
            }

            if (this.latchesState.equals(FcsEnumerations.LockStatus.LOCKED) 
                    && this.presenceStatus.equals(FcsEnumerations.FilterPresenceStatus.ENGAGED)) {
                this.trucksEmpty = false;
            } else {
                this.trucksEmpty = true;
            }
            FCSLOG.debug(name + ": latches are updated");

        } finally {
            this.updatingLatches = false;
            this.stateUpdated.signal();
            lock.unlock();
            publishDataAndNotifyObservers();
        }

    }

    /**
     * **************************************************************************************************
     */
    /**
     * ******************** END of Methods related to the latches ***************************************
     */
    /**
     * **************************************************************************************************
     */
    /**
     * **************************************************************************************************
     */
    /**
     * ******************** Methods to retrieve the trucks location *************************************
     */
    /**
     * **************************************************************************************************
     */
    /**
     * This method returns the truckLocation if it is not been updated otherwise
     * it waits until the end of the update.
     *
     * @return the trucksLocation
     */
    public AutoChangerTrucksLocation getTrucksLocation() {
        lock.lock();
        try {
            while (updatingTrucksLocation) {
                try {
                    trucksLocationUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(ex);
                }
            }
            return trucksLocation;
        } finally {
            lock.unlock();
        }
    }

    /**
     *
     * @return true if the trucks are at standby position, otherwise false
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, 
            description = "Return true if the trucks are at STANDBY position")
    @Override
    public boolean isAtStandby() {
        return getTrucksLocation() == AutoChangerTrucksLocation.STANDBY;
    }

    /**
     * For the single-filter-test only ?
     *
     * @return true if the trucks are at standback position, otherwise false
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, 
            description = "Return true if the trucks are at STANDBACK position")
    public boolean isAtStandback() {
        return getTrucksLocation() == AutoChangerTrucksLocation.STANDBACK;
    }

    /**
     *
     * @return true if the trucks are at standback position, otherwise false
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, 
            description = "Return true if the trucks are at HANDOFF position")
    @Override
    public boolean isAtHandoff() {
        return getTrucksLocation() == AutoChangerTrucksLocation.HANDOFF;
    }

    /**
     *
     * @return true if the trucks are at standback position, otherwise false
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, 
            description = "Return true if the trucks are at ONLINE position")
    public boolean isOnline() {
        return getTrucksLocation() == AutoChangerTrucksLocation.ONLINE;
    }

    /**
     * This methods updates the trucks location in reading the sensors.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void updateTrucksLocationWithSensors()  {
        lock.lock();
        updatingTrucksLocation = false;
        try {
            this.railsSensorsDIO.updateValue();

            int dioHexaValue = this.railsSensorsDIO.getHexaValue();
            FCSLOG.debug(this.railsSensorsDIO.getName() + " HEXA VALUE=" + dioHexaValue);
            this.truckXminus.updateLocation(dioHexaValue);
            this.truckXplus.updateLocation(dioHexaValue);

            if (truckXminus.getTruckLocation() == truckXplus.getTruckLocation()) {
                this.trucksLocation = truckXminus.getTruckLocation();

            } else {
                String msg = getName() + "Trucks are not at the same location";
                FCSLOG.error(msg);
                throw new FcsHardwareException(msg);
            }

        } finally {
            this.updatingTrucksLocation = false;
            this.trucksLocationUpdated.signal();
            lock.unlock();
            publishDataAndNotifyObservers();
        }

    }





    /**
     * **************************************************************************************************
     */
    /**
     * ******************** END of Methods to retrieve the trucks location ******************************
     */
    /**
     * **************************************************************************************************
     */
}
