
package org.lsst.ccs.subsystems.fcs.simulation;

import java.math.BigDecimal;
import java.security.SecureRandom;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.subsystems.fcs.CarouselClampModule;
import org.lsst.ccs.subsystems.fcs.CarouselModule;
import org.lsst.ccs.subsystems.fcs.CarouselSocket;
import org.lsst.ccs.subsystems.fcs.FCSCst;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenADC;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenSensor14bits;

/**
 * A general simulated CanOpenSensor14bits.
 * The simulated sensor doesn't read its new value on a CANopen device but the new value is
 * generated by a random generator between a min and a max value.
 * These values can be changed with method setNewRange.
 * @author virieux
 */
public class SimuCanOpenSensor14bits extends CanOpenSensor14bits {
    
    @LookupField(strategy = Strategy.TOP)
    protected Subsystem s;
    
    @LookupField(strategy=Strategy.BYNAME)
    protected SimuCanOpenProxy tcpProxy;
    
    /*the clamp this simulated sensor belongs to*/
    @LookupField(strategy=Strategy.ANCESTORS)
    protected CarouselClampModule clamp;
    
    @LookupField(strategy=Strategy.BYNAME)
    protected FilterHolder autochanger;
    
    /*used in isAtStandby method*/
    @LookupField(strategy=Strategy.BYNAME)
    private  CarouselModule carousel;

            
    private int value1 = FCSCst.SENSOR14BITS_MIN;
    private int value2 = FCSCst.SENSOR14BITS_MAX;
    private final SecureRandom randomGenerator;
    /*
    This class listens to acPLutoGateway to be able to know when trucks are away from carousel.
    When trucks are away from carousel and loaded, carousel at standby is empty.
    When trucks are at STANDBY and a filter is in trucks, carousel is LOCKED.
     */
    @LookupField(strategy=Strategy.BYNAME)
    private SimuAutochangerPlutoGateway acSensorsGateway;

    /**
     * Creates a new SimuCanOpenSensor14bits from arguments :
     * @param adc
     * @param numOfAnalogInput 
     */
    public SimuCanOpenSensor14bits( 
            CanOpenADC adc, int numOfAnalogInput) {
        super(adc, numOfAnalogInput);
        this.randomGenerator = new SecureRandom();
    }
    
    /**
     * Initialize the clamp that this sensor monitors, the simulated tcpProxy with a
 PDO storage.
     * 
     */
    @Override
    public void init() {
        this.tcpProxy.getPdoStorage().setVarValue(this.adc.getNodeID(),
                   this.numOfAnalogInput, 0);
        /* for the whole FCS*/
        if (acSensorsGateway == null) {
        } else {
            FCSLOG.warning("It's normal to have message : null reference associated to acPlutoGateway because in carousel-standalone, "
                    + "a simulated carousel clamp sensor doesn't have to listen to autochanger plutoGateway.");
        }   
        FCSLOG.info(name + " belongs to clamp "+clamp.getName());
        
    }
    
    /**
     * Set a new range of values.
     * The method readNewValue will return a value in this range.
     * @param aValue1
     * @param aValue2 
     */
    void setNewRange(int aValue1, int aValue2) {
        if (aValue2 > FCSCst.SENSOR14BITS_MAX) throw new IllegalArgumentException(name + 
                "aValue2="+aValue2 +" Can't be >"+FCSCst.SENSOR14BITS_MAX);
        if (aValue1 < FCSCst.SENSOR14BITS_MIN) throw new IllegalArgumentException(name + 
        " aValue1="+aValue1 +" Can't be <"+FCSCst.SENSOR14BITS_MIN);
        
        if (aValue2 < aValue1) {
            throw new IllegalArgumentException(name +
                    " aValue1="+aValue1 + " aValue2="+aValue2 + "we must have: aValue1<aValue2");
        }
        this.value1 = aValue1;
        this.value2 = aValue2;
    }

    /**
     * The simulated sensor doesn't read its new value on a CANopen device but the new value is
     * generated by a random generator between a min and a max value. This values has been previously 
     * changed by a setNewRange command.
     * @return a random between value1 and value2
     */
    @Override
    public int readNewValue() {
        if (value2 > 0) {
            int n = randomGenerator.nextInt(value2);

            if (n > value1) {
                return n;
            } else {
                BigDecimal bd = new BigDecimal(value1 + n * (value2 - value1) / value2);
                return bd.intValue();
            }
        } else {
            String msg = getName() + ": ERROR in readNewValue: value2 should be positive";
            throw new IllegalArgumentException(msg);
        }

    }
    
    /**
     * In the simulator, instead of reading a new PDO to read a new value for the sensor, 
     * we set a value for the sensor and then we update the PDOStorage for this sensor.
     */
    protected void updateFakePDOStorage() {
        this.tcpProxy.getPdoStorage().setVarValue(this.adc.getNodeID(), 
                this.numOfAnalogInput, this.readNewValue());
    }

    /**
     * return true if this sensor is on the clamp which is at STANDBY.
     * return false if no socket is at STANDBY or if this sensor is on a clamp which is not at STANDBY.
     * Used by simulator of clamps sensors to know where a filter presence sensor or a clamp sensor has to be modified 
     * when the autochnager trucks arrive at STANDBY.
     * @return 
     */
    protected boolean isAtStandby() {
        CarouselSocket socket = carousel.getSocketAtStandby();
        if (socket == null) {
            return false;
        } else {
            String socketName = socket.getName();
            char numSocket = socketName.charAt(socketName.length() - 1);
            char myNum = getName().charAt(name.length() - 1);
            return numSocket == myNum;
        }
    }
}
